<?php

use App\Models\Role;
use App\Models\Society;
use App\Models\SubscriptionPlan;
use App\Models\SuperAdmin;
use App\Models\User;

beforeEach(function () {
    // Ensure roles exist
    Role::firstOrCreate(['name' => 'super_admin'], ['display_name' => 'Super Admin']);
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('super admin can view societies list', function () {
    $superAdmin = SuperAdmin::factory()->create();

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->get(route('super-admin.societies.index'));

    $response->assertOk();
});

test('super admin can create a new society', function () {
    $superAdmin = SuperAdmin::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $plan = SubscriptionPlan::factory()->create(['is_active' => true]);

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->post(route('super-admin.societies.store'), [
            'name' => 'Test Society',
            'registration_no' => 'REG123',
            'address' => '123 Test St',
            'city' => 'Test City',
            'state' => 'Test State',
            'pincode' => '123456',
            'phone' => '1234567890',
            'email' => 'society@test.com',
            'admin' => [
                'name' => 'Admin User',
                'email' => 'admin@test.com',
                'phone' => '9876543210',
                'password' => 'password123',
            ],
            'subscription' => [
                'plan_id' => $plan->id,
                'billing_cycle' => 'monthly',
                'status' => 'trial',
            ],
        ]);

    $response->assertRedirect(route('super-admin.societies.index'));
    $this->assertDatabaseHas('societies', ['name' => 'Test Society']);
    $this->assertDatabaseHas('users', ['email' => 'admin@test.com', 'role_id' => $role->id]);
});

test('super admin can view a society', function () {
    $superAdmin = SuperAdmin::factory()->create();
    $society = Society::factory()->create();

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->get(route('super-admin.societies.show', $society));

    $response->assertOk();
});

test('super admin can update a society', function () {
    $superAdmin = SuperAdmin::factory()->create();
    $society = Society::factory()->create();

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->put(route('super-admin.societies.update', $society), [
            'name' => 'Updated Society',
            'registration_no' => $society->registration_no,
            'address' => $society->address,
            'city' => $society->city,
            'state' => $society->state,
            'pincode' => $society->pincode,
            'phone' => $society->phone,
            'email' => $society->email,
        ]);

    $response->assertRedirect(route('super-admin.societies.index'));
    $this->assertDatabaseHas('societies', ['id' => $society->id, 'name' => 'Updated Society']);
});

test('super admin can delete a society', function () {
    $superAdmin = SuperAdmin::factory()->create();
    $society = Society::factory()->create();

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->delete(route('super-admin.societies.destroy', $society));

    $response->assertRedirect(route('super-admin.societies.index'));
    $this->assertSoftDeleted('societies', ['id' => $society->id]);
});

test('guests cannot access societies', function () {
    $response = $this->get(route('super-admin.societies.index'));

    $response->assertRedirect(route('super-admin.login'));
});

test('super admin can filter societies by status', function () {
    $superAdmin = SuperAdmin::factory()->create();
    Society::factory()->create(['status' => 'active']);
    Society::factory()->create(['status' => 'pending']);

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->get(route('super-admin.societies.index', ['status' => 'active']));

    $response->assertOk();
});

test('super admin can search societies', function () {
    $superAdmin = SuperAdmin::factory()->create();
    Society::factory()->create(['name' => 'Test Society']);

    $response = $this->actingAs($superAdmin, 'super_admin')
        ->get(route('super-admin.societies.index', ['search' => 'Test']));

    $response->assertOk();
});
