<?php

use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use App\Models\Visitor;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('society admin can view visitors list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Visitor::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.visitors.index', $society));

    $response->assertOk();
});

test('society admin can check in a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $visitor = Visitor::factory()->for($society)->create(['status' => 'approved']);

    $response = $this->actingAs($admin)
        ->post(route('society.visitors.check-in', [$society, $visitor]));

    $response->assertRedirect();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'checked_in']);
    $this->assertNotNull($visitor->fresh()->check_in_at);
});

test('society admin can check out a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $visitor = Visitor::factory()->for($society)->create([
        'status' => 'checked_in',
        'check_in_at' => now(),
    ]);

    $response = $this->actingAs($admin)
        ->post(route('society.visitors.check-out', [$society, $visitor]));

    $response->assertRedirect();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'checked_out']);
    $this->assertNotNull($visitor->fresh()->check_out_at);
});

test('society admin can approve a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $visitor = Visitor::factory()->for($society)->create(['status' => 'pending']);

    $response = $this->actingAs($admin)
        ->post(route('society.visitors.approve', [$society, $visitor]));

    $response->assertRedirect();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'approved']);
});

test('society admin can reject a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $visitor = Visitor::factory()->for($society)->create(['status' => 'pending']);

    $response = $this->actingAs($admin)
        ->post(route('society.visitors.reject', [$society, $visitor]), [
            'rejection_reason' => 'Not authorized',
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'rejected']);
});
