<?php

use App\Models\Building;
use App\Models\Role;
use App\Models\Society;
use App\Models\Unit;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('society admin can view units list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();
    Unit::factory()->forBuilding($building)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.units.index', $society));

    $response->assertOk();
});

test('society admin can create a unit', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->post(route('society.units.store', $society), [
            'building_id' => $building->id,
            'unit_no' => '101',
            'type' => 'apartment',
            'area' => 1200,
            'status' => 'occupied',
        ]);

    $response->assertRedirect(route('society.units.index', $society));
    $this->assertDatabaseHas('units', [
        'society_id' => $society->id,
        'building_id' => $building->id,
        'unit_no' => '101',
    ]);
});

test('society admin can update a unit', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();
    $unit = Unit::factory()->forBuilding($building)->create();

    $response = $this->actingAs($admin)
        ->put(route('society.units.update', [$society, $unit]), [
            'building_id' => $building->id,
            'unit_no' => '102',
            'type' => $unit->type,
            'area' => $unit->area,
            'status' => $unit->status,
        ]);

    $response->assertRedirect(route('society.units.index', $society));
    $this->assertDatabaseHas('units', ['id' => $unit->id, 'unit_no' => '102']);
});

test('society admin can delete a unit', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();
    $unit = Unit::factory()->forBuilding($building)->create();

    $response = $this->actingAs($admin)
        ->delete(route('society.units.destroy', [$society, $unit]));

    $response->assertRedirect(route('society.units.index', $society));
    $this->assertSoftDeleted('units', ['id' => $unit->id]);
});
