<?php

use App\Models\Building;
use App\Models\Role;
use App\Models\Society;
use App\Models\Unit;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('society admin can view residents list', function () {
    $society = Society::factory()->create();
    $adminRole = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $adminRole->id]);
    User::factory()->count(3)->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    $response = $this->actingAs($admin)
        ->get(route('society.residents.index', $society));

    $response->assertOk();
});

test('society admin can create a resident', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();
    $unit = Unit::factory()->forBuilding($building)->create();
    $role = Role::where('name', 'resident')->first();

    $response = $this->actingAs($admin)
        ->post(route('society.residents.store', $society), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'phone' => '1234567890',
            'unit_id' => $unit->id,
            'type' => 'owner',
            'password' => 'password123',
            'status' => 'active',
        ]);

    $response->assertRedirect(route('society.residents.index', $society));
    $this->assertDatabaseHas('users', [
        'society_id' => $society->id,
        'email' => 'john@example.com',
        'role_id' => $role->id,
    ]);
});

test('society admin can update a resident', function () {
    $society = Society::factory()->create();
    $adminRole = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $adminRole->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    $response = $this->actingAs($admin)
        ->put(route('society.residents.update', [$society, $resident]), [
            'name' => 'Updated Name',
            'email' => $resident->email,
            'phone' => $resident->phone,
            'unit_id' => $resident->unit_id,
            'status' => $resident->status,
        ]);

    $response->assertRedirect(route('society.residents.index', $society));
    $this->assertDatabaseHas('users', ['id' => $resident->id, 'name' => 'Updated Name']);
});

test('society admin can delete a resident', function () {
    $society = Society::factory()->create();
    $adminRole = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $adminRole->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    $response = $this->actingAs($admin)
        ->delete(route('society.residents.destroy', [$society, $resident]));

    $response->assertRedirect(route('society.residents.index', $society));
    $this->assertSoftDeleted('users', ['id' => $resident->id]);
});
