<?php

use App\Models\Bill;
use App\Models\Payment;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('society admin can view payments list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Payment::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.payments.index', $society));

    $response->assertOk();
});

test('society admin can create a payment', function () {
    $society = Society::factory()->create();
    $adminRole = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $adminRole->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $bill = Bill::factory()->for($society)->for($resident)->create(['amount' => 5000.00]);

    $response = $this->actingAs($admin)
        ->post(route('society.payments.store', $society), [
            'user_id' => $resident->id,
            'bill_id' => $bill->id,
            'amount' => 5000.00,
            'payment_method' => 'cash',
            'payment_date' => now()->format('Y-m-d'),
        ]);

    $response->assertRedirect(route('society.payments.index', $society));
    $this->assertDatabaseHas('payments', [
        'society_id' => $society->id,
        'user_id' => $resident->id,
        'amount' => 5000.00,
    ]);
});

test('society admin can reconcile a payment', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $payment = Payment::factory()->for($society)->create(['status' => 'pending']);

    $response = $this->actingAs($admin)
        ->post(route('society.payments.reconcile', [$society, $payment]), [
            'status' => 'completed',
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('payments', ['id' => $payment->id, 'status' => 'completed']);
});

test('society admin can generate QR code for payment', function () {
    $society = Society::factory()->create();
    $adminRole = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $adminRole->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    $response = $this->actingAs($admin)
        ->post(route('society.payments.generate-qr', $society), [
            'user_id' => $resident->id,
            'amount' => 5000.00,
        ]);

    $response->assertOk();
    $response->assertJsonStructure(['qr_code', 'payment_link_id']);
});
