<?php

use App\Models\MaintenanceTicket;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use App\Models\Vendor;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('society admin can view maintenance tickets list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    MaintenanceTicket::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.maintenance.index', $society));

    $response->assertOk();
});

test('society admin can view a maintenance ticket', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $ticket = MaintenanceTicket::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.maintenance.show', [$society, $ticket]));

    $response->assertOk();
});

test('society admin can update ticket status', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $ticket = MaintenanceTicket::factory()->for($society)->create(['status' => 'open']);

    $response = $this->actingAs($admin)
        ->post(route('society.maintenance.update-status', [$society, $ticket]), [
            'status' => 'in_progress',
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('maintenance_tickets', ['id' => $ticket->id, 'status' => 'in_progress']);
});

test('society admin can assign vendor to ticket', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $ticket = MaintenanceTicket::factory()->for($society)->create();
    $vendor = Vendor::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->post(route('society.maintenance.assign-vendor', [$society, $ticket]), [
            'vendor_id' => $vendor->id,
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('maintenance_tickets', ['id' => $ticket->id, 'vendor_id' => $vendor->id]);
});

test('society admin can resolve a ticket', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $ticket = MaintenanceTicket::factory()->for($society)->create(['status' => 'in_progress']);

    $response = $this->actingAs($admin)
        ->post(route('society.maintenance.resolve', [$society, $ticket]), [
            'resolution_notes' => 'Issue resolved successfully',
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('maintenance_tickets', ['id' => $ticket->id, 'status' => 'resolved']);
});
