<?php

use App\Models\Expense;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('society admin can view expenses list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Expense::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.expenses.index', $society));

    $response->assertOk();
});

test('society admin can create an expense', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->post(route('society.expenses.store', $society), [
            'category' => 'maintenance',
            'description' => 'Plumbing repair',
            'amount' => 5000.00,
            'expense_date' => now()->format('Y-m-d'),
        ]);

    $response->assertRedirect(route('society.expenses.index', $society));
    $this->assertDatabaseHas('expenses', [
        'society_id' => $society->id,
        'amount' => 5000.00,
    ]);
});

test('society admin can update an expense', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $expense = Expense::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->put(route('society.expenses.update', [$society, $expense]), [
            'category' => $expense->category,
            'description' => 'Updated description',
            'amount' => $expense->amount,
            'expense_date' => $expense->expense_date->format('Y-m-d'),
        ]);

    $response->assertRedirect(route('society.expenses.index', $society));
    $this->assertDatabaseHas('expenses', ['id' => $expense->id, 'description' => 'Updated description']);
});

test('society admin can delete an expense', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $expense = Expense::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->delete(route('society.expenses.destroy', [$society, $expense]));

    $response->assertRedirect(route('society.expenses.index', $society));
    $this->assertSoftDeleted('expenses', ['id' => $expense->id]);
});
