<?php

use App\Models\Building;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('society admin can view buildings list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Building::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.buildings.index', $society));

    $response->assertOk();
});

test('society admin can create a building', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->post(route('society.buildings.store', $society), [
            'name' => 'Building A',
            'floors' => 10,
            'total_units' => 40,
        ]);

    $response->assertRedirect(route('society.buildings.index', $society));
    $this->assertDatabaseHas('buildings', [
        'society_id' => $society->id,
        'name' => 'Building A',
    ]);
});

test('society admin can update a building', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->put(route('society.buildings.update', [$society, $building]), [
            'name' => 'Updated Building',
            'floors' => $building->floors,
            'total_units' => $building->total_units,
        ]);

    $response->assertRedirect(route('society.buildings.index', $society));
    $this->assertDatabaseHas('buildings', ['id' => $building->id, 'name' => 'Updated Building']);
});

test('society admin can delete a building', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $building = Building::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->delete(route('society.buildings.destroy', [$society, $building]));

    $response->assertRedirect(route('society.buildings.index', $society));
    $this->assertSoftDeleted('buildings', ['id' => $building->id]);
});
