<?php

use App\Models\BillCycle;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('society admin can view bill cycles list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    BillCycle::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.bill-cycles.index', $society));

    $response->assertOk();
});

test('society admin can create a bill cycle', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->post(route('society.bill-cycles.store', $society), [
            'name' => 'January 2024',
            'year' => 2024,
            'generate_date' => now()->format('Y-m-d'),
            'due_date' => now()->addDays(15)->format('Y-m-d'),
        ]);

    $response->assertRedirect(route('society.bill-cycles.index', $society));
    $this->assertDatabaseHas('bill_cycles', [
        'society_id' => $society->id,
        'name' => 'January 2024',
    ]);
});

test('society admin can generate bills for a bill cycle', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    User::factory()->count(3)->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $billCycle = BillCycle::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->post(route('society.bill-cycles.generate', [$society, $billCycle]));

    $response->assertRedirect();
    // Verify bills were created for residents
    expect(\App\Models\Bill::where('bill_cycle_id', $billCycle->id)->count())->toBeGreaterThan(0);
});
