<?php

use App\Models\Bill;
use App\Models\BillCycle;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('society admin can view bills list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->get(route('society.bills.index', $society));

    $response->assertOk();
});

test('society admin can create a bill', function () {
    $society = Society::factory()->create();
    $adminRole = Role::where('name', 'society_admin')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $adminRole->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $billCycle = BillCycle::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->post(route('society.bills.store', $society), [
            'user_id' => $resident->id,
            'bill_cycle_id' => $billCycle->id,
            'amount' => 5000.00,
            'due_date' => now()->addDays(7)->format('Y-m-d'),
            'description' => 'Monthly maintenance',
        ]);

    $response->assertRedirect(route('society.bills.index', $society));
    $this->assertDatabaseHas('bills', [
        'society_id' => $society->id,
        'user_id' => $resident->id,
        'amount' => 5000.00,
    ]);
});

test('society admin can view a bill', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $bill = Bill::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.bills.show', [$society, $bill]));

    $response->assertOk();
});

test('society admin can revise a bill', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $bill = Bill::factory()->for($society)->create(['status' => 'due']);

    $response = $this->actingAs($admin)
        ->post(route('society.bills.revise', [$society, $bill]), [
            'amount' => 6000.00,
            'reason' => 'Additional charges',
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('bills', [
        'id' => $bill->id,
        'amount' => 6000.00,
    ]);
});

test('society admin can approve bills in bulk', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $bill1 = Bill::factory()->for($society)->create(['status' => 'under_review']);
    $bill2 = Bill::factory()->for($society)->create(['status' => 'under_review']);

    $response = $this->actingAs($admin)
        ->post(route('society.bills.approve', $society), [
            'bill_ids' => [$bill1->id, $bill2->id],
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('bills', ['id' => $bill1->id, 'status' => 'due']);
    $this->assertDatabaseHas('bills', ['id' => $bill2->id, 'status' => 'due']);
});

test('unauthorized user cannot access bills', function () {
    $society = Society::factory()->create();
    $otherSociety = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $otherSociety->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->get(route('society.bills.index', $society));

    $response->assertForbidden();
});
