<?php

use App\Models\BankAccount;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
    Role::firstOrCreate(['name' => 'accountant'], ['display_name' => 'Accountant']);
});

test('society admin can view bank accounts list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    BankAccount::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.bank-accounts.index', $society));

    $response->assertOk();
});

test('society admin can create a bank account', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->post(route('society.bank-accounts.store', $society), [
            'account_name' => 'Main Account',
            'account_number' => '1234567890',
            'ifsc_code' => 'SBIN0001234',
            'bank_name' => 'State Bank of India',
            'branch_name' => 'Main Branch',
            'account_type' => 'current',
            'balance' => 100000.00,
        ]);

    $response->assertRedirect(route('society.bank-accounts.index', $society));
    $this->assertDatabaseHas('bank_accounts', [
        'society_id' => $society->id,
        'account_name' => 'Main Account',
    ]);
});

test('society admin can update bank account balance', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $bankAccount = BankAccount::factory()->for($society)->create(['balance' => 100000.00]);

    $response = $this->actingAs($admin)
        ->post(route('society.bank-accounts.update-balance', [$society, $bankAccount]), [
            'balance' => 150000.00,
            'notes' => 'Balance updated',
        ]);

    $response->assertRedirect();
    $this->assertDatabaseHas('bank_accounts', ['id' => $bankAccount->id, 'balance' => 150000.00]);
});

test('accountant can also access bank accounts', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'accountant')->first();
    $accountant = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($accountant)
        ->get(route('society.bank-accounts.index', $society));

    $response->assertOk();
});
