<?php

use App\Models\Announcement;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('society admin can view announcements list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Announcement::factory()->for($society)->count(3)->create();

    $response = $this->actingAs($admin)
        ->get(route('society.announcements.index', $society));

    $response->assertOk();
});

test('society admin can create an announcement', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    $response = $this->actingAs($admin)
        ->post(route('society.announcements.store', $society), [
            'title' => 'Important Notice',
            'content' => 'This is an important announcement',
            'priority' => 'high',
            'is_published' => true,
        ]);

    $response->assertRedirect(route('society.announcements.index', $society));
    $this->assertDatabaseHas('announcements', [
        'society_id' => $society->id,
        'title' => 'Important Notice',
    ]);
});

test('society admin can toggle announcement publish status', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $announcement = Announcement::factory()->for($society)->create(['is_published' => false]);

    $response = $this->actingAs($admin)
        ->post(route('society.announcements.toggle-publish', [$society, $announcement]));

    $response->assertRedirect();
    $this->assertDatabaseHas('announcements', ['id' => $announcement->id, 'is_published' => true]);
});

test('society admin can delete an announcement', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $announcement = Announcement::factory()->for($society)->create();

    $response = $this->actingAs($admin)
        ->delete(route('society.announcements.destroy', [$society, $announcement]));

    $response->assertRedirect(route('society.announcements.index', $society));
    $this->assertSoftDeleted('announcements', ['id' => $announcement->id]);
});
