<?php

use App\Models\MaintenanceTicket;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use App\Models\Vendor;
use Laravel\Sanctum\Sanctum;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'vendor'], ['display_name' => 'Vendor']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('vendor can view assigned tickets', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'vendor')->first();
    $vendor = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $vendorModel = Vendor::factory()->for($society)->create(['user_id' => $vendor->id]);
    MaintenanceTicket::factory()->for($society)->create(['vendor_id' => $vendorModel->id]);

    Sanctum::actingAs($vendor);

    $response = $this->getJson('/api/vendor/tickets');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('vendor can update ticket status', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'vendor')->first();
    $vendor = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $vendorModel = Vendor::factory()->for($society)->create(['user_id' => $vendor->id]);
    $ticket = MaintenanceTicket::factory()->for($society)->create([
        'vendor_id' => $vendorModel->id,
        'status' => 'assigned',
    ]);

    Sanctum::actingAs($vendor);

    $response = $this->putJson("/api/vendor/tickets/{$ticket->id}", [
        'status' => 'in_progress',
        'notes' => 'Work started',
    ]);

    $response->assertOk();
    $this->assertDatabaseHas('maintenance_tickets', ['id' => $ticket->id, 'status' => 'in_progress']);
});

test('non-vendor cannot access vendor routes', function () {
    $society = Society::factory()->create();
    $residentRole = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    Sanctum::actingAs($resident);

    $response = $this->getJson('/api/vendor/tickets');

    $response->assertForbidden();
});
