<?php

namespace Tests\Feature\Api\Vendor;

use App\Models\MaintenanceTicket;
use App\Models\Society;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TicketTest extends TestCase
{
    use RefreshDatabase;

    protected $society;
    protected $vendor;
    protected $resident;

    protected function setUp(): void
    {
        parent::setUp();

        // Create society
        $this->society = Society::factory()->create();

        // Create vendor user
        $vendorRole = \App\Models\Role::firstOrCreate(['name' => 'vendor'], ['display_name' => 'Vendor']);
        
        // Assign permissions
        $permissions = [
            'maintenance.view',
            'maintenance.update_status',
        ];

        foreach ($permissions as $perm) {
            $permission = \App\Models\Permission::firstOrCreate(
                ['name' => $perm],
                [
                    'display_name' => ucwords(str_replace('.', ' ', $perm)),
                    'description' => 'Permission for ' . $perm,
                    'group' => 'maintenance',
                ]
            );
            $vendorRole->permissions()->syncWithoutDetaching([$permission->id]);
        }

        $this->vendor = User::factory()->create([
            'society_id' => $this->society->id,
            'role_id' => $vendorRole->id,
        ]);

        // Create resident user (ticket owner)
        $this->resident = User::factory()->create([
            'society_id' => $this->society->id,
            'role_id' => \App\Models\Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident'])->id,
        ]);
    }

    public function test_vendor_can_list_assigned_tickets()
    {
        MaintenanceTicket::factory()->count(3)->create([
            'society_id' => $this->society->id,
            'vendor_id' => $this->vendor->id,
            'status' => 'assigned',
        ]);

        // Create a ticket assigned to another vendor
        MaintenanceTicket::factory()->create([
            'society_id' => $this->society->id,
            'status' => 'assigned',
        ]);

        $response = $this->actingAs($this->vendor)
            ->getJson('/api/vendor/tickets');

        $response->assertStatus(200)
            ->assertJsonCount(3, 'data');
    }

    public function test_vendor_can_update_ticket_status()
    {
        $ticket = MaintenanceTicket::factory()->create([
            'society_id' => $this->society->id,
            'vendor_id' => $this->vendor->id,
            'status' => 'assigned',
        ]);

        $updateData = [
            'status' => 'in_progress',
            'resolution_notes' => 'Started working on it.',
        ];

        $response = $this->actingAs($this->vendor)
            ->putJson("/api/vendor/tickets/{$ticket->id}", $updateData);

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'id' => $ticket->id,
                    'status' => 'in_progress',
                    'resolution_notes' => 'Started working on it.',
                ]
            ]);

        $this->assertDatabaseHas('maintenance_tickets', [
            'id' => $ticket->id,
            'status' => 'in_progress',
            'resolution_notes' => 'Started working on it.',
        ]);
    }
}
