<?php

use App\Models\Bill;
use App\Models\MaintenanceTicket;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use App\Models\Visitor;
use Laravel\Sanctum\Sanctum;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('resident can view their bills', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Bill::factory()->for($society)->for($resident)->count(3)->create();

    Sanctum::actingAs($resident);

    $response = $this->getJson('/api/resident/bills');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('resident can view a specific bill', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $bill = Bill::factory()->for($society)->for($resident)->create();

    Sanctum::actingAs($resident);

    $response = $this->getJson("/api/resident/bills/{$bill->id}");

    $response->assertOk();
    $response->assertJsonPath('data.id', $bill->id);
});

test('resident can create maintenance ticket', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    Sanctum::actingAs($resident);

    $response = $this->postJson('/api/resident/tickets', [
        'title' => 'Leakage Issue',
        'description' => 'Water leakage in bathroom',
        'priority' => 'high',
        'category' => 'plumbing',
    ]);

    $response->assertCreated();
    $this->assertDatabaseHas('maintenance_tickets', [
        'society_id' => $society->id,
        'user_id' => $resident->id,
        'title' => 'Leakage Issue',
    ]);
});

test('resident can view their maintenance tickets', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    MaintenanceTicket::factory()->for($society)->for($resident)->count(3)->create();

    Sanctum::actingAs($resident);

    $response = $this->getJson('/api/resident/tickets');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('resident can view their visitors', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Visitor::factory()->for($society)->for($resident)->count(3)->create();

    Sanctum::actingAs($resident);

    $response = $this->getJson('/api/resident/visitors');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('resident can pre-approve a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    Sanctum::actingAs($resident);

    $response = $this->postJson('/api/resident/visitors/pre-approve', [
        'name' => 'John Visitor',
        'phone' => '1234567890',
        'purpose' => 'Meeting',
        'expected_date' => now()->addDay()->format('Y-m-d'),
        'expected_time' => '14:00',
    ]);

    $response->assertCreated();
    $this->assertDatabaseHas('visitors', [
        'society_id' => $society->id,
        'user_id' => $resident->id,
        'name' => 'John Visitor',
        'status' => 'approved',
    ]);
});

test('resident can approve a pending visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $visitor = Visitor::factory()->for($society)->for($resident)->create(['status' => 'pending']);

    Sanctum::actingAs($resident);

    $response = $this->postJson("/api/resident/visitors/{$visitor->id}/approve");

    $response->assertOk();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'approved']);
});

test('resident can reject a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $visitor = Visitor::factory()->for($society)->for($resident)->create(['status' => 'pending']);

    Sanctum::actingAs($resident);

    $response = $this->postJson("/api/resident/visitors/{$visitor->id}/reject");

    $response->assertOk();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'rejected']);
});
