<?php

namespace Tests\Feature\Api\Resident;

use App\Models\Bill;
use App\Models\BillCycle;
use App\Models\Society;
use App\Models\Unit;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class BillTest extends TestCase
{
    use RefreshDatabase;

    protected $society;
    protected $resident;
    protected $unit;
    protected $bill;

    protected function setUp(): void
    {
        parent::setUp();

        // Create society
        $this->society = Society::factory()->create();

        // Create unit
        $this->unit = Unit::factory()->create([
            'society_id' => $this->society->id,
            'unit_no' => '101',
        ]);

        // Create resident user
        $this->resident = User::factory()->create([
            'society_id' => $this->society->id,
            'unit_id' => $this->unit->id,
            'role_id' => \App\Models\Role::where('name', 'resident')->first()->id ?? \App\Models\Role::factory()->create(['name' => 'resident'])->id,
        ]);

        // Create bill cycle
        $cycle = BillCycle::factory()->create([
            'society_id' => $this->society->id,
        ]);

        // Create bill for resident
        $this->bill = Bill::factory()->create([
            'society_id' => $this->society->id,
            'user_id' => $this->resident->id,
            'bill_cycle_id' => $cycle->id,
            'amount' => 5000,
            'status' => 'due',
        ]);
    }

    public function test_resident_can_view_own_bills()
    {
        $response = $this->actingAs($this->resident)
            ->getJson('/api/resident/bills');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    '*' => [
                        'id',
                        'bill_no',
                        'amount',
                        'status',
                        'due_date',
                    ]
                ]
            ]);
    }

    public function test_resident_can_view_own_bill_details()
    {
        $response = $this->actingAs($this->resident)
            ->getJson("/api/resident/bills/{$this->bill->id}");

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'id' => $this->bill->id,
                    'amount' => '5000.00',
                ]
            ]);
    }

    public function test_resident_cannot_view_others_bills()
    {
        $otherResident = User::factory()->create([
            'society_id' => $this->society->id,
            'role_id' => $this->resident->role_id,
        ]);

        $response = $this->actingAs($otherResident)
            ->getJson("/api/resident/bills/{$this->bill->id}");

        $response->assertStatus(403);
    }
}
