<?php

namespace Tests\Feature\Api\Mobile;

use App\Models\Announcement;
use App\Models\CommunityPost;
use App\Models\Society;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class MobileApiTest extends TestCase
{
    use RefreshDatabase;

    protected $society;
    protected $resident;

    protected function setUp(): void
    {
        parent::setUp();

        $this->society = Society::factory()->create();

        $role = \App\Models\Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
        $unit = \App\Models\Unit::factory()->create(['society_id' => $this->society->id]);
        
        $this->resident = User::factory()->create([
            'society_id' => $this->society->id,
            'role_id' => $role->id,
            'unit_id' => $unit->id,
        ]);
    }

    public function test_can_get_profile()
    {
        $response = $this->actingAs($this->resident)
            ->getJson('/api/profile');

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'id' => $this->resident->id,
                    'name' => $this->resident->name,
                    'society' => [
                        'id' => $this->society->id,
                    ],
                ]
            ]);
    }

    public function test_can_list_announcements()
    {
        Announcement::factory()->count(3)->create([
            'society_id' => $this->society->id,
            'is_published' => true,
            'start_date' => now()->subDay(),
        ]);

        $response = $this->actingAs($this->resident)
            ->getJson('/api/announcements');

        $response->assertStatus(200)
            ->assertJsonCount(3, 'data');
    }

    public function test_can_acknowledge_announcement()
    {
        $announcement = Announcement::factory()->create([
            'society_id' => $this->society->id,
            'is_published' => true,
            'require_acknowledgement' => true,
        ]);

        $response = $this->actingAs($this->resident)
            ->postJson("/api/announcements/{$announcement->id}/acknowledge");

        $response->assertStatus(200);

        $this->assertDatabaseHas('announcement_acknowledgements', [
            'announcement_id' => $announcement->id,
            'user_id' => $this->resident->id,
        ]);
    }

    public function test_can_list_community_posts()
    {
        $group = \App\Models\CommunityGroup::create([
            'society_id' => $this->society->id,
            'name' => 'General',
            'description' => 'General',
        ]);

        CommunityPost::create([
            'society_id' => $this->society->id,
            'group_id' => $group->id,
            'user_id' => $this->resident->id,
            'title' => 'Hello',
            'content' => 'Hello World',
            'status' => 'published',
        ]);

        $response = $this->actingAs($this->resident)
            ->getJson('/api/community/posts');

        $response->assertStatus(200)
            ->assertJsonCount(1, 'data');
    }

    public function test_can_create_community_post()
    {
        $postData = [
            'title' => 'New Post Title',
            'content' => 'New Post',
        ];

        $response = $this->actingAs($this->resident)
            ->postJson('/api/community/posts', $postData);

        $response->assertStatus(201)
            ->assertJson([
                'data' => [
                    'content' => 'New Post',
                ]
            ]);
    }

    public function test_resident_can_create_ticket()
    {
        $ticketData = [
            'category' => 'plumbing',
            'title' => 'Leaky Tap',
            'description' => 'Kitchen sink tap is leaking.',
            'priority' => 'medium',
        ];

        $response = $this->actingAs($this->resident)
            ->postJson('/api/resident/tickets', $ticketData);

        $response->assertStatus(201)
            ->assertJson([
                'data' => [
                    'title' => 'Leaky Tap',
                    'status' => 'open',
                ]
            ]);
    }

    public function test_resident_can_pre_approve_visitor()
    {
        $visitorData = [
            'name' => 'Guest Name',
            'phone' => '9876543210',
            'type' => 'guest',
            'expected_date' => now()->addDay()->format('Y-m-d'),
        ];

        $response = $this->actingAs($this->resident)
            ->postJson('/api/resident/visitors/pre-approve', $visitorData);

        $response->assertStatus(201)
            ->assertJson([
                'data' => [
                    'name' => 'Guest Name',
                    'status' => 'approved',
                ]
            ]);
    }
}
