<?php

namespace Tests\Feature\Api\Guard;

use App\Models\Society;
use App\Models\User;
use App\Models\Visitor;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class VisitorTest extends TestCase
{
    use RefreshDatabase;

    protected $society;
    protected $guard;
    protected $resident;

    protected function setUp(): void
    {
        parent::setUp();

        // Create society
        $this->society = Society::factory()->create();

        // Create guard user
        $guardRole = \App\Models\Role::firstOrCreate(['name' => 'guard'], ['display_name' => 'Guard']);
        
        // Create permissions and assign to guard role
        $permissions = [
            'visitors.view',
            'visitors.check_in',
            'visitors.check_out',
        ];

        foreach ($permissions as $perm) {
            $permission = \App\Models\Permission::firstOrCreate(
                ['name' => $perm],
                [
                    'display_name' => ucwords(str_replace('.', ' ', $perm)),
                    'description' => 'Permission for ' . $perm,
                    'group' => 'visitors',
                ]
            );
            $guardRole->permissions()->syncWithoutDetaching([$permission->id]);
        }

        $this->guard = User::factory()->create([
            'society_id' => $this->society->id,
            'role_id' => $guardRole->id,
        ]);

        // Create resident user (to be visited)
        $this->resident = User::factory()->create([
            'society_id' => $this->society->id,
            'role_id' => \App\Models\Role::where('name', 'resident')->first()->id ?? \App\Models\Role::factory()->create(['name' => 'resident'])->id,
        ]);
    }

    public function test_guard_can_list_active_visitors()
    {
        Visitor::factory()->count(3)->create([
            'society_id' => $this->society->id,
            'status' => 'checked_in',
        ]);

        $response = $this->actingAs($this->guard)
            ->getJson('/api/guard/visitors');

        $response->assertStatus(200)
            ->assertJsonCount(3, 'data');
    }

    public function test_guard_can_check_in_visitor()
    {
        $visitorData = [
            'name' => 'John Doe',
            'phone' => '1234567890',
            'vehicle_number' => 'MH-12-AB-1234',
            'type' => 'guest',
            'purpose' => 'Meeting',
            'resident_id' => $this->resident->id,
        ];

        $response = $this->actingAs($this->guard)
            ->postJson('/api/guard/visitors/check-in', $visitorData);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'name',
                    'status',
                    'entry_time',
                ]
            ]);

        $this->assertDatabaseHas('visitors', [
            'name' => 'John Doe',
            'status' => 'checked_in',
        ]);
    }

    public function test_guard_can_check_out_visitor()
    {
        $visitor = Visitor::factory()->create([
            'society_id' => $this->society->id,
            'status' => 'checked_in',
        ]);

        $response = $this->actingAs($this->guard)
            ->postJson("/api/guard/visitors/{$visitor->id}/check-out");

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'status' => 'checked_out',
                ]
            ]);

        $this->assertDatabaseHas('visitors', [
            'id' => $visitor->id,
            'status' => 'checked_out',
        ]);
    }
}
