<?php

use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use App\Models\Visitor;
use Laravel\Sanctum\Sanctum;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'guard'], ['display_name' => 'Guard']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('guard can view visitors list', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'guard')->first();
    $guard = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Visitor::factory()->for($society)->count(3)->create();

    Sanctum::actingAs($guard);

    $response = $this->getJson('/api/guard/visitors');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('guard can view expected visitors', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'guard')->first();
    $guard = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    Sanctum::actingAs($guard);

    $response = $this->getJson('/api/guard/visitors/expected');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('guard can verify visitor OTP', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'guard')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $guard = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $visitor = Visitor::factory()->for($society)->for($resident)->create([
        'status' => 'approved',
        'otp' => '123456',
    ]);

    Sanctum::actingAs($guard);

    $response = $this->postJson('/api/guard/visitors/verify-otp', [
        'visitor_id' => $visitor->id,
        'otp' => '123456',
    ]);

    $response->assertOk();
});

test('guard can check in a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'guard')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $guard = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $visitor = Visitor::factory()->for($society)->for($resident)->create(['status' => 'approved']);

    Sanctum::actingAs($guard);

    $response = $this->postJson('/api/guard/visitors/check-in', [
        'visitor_id' => $visitor->id,
    ]);

    $response->assertOk();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'checked_in']);
});

test('guard can check out a visitor', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'guard')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $guard = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $visitor = Visitor::factory()->for($society)->for($resident)->create([
        'status' => 'checked_in',
        'check_in_at' => now(),
    ]);

    Sanctum::actingAs($guard);

    $response = $this->postJson("/api/guard/visitors/{$visitor->id}/check-out");

    $response->assertOk();
    $this->assertDatabaseHas('visitors', ['id' => $visitor->id, 'status' => 'checked_out']);
});

test('non-guard cannot access guard routes', function () {
    $society = Society::factory()->create();
    $residentRole = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    Sanctum::actingAs($resident);

    $response = $this->getJson('/api/guard/visitors');

    $response->assertForbidden();
});
