<?php

use App\Models\EmergencyAlert;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Laravel\Sanctum\Sanctum;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
    Role::firstOrCreate(['name' => 'society_admin'], ['display_name' => 'Society Admin']);
});

test('resident can trigger emergency alert', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    Sanctum::actingAs($resident);

    $response = $this->postJson('/api/emergency/trigger', [
        'type' => 'medical',
        'location' => 'Unit 101',
        'description' => 'Medical emergency',
    ]);

    $response->assertCreated();
    $this->assertDatabaseHas('emergency_alerts', [
        'society_id' => $society->id,
        'user_id' => $resident->id,
        'status' => 'active',
    ]);
});

test('society admin can view active emergency alerts', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    EmergencyAlert::factory()->for($society)->create(['status' => 'active']);

    Sanctum::actingAs($admin);

    $response = $this->getJson('/api/emergency/active');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('society admin can resolve emergency alert', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'society_admin')->first();
    $admin = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    $alert = EmergencyAlert::factory()->for($society)->create(['status' => 'active']);

    Sanctum::actingAs($admin);

    $response = $this->postJson("/api/emergency/{$alert->id}/resolve", [
        'resolution_notes' => 'Emergency handled',
    ]);

    $response->assertOk();
    $this->assertDatabaseHas('emergency_alerts', ['id' => $alert->id, 'status' => 'resolved']);
});
