<?php

use App\Models\Bill;
use App\Models\CollectorWallet;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Laravel\Sanctum\Sanctum;

beforeEach(function () {
    Role::firstOrCreate(['name' => 'collector'], ['display_name' => 'Collector']);
    Role::firstOrCreate(['name' => 'resident'], ['display_name' => 'Resident']);
});

test('collector can view dashboard', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'collector')->first();
    $collector = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    CollectorWallet::factory()->for($collector)->create();

    Sanctum::actingAs($collector);

    $response = $this->getJson('/api/collector/dashboard');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('collector can view dues', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'collector')->first();
    $collector = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    Bill::factory()->for($society)->count(5)->create(['status' => 'due']);

    Sanctum::actingAs($collector);

    $response = $this->getJson('/api/collector/dues');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('collector can view cleared payments', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'collector')->first();
    $collector = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);

    Sanctum::actingAs($collector);

    $response = $this->getJson('/api/collector/cleared');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('collector can store a payment', function () {
    $society = Society::factory()->create();
    $collectorRole = Role::where('name', 'collector')->first();
    $residentRole = Role::where('name', 'resident')->first();
    $collector = User::factory()->create(['society_id' => $society->id, 'role_id' => $collectorRole->id]);
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);
    $bill = Bill::factory()->for($society)->for($resident)->create(['amount' => 5000.00]);

    Sanctum::actingAs($collector);

    $response = $this->postJson('/api/collector/payments', [
        'user_id' => $resident->id,
        'bill_id' => $bill->id,
        'amount' => 5000.00,
        'payment_method' => 'cash',
    ]);

    $response->assertCreated();
    $this->assertDatabaseHas('payments', [
        'society_id' => $society->id,
        'user_id' => $resident->id,
        'amount' => 5000.00,
    ]);
});

test('collector can view wallet', function () {
    $society = Society::factory()->create();
    $role = Role::where('name', 'collector')->first();
    $collector = User::factory()->create(['society_id' => $society->id, 'role_id' => $role->id]);
    CollectorWallet::factory()->for($collector)->create(['balance' => 10000.00]);

    Sanctum::actingAs($collector);

    $response = $this->getJson('/api/collector/wallet');

    $response->assertOk();
    $response->assertJsonStructure(['data']);
});

test('non-collector cannot access collector routes', function () {
    $society = Society::factory()->create();
    $residentRole = Role::where('name', 'resident')->first();
    $resident = User::factory()->create(['society_id' => $society->id, 'role_id' => $residentRole->id]);

    Sanctum::actingAs($resident);

    $response = $this->getJson('/api/collector/dashboard');

    $response->assertForbidden();
});
