<?php

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use Laravel\Fortify\Features;

use App\Models\SubscriptionPlan;
use App\Http\Controllers\OnboardingController;
use App\Http\Controllers\InstallController;

// Installation Wizard Routes (protected by CheckInstallation middleware)
Route::prefix('install')->name('install.')->group(function () {
    Route::get('/', [InstallController::class, 'index'])->name('index');
    Route::get('/welcome', [InstallController::class, 'welcome'])->name('welcome');
    Route::get('/requirements', [InstallController::class, 'requirements'])->name('requirements');
    Route::post('/requirements', [InstallController::class, 'checkRequirements']);
    Route::get('/database', [InstallController::class, 'database'])->name('database');
    Route::post('/database', [InstallController::class, 'configureDatabase']);
    Route::get('/admin', [InstallController::class, 'admin'])->name('admin');
    Route::post('/admin', [InstallController::class, 'createAdmin']);
    Route::get('/finish', [InstallController::class, 'finish'])->name('finish');
    Route::post('/complete', [InstallController::class, 'complete'])->name('complete');
});

Route::get('/', [\App\Http\Controllers\WelcomeController::class, 'index'])->middleware(\App\Http\Middleware\CheckInstallation::class)->name('home');

Route::get('/pricing', function () {
    $plans = SubscriptionPlan::where('is_active', true)
        ->orderBy('sort_order')
        ->get();

    return Inertia::render('Pricing', [
        'plans' => $plans,
    ]);
})->name('pricing');

// Legal Pages
Route::get('/privacy-policy', function () {
    $setting = \App\Models\SystemSetting::where('key', 'privacy_policy')->first();
    return Inertia::render('Legal', [
        'title' => 'Privacy Policy',
        'content' => $setting ? $setting->value : '<p>Content coming soon...</p>',
        'updated_at' => $setting ? $setting->updated_at : null,
    ]);
})->name('privacy-policy');

Route::get('/terms', function () {
    $setting = \App\Models\SystemSetting::where('key', 'terms_of_service')->first();
    return Inertia::render('Legal', [
        'title' => 'Terms of Service',
        'content' => $setting ? $setting->value : '<p>Content coming soon...</p>',
        'updated_at' => $setting ? $setting->updated_at : null,
    ]);
})->name('terms');

Route::get('/refund-policy', function () {
    $setting = \App\Models\SystemSetting::where('key', 'refund_policy')->first();
    return Inertia::render('Legal', [
        'title' => 'Refund Policy',
        'content' => $setting ? $setting->value : '<p>Content coming soon...</p>',
        'updated_at' => $setting ? $setting->updated_at : null,
    ]);
})->name('refund-policy');

Route::get('/about', function () {
    $setting = \App\Models\SystemSetting::where('key', 'about_us')->first();
    return Inertia::render('Legal', [
        'title' => 'About Us',
        'content' => $setting ? $setting->value : '<p>Content coming soon...</p>',
        'updated_at' => $setting ? $setting->updated_at : null,
    ]);
})->name('about');

Route::middleware(['auth', 'verified'])->group(function () {
    Route::get('dashboard', function () {
        $user = Auth::user();

        if (! $user) {
            return redirect()->route('login')
                ->with('error', 'User not found. Please login again.');
        }

        // Load role relationship if not already loaded
        if (! $user->relationLoaded('role')) {
            $user->load('role');
        }

        // If user is a web admin (society_admin, sub_admin, accountant) with a society, redirect to their society dashboard
        if ($user->isWebAdmin() && $user->society_id) {
            // Check role and route to appropriate dashboard
            $roleName = $user->role?->name ?? null;

            if ($roleName === 'sub_admin') {
                // Sub Admin has their own route file (sub-admin.php)
                return redirect()->route('sub-admin.dashboard', ['society' => $user->society_id]);
            } elseif ($roleName === 'accountant') {
                // Accountant has their own route file (accountant.php)
                return redirect()->route('accountant.dashboard', ['society' => $user->society_id]);
            } elseif ($roleName === 'society_admin') {
                // Society Admin route
                return redirect()->route('society.dashboard', ['society' => $user->society_id]);
            }
        }


        $plans = SubscriptionPlan::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        // Get payment gateway settings for onboarding
        $settings = [
            'razorpay_enabled' => \App\Models\SystemSetting::getValue('razorpay_enabled', '0'),
            'stripe_enabled' => \App\Models\SystemSetting::getValue('stripe_enabled', '0'),
            'paypal_enabled' => \App\Models\SystemSetting::getValue('paypal_enabled', '0'),
        ];

        return Inertia::render('Dashboard', [
            'plans' => $plans,
            'settings' => $settings,
        ]);
    })->name('dashboard');

    Route::post('/onboarding/society', [OnboardingController::class, 'store'])->name('onboarding.society.store');
    Route::post('/onboarding/payment/create-order', [OnboardingController::class, 'createPaymentOrder'])->name('onboarding.payment.create-order');

    // Subscription Payment Routes (for activating trial/pending/expired subscriptions)
    Route::post('/subscription/create-payment-order', [\App\Http\Controllers\Api\SubscriptionPaymentController::class, 'createPaymentOrder'])->name('subscription.create-payment-order');
    Route::post('/subscription/verify-payment', [\App\Http\Controllers\Api\SubscriptionPaymentController::class, 'verifyPayment'])->name('subscription.verify-payment');

    // Subscription Renew Page
    Route::get('/subscription/renew', function () {
        $user = Auth::user();
        $subscription = $user->society?->subscription;

        $paymentRequired = null;
        if ($subscription) {
            if (in_array($subscription->status, ['trial', 'pending_payment'])) {
                $paymentRequired = [
                    'status' => $subscription->status,
                    'message' => 'Complete your payment to activate your subscription.',
                ];
            } elseif ($subscription->isExpired()) {
                $paymentRequired = [
                    'status' => 'expired',
                    'message' => 'Your subscription has expired. Renew now to continue using the platform.',
                ];
            } else {
                $paymentRequired = [
                    'status' => 'active',
                    'message' => 'Renew your subscription early to avoid any service interruption.',
                ];
            }
        }

        return \Inertia\Inertia::render('SubscriptionRenew', [
            'subscription' => $subscription ? [
                'plan_name' => $subscription->plan_display_name ?? $subscription->plan_name,
                'status' => $subscription->status,
                'billing_cycle' => $subscription->billing_cycle,
                'end_date' => $subscription->end_date?->format('Y-m-d'),
                'monthly_price' => $subscription->monthly_price,
                'annual_price' => $subscription->annual_price,
            ] : null,
            'paymentRequired' => $paymentRequired,
        ]);
    })->name('subscription.renew');

    // Society Admin routes are handled in routes/society-admin.php
});

require __DIR__ . '/settings.php';
require __DIR__ . '/super-admin.php';
require __DIR__ . '/accountant.php';
require __DIR__ . '/sub-admin.php';
require __DIR__ . '/society-admin.php';
