<?php

use App\Http\Controllers\Api\AnnouncementController;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\CollectorController;
use App\Http\Controllers\Api\CollectorPaymentController;
use App\Http\Controllers\Api\AdController;
use App\Http\Controllers\Api\CommunityController;
use App\Http\Controllers\Api\FacilityController;
use App\Http\Controllers\Api\FacilityBookingController;
use App\Http\Controllers\Api\FcmTokenController;
use App\Http\Controllers\Api\GuardController;
use App\Http\Controllers\Api\ProfileController;
use App\Http\Controllers\Api\ResidentController;
use App\Http\Controllers\Api\SocietyController;
use App\Http\Controllers\Api\VendorController;
use App\Http\Controllers\Api\FamilyMemberController; // [NEW]
use App\Http\Controllers\Api\TenantController;       // [NEW]
use App\Http\Controllers\Api\RentRecordController;   // [NEW]
use App\Http\Controllers\Api\UserController;         // [NEW]
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

Route::get('/user', function (Request $request) {
    return $request->user();
})->middleware('auth:sanctum');

// Auth Routes (Public)
Route::post('/login', [AuthController::class, 'login']);
Route::post('/send-otp', [AuthController::class, 'sendOtp']);
Route::post('/verify-otp', [AuthController::class, 'verifyOtp']);
Route::post('/logout', [AuthController::class, 'logout'])->middleware('auth:sanctum');

Route::middleware(['auth:sanctum'])->group(function () {
    // Users API Routes
    Route::get('/users', [UserController::class, 'index']);
    Route::get('/users/stats', [UserController::class, 'stats']);
    Route::get('/users/{id}', [UserController::class, 'show']);

    // FCM Token registration for mobile users
    Route::post('/fcm-token', [FcmTokenController::class, 'updateToken']);

    // Emergency Panic Alert
    Route::post('/emergency/trigger', [\App\Http\Controllers\Api\EmergencyController::class, 'trigger']);
    Route::get('/emergency/active', [\App\Http\Controllers\Api\EmergencyController::class, 'active']);
    Route::post('/emergency/{alert}/resolve', [\App\Http\Controllers\Api\EmergencyController::class, 'resolve']);

    // Collector API Routes (requires collector role)
    Route::prefix('collector')->middleware('role:collector')->group(function () {
        Route::get('/dashboard', [CollectorController::class, 'dashboard']);
        Route::get('/dues', [CollectorController::class, 'dues']);
        Route::get('/cleared', [CollectorController::class, 'cleared']);
        Route::post('/payments', [CollectorController::class, 'storePayment']);
        Route::get('/reports', [CollectorController::class, 'reports']);
        Route::get('/units/{unit_id}', [CollectorController::class, 'unitDetails']);
        Route::get('/wallet', [CollectorController::class, 'wallet']);
        Route::post('/handover', [CollectorController::class, 'handover']);

        // Payment Config (get available gateways)
        Route::get('/payment-config', [\App\Http\Controllers\Api\SocietyController::class, 'paymentConfig']);

        // Advanced Payment Flow
        Route::get('/bills', [CollectorPaymentController::class, 'bills']);
        Route::post('/payment/collect', [CollectorPaymentController::class, 'collect']);
        Route::get('/payment/status/{request_id}', [CollectorPaymentController::class, 'checkStatus']);
        Route::get('/payment/history', [CollectorPaymentController::class, 'history']);

        // Facility Bookings (Collector can book on behalf of residents)
        Route::get('/facility-bookings', [FacilityBookingController::class, 'index']);
        Route::post('/facility-bookings', [FacilityBookingController::class, 'store']);
        Route::get('/facility-bookings/{id}', [FacilityBookingController::class, 'show']);
        Route::post('/facility-bookings/{id}/payment', [FacilityBookingController::class, 'updatePayment']);
        Route::post('/facility-bookings/{id}/check-in', [FacilityBookingController::class, 'checkIn']);
        Route::post('/facility-bookings/{id}/check-out', [FacilityBookingController::class, 'checkOut']);
    });

    // Resident API Routes
    Route::prefix('resident')->group(function () {
        // Payment Config (get available gateways)
        Route::get('/payment-config', [\App\Http\Controllers\Api\SocietyController::class, 'paymentConfig']);
        
        Route::get('/bills', [ResidentController::class, 'bills']);
        Route::get('/bills/{bill}', [ResidentController::class, 'showBill']);
        Route::post('/bills/create-order', [ResidentController::class, 'createOrder']);
        Route::post('/bills/verify-payment', [ResidentController::class, 'verifyPayment']);
        
        // Maintenance Tickets
        Route::get('/tickets', [ResidentController::class, 'tickets']);
        Route::post('/tickets', [ResidentController::class, 'storeTicket']);
        Route::get('/tickets/{ticket}', [ResidentController::class, 'showTicket']);
        
        // Ticket Comments (Resident)
        Route::get('/tickets/{ticket}/comments', [\App\Http\Controllers\Api\CommentController::class, 'index']);
        Route::post('/tickets/{ticket}/comments', [\App\Http\Controllers\Api\CommentController::class, 'store']);

        // Visitors
        Route::get('/visitors', [ResidentController::class, 'visitors']);
        Route::post('/visitors/pre-approve', [ResidentController::class, 'preApproveVisitor']);
        Route::post('/visitors/{visitor}/approve', [ResidentController::class, 'approveVisitor']);
        Route::post('/visitors/{visitor}/reject', [ResidentController::class, 'rejectVisitor']);
        
        // Payments
        Route::get('/payments', [ResidentController::class, 'payments']);

        // Facilities & Bookings
        Route::get('/facilities', [FacilityController::class, 'index']);
        Route::get('/facilities/{id}', [FacilityController::class, 'show']);
        Route::get('/facilities/{id}/slots', [FacilityController::class, 'getSlots']);
        Route::get('/facilities/{id}/availability', [FacilityController::class, 'checkAvailability']);
        Route::get('/facility-bookings', [FacilityBookingController::class, 'index']);
        Route::post('/facility-bookings', [FacilityBookingController::class, 'store']);
        Route::get('/facility-bookings/{id}', [FacilityBookingController::class, 'show']);
        Route::post('/facility-bookings/{id}/cancel', [FacilityBookingController::class, 'cancel']);
        
        // Razorpay Payment Integration for Facility Bookings
        Route::post('/facility-bookings/create-order', [\App\Http\Controllers\Resident\FacilityBookingController::class, 'createOrder']);
        Route::post('/facility-bookings/verify-payment', [\App\Http\Controllers\Resident\FacilityBookingController::class, 'verifyPayment']);
        Route::post('/facility-bookings/payment-failed', [\App\Http\Controllers\Resident\FacilityBookingController::class, 'paymentFailed']);

        // Family Members Management (Owner only)
        Route::get('/family-members', [FamilyMemberController::class, 'index']);
        Route::post('/family-members', [FamilyMemberController::class, 'store']);
        Route::delete('/family-members/{id}', [FamilyMemberController::class, 'destroy']);

        // Tenant Management (Owner only)
        Route::get('/tenants', [TenantController::class, 'index']);
        Route::post('/tenants', [TenantController::class, 'store']);
        Route::post('/tenants/{id}/documents', [TenantController::class, 'uploadDocument']);
        Route::delete('/tenants/{id}', [TenantController::class, 'destroy']);

        // Rent Management (Owner only)
        Route::get('/rent-records', [RentRecordController::class, 'ownerIndex']);
        Route::get('/rent-records/recent', [RentRecordController::class, 'recentTransactions']);
        Route::get('/tenants/{tenantId}/rent-records', [RentRecordController::class, 'tenantRecords']);
        Route::post('/rent-records', [RentRecordController::class, 'store']);
        Route::put('/rent-records/{id}', [RentRecordController::class, 'update']);
        Route::post('/rent-records/{id}/mark-paid', [RentRecordController::class, 'markPaid']);
        Route::post('/rent-records/{id}/approve-payment', [RentRecordController::class, 'approvePayment']);
    });

    // Common Mobile Routes (Authenticated)
    Route::get('/profile', [ProfileController::class, 'show']);
    Route::put('/profile', [ProfileController::class, 'update']);
    Route::post('/change-password', [ProfileController::class, 'changePassword']);
    
    // Announcements (Available to All Roles: Resident, Guard, Collector, Staff)
    Route::get('/announcements', [AnnouncementController::class, 'index']);
    Route::get('/announcements/{announcement}', [AnnouncementController::class, 'show']);
    Route::post('/announcements/{announcement}/acknowledge', [AnnouncementController::class, 'acknowledge']);

    // Community
    // Community
    Route::get('/community/posts', [CommunityController::class, 'index']);
    Route::post('/community/posts', [CommunityController::class, 'store']);
    Route::post('/community/posts/{id}/like', [CommunityController::class, 'like']);
    Route::post('/community/posts/{id}/comments', [CommunityController::class, 'comment']);
    Route::get('/community/directory', [CommunityController::class, 'directory']);

    // Ads
    Route::get('/ads/slots', [AdController::class, 'slots']);
    Route::post('/ads/create', [AdController::class, 'create']);
    Route::get('/ads/my-ads', [AdController::class, 'myAds']);
    Route::get('/ads/active', [AdController::class, 'active']);
    
    // Society Information
    Route::get('/society/info', [SocietyController::class, 'info']);
    Route::get('/society/payment-config', [SocietyController::class, 'paymentConfig']);
    Route::get('/society/buildings', [SocietyController::class, 'buildings']);

    // Guard API Routes (requires guard role)
    Route::prefix('guard')->middleware('role:guard')->group(function () {
        Route::get('/visitors', [GuardController::class, 'visitors']);
        Route::get('/visitors/expected', [GuardController::class, 'expectedVisitors']);
        Route::get('/visitors/verified', [GuardController::class, 'verifiedVisitors']);
        Route::post('/visitors/verify-otp', [GuardController::class, 'verifyVisitorOtp']);
        Route::post('/visitors/check-in', [GuardController::class, 'checkIn']);
        Route::post('/visitors/{visitor}/check-out', [GuardController::class, 'checkOut']);
    });

    // Tenant API Routes (requires tenant role)
    Route::prefix('tenant')->middleware('role:resident')->group(function () {
        Route::get('/rent-records', [RentRecordController::class, 'tenantIndex']);
        Route::get('/rent-records/recent', [RentRecordController::class, 'recentTransactions']);
    });

    // Vendor API Routes (requires vendor role)
    Route::prefix('vendor')->middleware('role:vendor')->group(function () {
        Route::get('/tickets', [VendorController::class, 'tickets']);
        Route::put('/tickets/{ticket}', [VendorController::class, 'updateTicket']);
    });

    // Society Admin API Routes (requires society_admin or sub_admin role)
    Route::prefix('society-admin')->middleware('role:society_admin,sub_admin')->group(function () {
        Route::post('/tickets/{ticket}/reply', [\App\Http\Controllers\Api\CommentController::class, 'store']);
    });
});

// Subscription Payment Routes (Authenticated - for activating trial/pending subscriptions)
Route::middleware(['auth:sanctum'])->group(function () {
    Route::get('/subscription/payment-gateways', [\App\Http\Controllers\Api\SubscriptionPaymentController::class, 'getPaymentGateways']);
    Route::post('/subscription/create-payment-order', [\App\Http\Controllers\Api\SubscriptionPaymentController::class, 'createPaymentOrder']);
    Route::post('/subscription/verify-payment', [\App\Http\Controllers\Api\SubscriptionPaymentController::class, 'verifyPayment']);
});

// Razorpay Webhook (Public - No Authentication)
Route::post('/webhooks/razorpay', [\App\Http\Controllers\WebhookController::class, 'handleRazorpay']);
