import { login } from '@/routes';
import { store } from '@/routes/register';
import { Form, Head, Link, usePage } from '@inertiajs/react';
import { useState } from 'react';

import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Spinner } from '@/components/ui/spinner';
import { useAppearance } from '@/hooks/use-appearance';
import { Building2, Lock, Mail, Moon, Sun, Users, Bell, Shield, TrendingUp, Sparkles, Zap, User, Phone } from 'lucide-react';

export default function Register() {
    const { settings } = usePage().props as any;
    const params = new URLSearchParams(window.location.search);
    const selectedPlan = params.get('plan');
    const isPaidPlan = selectedPlan && selectedPlan !== 'free';

    const [paymentMethod, setPaymentMethod] = useState<string>('');
    const { appearance, updateAppearance } = useAppearance();

    // Filter enabled gateways
    const gateways = [
        { id: 'stripe', name: 'Stripe', enabled: settings?.stripe_enabled === '1' },
        { id: 'razorpay', name: 'Razorpay', enabled: settings?.razorpay_enabled === '1' },
        { id: 'paypal', name: 'PayPal', enabled: settings?.paypal_enabled === '1' },
    ].filter(g => g.enabled);

    return (
        <>
            <Head title="Create an account" />
            
            <div className="relative flex min-h-screen overflow-hidden">
                {/* Left Side - Decorative Brand Section (60%) */}
                <div className="hidden lg:flex lg:w-[60%] relative bg-gradient-to-br from-sky-50 via-teal-50 to-teal-50 dark:from-slate-900 dark:via-teal-950 dark:to-teal-950">
                    {/* Soft Animated Background Gradient Orbs */}
                    <div className="absolute top-0 left-0 w-96 h-96 bg-cyan-200/40 dark:bg-cyan-500/20 rounded-full blur-3xl animate-pulse" />
                    <div className="absolute top-1/4 right-1/4 w-80 h-80 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }} />
                    <div className="absolute bottom-1/4 left-1/3 w-72 h-72 bg-cyan-200/40 dark:bg-cyan-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                    <div className="absolute bottom-0 right-0 w-96 h-96 bg-pink-200/40 dark:bg-pink-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '3s' }} />
                    
                    {/* Subtle Overlay */}
                    <div className="absolute inset-0 bg-white/10 dark:bg-black/10" />
                    
                    {/* Floating Icons Animation */}
                    <div className="absolute inset-0 overflow-hidden pointer-events-none">
                        <div className="absolute top-20 left-10 animate-float" style={{ animationDelay: '0s' }}>
                            <div className="w-16 h-16 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-12 shadow-lg">
                                <Users className="w-8 h-8 text-cyan-600 dark:text-cyan-400" />
                            </div>
                        </div>
                        <div className="absolute top-40 right-20 animate-float" style={{ animationDelay: '1s' }}>
                            <div className="w-20 h-20 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-12 shadow-lg">
                                <Bell className="w-10 h-10 text-teal-600 dark:text-teal-400" />
                            </div>
                        </div>
                        <div className="absolute bottom-40 left-20 animate-float" style={{ animationDelay: '2s' }}>
                            <div className="w-14 h-14 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-6 shadow-lg">
                                <Shield className="w-7 h-7 text-cyan-600 dark:text-cyan-400" />
                            </div>
                        </div>
                        <div className="absolute bottom-20 right-32 animate-float" style={{ animationDelay: '3s' }}>
                            <div className="w-18 h-18 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-6 shadow-lg">
                                <TrendingUp className="w-9 h-9 text-pink-600 dark:text-pink-400" />
                            </div>
                        </div>
                        <div className="absolute top-1/2 left-1/4 animate-float" style={{ animationDelay: '1.5s' }}>
                            <div className="w-12 h-12 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-45 shadow-lg">
                                <Sparkles className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
                            </div>
                        </div>
                        <div className="absolute top-1/3 right-1/3 animate-float" style={{ animationDelay: '2.5s' }}>
                            <div className="w-16 h-16 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-12 shadow-lg">
                                <Zap className="w-8 h-8 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                    </div>
                    
                    {/* Content */}
                    <div className="relative z-10 flex flex-col justify-center px-12 xl:px-16 w-full">
                        <div className="max-w-xl">
                            {/* Logo with Animation */}
                            <div className="mb-10 flex items-center gap-4 animate-fade-in-up">
                                <div className="flex h-16 w-16 items-center justify-center rounded-2xl bg-gradient-to-br from-sky-500 to-teal-600 shadow-2xl animate-bounce-slow">
                                    <Building2 className="h-9 w-9 text-white" />
                                </div>
                                <div>
                                    <span className="text-3xl font-bold text-gray-800 dark:text-white">
                                        Societify
                                    </span>
                                    <p className="text-gray-600 dark:text-gray-300 text-sm font-medium">Society Management</p>
                                </div>
                            </div>

                            <div className="animate-fade-in-up" style={{ animationDelay: '0.2s' }}>
                                <h1 className="text-5xl xl:text-6xl font-bold mb-6 leading-tight text-gray-900 dark:text-white">
                                    Join Our
                                    <br />
                                    <span className="bg-gradient-to-r from-sky-600 via-teal-600 to-teal-600 bg-clip-text text-transparent">
                                        Growing Community
                                    </span>
                                </h1>
                            </div>
                            
                            <p className="text-xl text-gray-700 dark:text-gray-300 mb-12 leading-relaxed animate-fade-in-up" style={{ animationDelay: '0.4s' }}>
                                Create your account today and start managing your society with ease.
                                Secure, efficient, and user-friendly.
                            </p>

                            {/* Animated Feature Cards */}
                            <div className="grid grid-cols-2 gap-4 animate-fade-in-up" style={{ animationDelay: '0.6s' }}>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-sky-500 to-cyan-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Users className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Community</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Connect with neighbors</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-teal-500 to-teal-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Shield className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Secure</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Your data is safe</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Right Side - Register Form (40%) */}
                <div className="flex-1 lg:w-[40%] flex items-center justify-center p-6 lg:p-12 bg-background relative overflow-y-auto">
                    {/* Theme Toggle - Pill Shape with System Option */}
                    <div className="absolute top-6 right-6 z-20">
                        <div className="inline-flex items-center rounded-full bg-muted p-1 shadow-sm border border-border/50">
                            <button
                                onClick={() => updateAppearance('light')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'light'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Sun className="h-4 w-4" />
                                <span className="hidden sm:inline">Light</span>
                            </button>
                            <button
                                onClick={() => updateAppearance('dark')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'dark'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Moon className="h-4 w-4" />
                                <span className="hidden sm:inline">Dark</span>
                            </button>
                            <button
                                onClick={() => updateAppearance('system')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'system'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Building2 className="h-4 w-4" />
                                <span className="hidden sm:inline">Auto</span>
                            </button>
                        </div>
                    </div>

                    <div className="w-full max-w-md py-8">
                        {/* Mobile Logo */}
                        <div className="lg:hidden mb-8 flex items-center justify-center gap-3">
                            <div className="flex h-12 w-12 items-center justify-center rounded-xl bg-gradient-to-br from-sky-500 to-teal-600">
                                <Building2 className="h-7 w-7 text-white" />
                            </div>
                            <span className="text-2xl font-bold bg-gradient-to-r from-sky-600 to-teal-600 bg-clip-text text-transparent">
                                Societify
                            </span>
                        </div>

                        {/* Register Card */}
                        <div className="rounded-2xl border border-border/50 bg-card/50 backdrop-blur-sm p-8 shadow-xl">
                            <div className="mb-8">
                                <h2 className="text-2xl font-bold text-foreground mb-2">
                                    Create an account
                                </h2>
                                <p className="text-muted-foreground">
                                    Enter your details below to get started
                                </p>
                            </div>

                            <Form
                                {...store.form()}
                                resetOnSuccess={['password', 'password_confirmation']}
                                disableWhileProcessing
                                className="space-y-6"
                                onSubmit={(e: any) => {
                                    // Handle payment method logic if needed
                                }}
                            >
                                {/* @ts-ignore */}
                                {({ processing, errors, setData, data }) => (
                                    <>
                                        {/* Name Field */}
                                        <div className="space-y-2">
                                            <Label htmlFor="name" className="text-sm font-medium">Name</Label>
                                            <div className="relative">
                                                <User className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    id="name"
                                                    type="text"
                                                    required
                                                    autoFocus
                                                    tabIndex={1}
                                                    autoComplete="name"
                                                    name="name"
                                                    placeholder="Full name"
                                                    value={data?.name}
                                                    onChange={(e) => setData && setData('name', e.target.value)}
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.name} />
                                        </div>

                                        {/* Email Field */}
                                        <div className="space-y-2">
                                            <Label htmlFor="email" className="text-sm font-medium">Email address</Label>
                                            <div className="relative">
                                                <Mail className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    id="email"
                                                    type="email"
                                                    required
                                                    tabIndex={2}
                                                    autoComplete="email"
                                                    name="email"
                                                    placeholder="name@example.com"
                                                    value={data?.email}
                                                    onChange={(e) => setData && setData('email', e.target.value)}
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.email} />
                                        </div>

                                        {/* Phone Field */}
                                        <div className="space-y-2">
                                            <Label htmlFor="phone" className="text-sm font-medium">Phone Number</Label>
                                            <div className="relative">
                                                <Phone className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    id="phone"
                                                    type="tel"
                                                    required
                                                    tabIndex={3}
                                                    autoComplete="tel"
                                                    name="phone"
                                                    placeholder="+91 98765 43210"
                                                    value={data?.phone}
                                                    onChange={(e) => setData && setData('phone', e.target.value)}
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.phone} />
                                        </div>

                                        {/* Password Field */}
                                        <div className="space-y-2">
                                            <Label htmlFor="password" className="text-sm font-medium">Password</Label>
                                            <div className="relative">
                                                <Lock className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    id="password"
                                                    type="password"
                                                    required
                                                    tabIndex={4}
                                                    autoComplete="new-password"
                                                    name="password"
                                                    placeholder="••••••••"
                                                    value={data?.password}
                                                    onChange={(e) => setData && setData('password', e.target.value)}
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.password} />
                                        </div>

                                        {/* Confirm Password Field */}
                                        <div className="space-y-2">
                                            <Label htmlFor="password_confirmation" className="text-sm font-medium">Confirm Password</Label>
                                            <div className="relative">
                                                <Lock className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    id="password_confirmation"
                                                    type="password"
                                                    required
                                                    tabIndex={5}
                                                    autoComplete="new-password"
                                                    name="password_confirmation"
                                                    placeholder="••••••••"
                                                    value={data?.password_confirmation}
                                                    onChange={(e) => setData && setData('password_confirmation', e.target.value)}
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.password_confirmation} />
                                        </div>

                                        {isPaidPlan && gateways.length > 0 && (
                                            <div className="space-y-2">
                                                <Label className="text-sm font-medium">Payment Method</Label>
                                                <div className="grid grid-cols-3 gap-4">
                                                    {gateways.map((gateway) => (
                                                        <div
                                                            key={gateway.id}
                                                            className={`border rounded-lg p-3 cursor-pointer flex items-center justify-center transition-all ${
                                                                paymentMethod === gateway.id
                                                                    ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20 ring-2 ring-blue-500 ring-offset-2'
                                                                    : 'hover:border-blue-200 dark:hover:border-blue-800 hover:bg-blue-50/50 dark:hover:bg-blue-900/10'
                                                            }`}
                                                            onClick={() => {
                                                                setPaymentMethod(gateway.id);
                                                                if (setData) {
                                                                    setData('payment_method', gateway.id);
                                                                    setData('plan', selectedPlan);
                                                                }
                                                            }}
                                                        >
                                                            <span className="font-medium text-sm">{gateway.name}</span>
                                                        </div>
                                                    ))}
                                                </div>
                                                {!paymentMethod && (
                                                    <p className="text-sm text-red-500">Please select a payment method.</p>
                                                )}
                                            </div>
                                        )}

                                        <Button
                                            type="submit"
                                            className="w-full h-11 text-base font-medium bg-gradient-to-r from-sky-600 to-teal-600 hover:from-sky-700 hover:to-teal-700 text-white shadow-lg hover:shadow-xl transition-all"
                                            tabIndex={6} 
                                            data-test="register-user-button"
                                            disabled={!!(isPaidPlan && !paymentMethod) || processing}
                                        >
                                            {processing && <Spinner />}
                                            {isPaidPlan ? 'Proceed to Payment' : 'Create account'}
                                        </Button>
                                    </>
                                )}
                            </Form>

                            {/* Footer */}
                            <div className="mt-8 pt-6 border-t border-border/50 text-center text-sm text-muted-foreground">
                                Already have an account?{' '}
                                <Link
                                    href={login()}
                                    className="text-cyan-600 hover:text-cyan-700 dark:text-cyan-400 dark:hover:text-cyan-300 font-medium transition-colors"
                                    tabIndex={7}
                                >
                                    Log in
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <style>{`
                @keyframes float {
                    0%, 100% {
                        transform: translateY(0px);
                    }
                    50% {
                        transform: translateY(-20px);
                    }
                }
                
                @keyframes bounce-slow {
                    0%, 100% {
                        transform: translateY(0);
                    }
                    50% {
                        transform: translateY(-10px);
                    }
                }
                
                @keyframes fade-in-up {
                    from {
                        opacity: 0;
                        transform: translateY(20px);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0);
                    }
                }
                
                .animate-float {
                    animation: float 6s ease-in-out infinite;
                }
                
                .animate-bounce-slow {
                    animation: bounce-slow 3s ease-in-out infinite;
                }
                
                .animate-fade-in-up {
                    animation: fade-in-up 0.8s ease-out forwards;
                    opacity: 0;
                }
            `}</style>
        </>
    );
}
