import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Spinner } from '@/components/ui/spinner';
import { register } from '@/routes';
import { request } from '@/routes/password';
import { store } from '@/routes/login';
import { useAppearance } from '@/hooks/use-appearance';
import { Form, Head, Link } from '@inertiajs/react';
import { Building2, Lock, Mail, Moon, Sun, Users, Bell, Shield, TrendingUp, Sparkles, Zap, Copy, Check } from 'lucide-react';
import { Badge } from '@/components/ui/badge';
import { useState, useRef } from 'react';

interface LoginProps {
    status?: string;
    canResetPassword: boolean;
    canRegister: boolean;
    demoMode?: boolean;
    demoCredentials?: {
        email: string;
        password: string;
        role?: string;
    } | Array<{
        email: string;
        password: string;
        role?: string;
    }> | null;
}

export default function Login({
    status,
    canResetPassword,
    demoMode,
    demoCredentials,
}: LoginProps) {
    const { appearance, updateAppearance } = useAppearance();
    const isDark = appearance === 'dark';
    const [copied, setCopied] = useState<number | null>(null);
    const emailRef = useRef<HTMLInputElement>(null);
    const passwordRef = useRef<HTMLInputElement>(null);

    // Normalize demoCredentials to always be an array
    const credentialsList = demoMode && demoCredentials 
        ? (Array.isArray(demoCredentials) ? demoCredentials : [demoCredentials])
        : [];

    const handleCopyAndFill = (credential: { email: string; password: string; role?: string }, index: number) => {
        if (emailRef.current && passwordRef.current) {
            emailRef.current.value = credential.email;
            passwordRef.current.value = credential.password;
            // Trigger input events for React
            emailRef.current.dispatchEvent(new Event('input', { bubbles: true }));
            passwordRef.current.dispatchEvent(new Event('input', { bubbles: true }));
            setCopied(index);
            setTimeout(() => setCopied(null), 2000);
        }
    };

    return (
        <>
            <Head title="Log in" />
            
            <div className="relative flex min-h-screen overflow-hidden">
                {/* Left Side - Decorative Brand Section (60%) */}
                <div className="hidden lg:flex lg:w-[60%] relative bg-gradient-to-br from-sky-50 via-cyan-50 to-teal-50 dark:from-slate-900 dark:via-sky-950 dark:to-cyan-950">
                    {/* Soft Animated Background Gradient Orbs */}
                    <div className="absolute top-0 left-0 w-96 h-96 bg-cyan-200/40 dark:bg-cyan-500/20 rounded-full blur-3xl animate-pulse" />
                    <div className="absolute top-1/4 right-1/4 w-80 h-80 bg-sky-200/40 dark:bg-sky-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }} />
                    <div className="absolute bottom-1/4 left-1/3 w-72 h-72 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                    <div className="absolute bottom-0 right-0 w-96 h-96 bg-emerald-200/40 dark:bg-emerald-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '3s' }} />
                    
                    {/* Subtle Overlay */}
                    <div className="absolute inset-0 bg-white/10 dark:bg-black/10" />
                    
                    {/* Floating Icons Animation */}
                    <div className="absolute inset-0 overflow-hidden pointer-events-none">
                        <div className="absolute top-20 left-10 animate-float" style={{ animationDelay: '0s' }}>
                            <div className="w-16 h-16 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-12 shadow-lg">
                                <Users className="w-8 h-8 text-cyan-600 dark:text-cyan-400" />
                            </div>
                        </div>
                        <div className="absolute top-40 right-20 animate-float" style={{ animationDelay: '1s' }}>
                            <div className="w-20 h-20 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-12 shadow-lg">
                                <Bell className="w-10 h-10 text-sky-600 dark:text-sky-400" />
                            </div>
                        </div>
                        <div className="absolute bottom-40 left-20 animate-float" style={{ animationDelay: '2s' }}>
                            <div className="w-14 h-14 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-6 shadow-lg">
                                <Shield className="w-7 h-7 text-teal-600 dark:text-teal-400" />
                            </div>
                        </div>
                        <div className="absolute bottom-20 right-32 animate-float" style={{ animationDelay: '3s' }}>
                            <div className="w-18 h-18 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-6 shadow-lg">
                                <TrendingUp className="w-9 h-9 text-emerald-600 dark:text-emerald-400" />
                            </div>
                        </div>
                        <div className="absolute top-1/2 left-1/4 animate-float" style={{ animationDelay: '1.5s' }}>
                            <div className="w-12 h-12 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-45 shadow-lg">
                                <Sparkles className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
                            </div>
                        </div>
                        <div className="absolute top-1/3 right-1/3 animate-float" style={{ animationDelay: '2.5s' }}>
                            <div className="w-16 h-16 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-12 shadow-lg">
                                <Zap className="w-8 h-8 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                    </div>
                    
                    {/* Content */}
                    <div className="relative z-10 flex flex-col justify-center px-12 xl:px-16 w-full">
                        <div className="max-w-xl">
                            {/* Logo with Animation */}
                            <div className="mb-10 flex items-center gap-4 animate-fade-in-up">
                                <div className="flex h-16 w-16 items-center justify-center rounded-2xl bg-gradient-to-br from-sky-500 to-cyan-600 shadow-2xl animate-bounce-slow">
                                    <Building2 className="h-9 w-9 text-white" />
                                </div>
                                <div>
                                    <span className="text-3xl font-bold text-gray-800 dark:text-white">
                                        Societify
                                    </span>
                                    <p className="text-gray-600 dark:text-gray-300 text-sm font-medium">Society Management</p>
                                </div>
                            </div>

                            <div className="animate-fade-in-up" style={{ animationDelay: '0.2s' }}>
                                <h1 className="text-5xl xl:text-6xl font-bold mb-6 leading-tight text-gray-900 dark:text-white">
                                    Transform Your
                                    <br />
                                    <span className="bg-gradient-to-r from-sky-600 via-cyan-600 to-teal-600 bg-clip-text text-transparent">
                                        Community Living
                                    </span>
                                </h1>
                            </div>
                            
                            <p className="text-xl text-gray-700 dark:text-gray-300 mb-12 leading-relaxed animate-fade-in-up" style={{ animationDelay: '0.4s' }}>
                                Experience seamless society management with our all-in-one platform. 
                                Manage bills, visitors, maintenance, and community engagement effortlessly.
                            </p>

                            {/* Animated Feature Cards */}
                            <div className="grid grid-cols-2 gap-4 animate-fade-in-up" style={{ animationDelay: '0.6s' }}>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-cyan-500 to-cyan-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Users className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Multi-Society</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Manage multiple communities</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-sky-500 to-sky-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Shield className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Secure & Safe</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Enterprise-grade security</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-teal-500 to-teal-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Bell className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Smart Alerts</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Real-time notifications</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-emerald-500 to-emerald-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <TrendingUp className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Analytics</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Comprehensive reports</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Right Side - Login Form (40%) */}
                <div className="flex-1 lg:w-[40%] flex items-center justify-center p-6 lg:p-12 bg-background relative">
                    {/* Theme Toggle - Pill Shape with System Option */}
                    <div className="absolute top-6 right-6 z-20">
                        <div className="inline-flex items-center rounded-full bg-muted p-1 shadow-sm border border-border/50">
                            <button
                                onClick={() => updateAppearance('light')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'light'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Sun className="h-4 w-4" />
                                <span className="hidden sm:inline">Light</span>
                            </button>
                            <button
                                onClick={() => updateAppearance('dark')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'dark'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Moon className="h-4 w-4" />
                                <span className="hidden sm:inline">Dark</span>
                            </button>
                            <button
                                onClick={() => updateAppearance('system')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'system'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Building2 className="h-4 w-4" />
                                <span className="hidden sm:inline">Auto</span>
                            </button>
                        </div>
                    </div>

                    <div className="w-full max-w-md">
                        {/* Mobile Logo */}
                        <div className="lg:hidden mb-8 flex items-center justify-center gap-3">
                            <div className="flex h-12 w-12 items-center justify-center rounded-xl bg-gradient-to-br from-sky-500 to-cyan-600">
                                <Building2 className="h-7 w-7 text-white" />
                            </div>
                            <span className="text-2xl font-bold bg-gradient-to-r from-sky-600 to-cyan-600 bg-clip-text text-transparent">
                                Societify
                            </span>
                        </div>

                        {/* Login Card */}
                        <div className="rounded-2xl border border-border/50 bg-card/50 backdrop-blur-sm p-8 shadow-xl">
                            <div className="mb-8">
                                <h2 className="text-2xl font-bold text-foreground mb-2">
                                    Welcome back
                                </h2>
                                <p className="text-muted-foreground">
                                    Sign in to access your society dashboard
                                </p>
                            </div>

                            {status && (
                                <div className="mb-6 rounded-lg bg-green-50 dark:bg-green-950/20 border border-green-200 dark:border-green-900/30 px-4 py-3 text-sm text-green-700 dark:text-green-400">
                                    {status}
                                </div>
                            )}

                            {/* Demo Credentials Cards */}
                            {demoMode && credentialsList.length > 0 && (
                                <div className="mb-6 space-y-3">
                                    <div className="flex items-center gap-2 mb-2">
                                        <Zap className="h-4 w-4 text-cyan-600 dark:text-cyan-400" />
                                        <span className="text-sm font-semibold text-foreground">Demo Credentials</span>
                                    </div>
                                    <div className="grid gap-3">
                                        {credentialsList.map((credential, index) => (
                                            <div 
                                                key={index}
                                                className="p-4 rounded-xl bg-cyan-500/10 border border-cyan-500/30 hover:bg-cyan-500/15 transition-colors"
                                            >
                                                <div className="flex items-center justify-between mb-3">
                                                    <Badge 
                                                        variant="outline" 
                                                        className="text-xs font-semibold bg-cyan-500/10 border-cyan-500/30 text-cyan-700 dark:text-cyan-300 px-2.5 py-0.5"
                                                    >
                                                        {credential.role || `Login ${index + 1}`}
                                                    </Badge>
                                                    <Button
                                                        type="button"
                                                        variant="outline"
                                                        size="sm"
                                                        onClick={() => handleCopyAndFill(credential, index)}
                                                        className="h-7 text-xs gap-1 border-cyan-500/30 hover:bg-cyan-500/20"
                                                    >
                                                        {copied === index ? (
                                                            <><Check className="h-3 w-3" /> Filled!</>
                                                        ) : (
                                                            <><Copy className="h-3 w-3" /> Copy</>  
                                                        )}
                                                    </Button>
                                                </div>
                                                <div className="space-y-1.5 text-sm">
                                                    <div className="flex items-center gap-2">
                                                        <span className="text-muted-foreground w-16 text-xs">Email:</span>
                                                        <code className="flex-1 bg-background/50 px-2 py-1 rounded text-cyan-700 dark:text-cyan-300 font-mono text-xs break-all">
                                                            {credential.email}
                                                        </code>
                                                    </div>
                                                    <div className="flex items-center gap-2">
                                                        <span className="text-muted-foreground w-16 text-xs">Password:</span>
                                                        <code className="flex-1 bg-background/50 px-2 py-1 rounded text-cyan-700 dark:text-cyan-300 font-mono text-xs">
                                                            {credential.password}
                                                        </code>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            )}

                            <Form
                                {...store.form()}
                                resetOnSuccess={['password']}
                                className="space-y-6"
                            >
                                {({ processing, errors }) => (
                                    <>
                                        {/* Email Field */}
                                        <div className="space-y-2">
                                            <Label htmlFor="email" className="text-sm font-medium">
                                                Email address
                                            </Label>
                                            <div className="relative">
                                                <Mail className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    ref={emailRef}
                                                    id="email"
                                                    type="email"
                                                    name="email"
                                                    required
                                                    autoFocus
                                                    tabIndex={1}
                                                    autoComplete="email"
                                                    placeholder="name@example.com"
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.email} />
                                        </div>

                                        {/* Password Field */}
                                        <div className="space-y-2">
                                            <div className="flex items-center justify-between">
                                                <Label htmlFor="password" className="text-sm font-medium">
                                                    Password
                                                </Label>
                                                {canResetPassword && (
                                                    <Link
                                                        href={request()}
                                                        className="text-sm text-cyan-600 hover:text-cyan-700 dark:text-cyan-400 dark:hover:text-cyan-300 font-medium transition-colors"
                                                        tabIndex={5}
                                                    >
                                                        Forgot password?
                                                    </Link>
                                                )}
                                            </div>
                                            <div className="relative">
                                                <Lock className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                                <Input
                                                    ref={passwordRef}
                                                    id="password"
                                                    type="password"
                                                    name="password"
                                                    required
                                                    tabIndex={2}
                                                    autoComplete="current-password"
                                                    placeholder="••••••••"
                                                    className="pl-10 h-11"
                                                />
                                            </div>
                                            <InputError message={errors.password} />
                                        </div>

                                        {/* Remember Me */}
                                        <div className="flex items-center space-x-2">
                                            <Checkbox
                                                id="remember"
                                                name="remember"
                                                tabIndex={3}
                                            />
                                            <Label 
                                                htmlFor="remember" 
                                                className="text-sm font-medium cursor-pointer select-none"
                                            >
                                                Remember me for 30 days
                                            </Label>
                                        </div>

                                        {/* Submit Button - Professional Blue */}

                                        <Button
                                            type="submit"
                                            className="w-full h-11 text-base font-medium bg-gradient-to-r from-sky-600 to-cyan-600 hover:from-sky-700 hover:to-cyan-700 text-white shadow-lg hover:shadow-xl transition-all"
                                            tabIndex={4}
                                            disabled={processing}
                                            data-test="login-button"
                                        >
                                            {processing && <Spinner />}
                                            {processing ? 'Signing in...' : 'Sign in'}
                                        </Button>
                                    </>
                                )}
                            </Form>

                            <div className="mt-6 text-center text-sm text-muted-foreground">
                                Don't have an account?{' '}
                                <Link
                                    href={register()}
                                    className="text-blue-600 hover:text-blue-700 dark:text-blue-400 dark:hover:text-blue-300 font-medium transition-colors"
                                    tabIndex={5}
                                >
                                    Create one
                                </Link>
                            </div>

                            {/* Footer */}
                            <div className="mt-8 pt-6 border-t border-border/50 text-center text-xs text-muted-foreground">
                                🔒 Protected by enterprise-grade security
                            </div>
                        </div>

                        {/* Support Text */}
                        <p className="mt-6 text-center text-sm text-muted-foreground">
                            Need help? Contact your society administrator
                        </p>
                    </div>
                </div>
            </div>

            <style>{`
                @keyframes float {
                    0%, 100% {
                        transform: translateY(0px);
                    }
                    50% {
                        transform: translateY(-20px);
                    }
                }
                
                @keyframes bounce-slow {
                    0%, 100% {
                        transform: translateY(0);
                    }
                    50% {
                        transform: translateY(-10px);
                    }
                }
                
                @keyframes fade-in-up {
                    from {
                        opacity: 0;
                        transform: translateY(20px);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0);
                    }
                }
                
                .animate-float {
                    animation: float 6s ease-in-out infinite;
                }
                
                .animate-bounce-slow {
                    animation: bounce-slow 3s ease-in-out infinite;
                }
                
                .animate-fade-in-up {
                    animation: fade-in-up 0.8s ease-out forwards;
                    opacity: 0;
                }
            `}</style>
        </>
    );
}
