import { Button } from '@/components/ui/button';
import PublicLayout from '@/layouts/public-layout';
import { Head, Link, usePage } from '@inertiajs/react';
import { ArrowRight, CheckCircle2, Shield, Users, Zap, Building2, BarChart3, Star, Lock, Smartphone, Globe, Play, Sparkles, TrendingUp, Bell, CreditCard } from 'lucide-react';
import * as Icons from 'lucide-react';
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from '@/components/ui/accordion';

interface WelcomeProps {
    features: any[];
    faqs: any[];
    testimonials: any[];
    cmsSettings: {
        hero_title: string;
        hero_subtitle: string;
        hero_badge: string;
        enable_features_section: boolean;
        enable_testimonials_section: boolean;
        enable_faq_section: boolean;
        enable_cta_section: boolean;
    };
    canRegister: boolean;
}

export default function Welcome({ features, faqs, testimonials, cmsSettings, canRegister }: WelcomeProps) {
    const { settings } = usePage().props as any;
    const appName = settings?.app_name || 'Societify';

    // Helper to render dynamic icon
    const renderIcon = (iconName: string, className?: string) => {
        // @ts-ignore
        const IconComponent = Icons[iconName] || Icons.Sparkles;
        return <IconComponent className={className} />;
    };

    return (
        <PublicLayout>
            <Head title={`Welcome to ${appName} - Premium Society Management`} />

            {/* Hero Section - Glassmorphism Design */}
            <section className="relative pt-28 pb-20 lg:pt-40 lg:pb-32 overflow-hidden">
                {/* Soft Gradient Background */}
                <div className="absolute inset-0 bg-gradient-to-b from-slate-50 via-white to-cyan-50/30 dark:from-slate-950 dark:via-slate-900 dark:to-cyan-950/20" />
                
                {/* Animated Gradient Orbs - Very Light */}
                <div className="absolute inset-0 overflow-hidden pointer-events-none">
                    <div className="absolute top-20 left-[10%] w-[500px] h-[500px] bg-gradient-to-br from-cyan-200/30 to-teal-200/20 dark:from-cyan-500/10 dark:to-teal-500/5 rounded-full blur-[100px] animate-pulse" />
                    <div className="absolute top-40 right-[5%] w-[400px] h-[400px] bg-gradient-to-br from-sky-200/25 to-blue-200/15 dark:from-sky-500/10 dark:to-blue-500/5 rounded-full blur-[80px] animate-pulse" style={{ animationDelay: '1s' }} />
                    <div className="absolute bottom-10 left-[30%] w-[350px] h-[350px] bg-gradient-to-br from-emerald-200/20 to-teal-200/15 dark:from-emerald-500/10 dark:to-teal-500/5 rounded-full blur-[80px] animate-pulse" style={{ animationDelay: '2s' }} />
                </div>

                {/* Subtle Grid Pattern */}
                <div className="absolute inset-0 bg-[linear-gradient(to_right,#94a3b810_1px,transparent_1px),linear-gradient(to_bottom,#94a3b810_1px,transparent_1px)] bg-[size:32px_32px] [mask-image:radial-gradient(ellipse_70%_50%_at_50%_0%,#000_60%,transparent_100%)]" />
                
                <div className="container mx-auto px-4 relative z-10">
                    <div className="max-w-5xl mx-auto text-center">
                        {/* Announcement Badge */}
                        <div className="inline-flex items-center gap-2 rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border border-slate-200/50 dark:border-slate-700/50 px-4 py-2 text-sm font-medium mb-8 shadow-sm animate-in fade-in slide-in-from-bottom-4 duration-700">
                            <span className="flex h-2 w-2 rounded-full bg-emerald-500 animate-pulse"></span>
                            <span className="text-slate-600 dark:text-slate-300">{cmsSettings?.hero_badge ? <span dangerouslySetInnerHTML={{ __html: cmsSettings.hero_badge }} /> : <>Trusted by <strong className="text-slate-900 dark:text-white">500+</strong> societies across India</>}</span>
                        </div>
                        
                        {/* Main Heading - Clean Typography */}
                        <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-bold tracking-tight mb-6 text-slate-900 dark:text-white animate-in fade-in slide-in-from-bottom-8 duration-1000">
                            {cmsSettings?.hero_title || 'Smart Society'}
                            <br />
                            <span className="bg-gradient-to-r from-cyan-600 via-teal-500 to-emerald-500 bg-clip-text text-transparent">Management Platform</span>
                        </h1>
                        
                        {/* Subtitle */}
                        <p className="text-lg md:text-xl text-slate-600 dark:text-slate-400 max-w-2xl mx-auto mb-10 animate-in fade-in slide-in-from-bottom-12 duration-1000 delay-200 leading-relaxed">
                            {cmsSettings?.hero_subtitle || 'Streamline billing, visitor management, and community engagement.'}
                        </p>
                        
                        {/* CTA Buttons */}
                        <div className="flex flex-col sm:flex-row items-center justify-center gap-4 animate-in fade-in slide-in-from-bottom-16 duration-1000 delay-300">
                            {settings?.enable_registration === '1' && (
                                <Link href="/register">
                                    <Button size="lg" className="h-13 px-8 text-base rounded-full bg-gradient-to-r from-cyan-600 to-teal-600 hover:from-cyan-700 hover:to-teal-700 shadow-lg shadow-cyan-500/20 hover:shadow-cyan-500/30 transition-all hover:scale-[1.02]">
                                        Get Started
                                        <ArrowRight className="ml-2 h-5 w-5" />
                                    </Button>
                                </Link>
                            )}
                            <Link href="#features">
                                <Button variant="outline" size="lg" className="h-13 px-8 text-base rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm border-slate-200 dark:border-slate-700 hover:bg-white dark:hover:bg-slate-800 transition-all hover:scale-[1.02]">
                                    {/* <Play className="mr-2 h-4 w-4" /> */}
                                   Features
                                    <Play className="mr-2 h-4 w-4" />
                                </Button>
                            </Link>
                        </div>
                    </div>

                    {/* Floating Glass Cards - Hero Metrics */}
                    <div className="mt-16 lg:mt-20 grid grid-cols-2 md:grid-cols-4 gap-4 max-w-4xl mx-auto animate-in fade-in slide-in-from-bottom-20 duration-1000 delay-500">
                        {[
                            { icon: Building2, label: 'Societies', value: '500+', color: 'text-cyan-600' },
                            { icon: Users, label: 'Residents', value: '50K+', color: 'text-emerald-600' },
                            { icon: CreditCard, label: 'Transactions', value: '₹10Cr+', color: 'text-blue-600' },
                            { icon: Star, label: 'Rating', value: '4.9/5', color: 'text-amber-600' },
                        ].map((stat, i) => (
                            <div key={i} className="group relative bg-white/70 dark:bg-slate-800/50 backdrop-blur-xl rounded-2xl border border-white/50 dark:border-slate-700/50 p-5 shadow-[0_8px_32px_rgba(0,0,0,0.06)] hover:shadow-[0_16px_48px_rgba(0,0,0,0.1)] transition-all duration-300 hover:-translate-y-1">
                                <div className="flex items-center gap-3">
                                    <div className="w-10 h-10 rounded-xl bg-slate-100 dark:bg-slate-700/50 flex items-center justify-center">
                                        <stat.icon className={`h-5 w-5 ${stat.color}`} />
                                    </div>
                                    <div>
                                        <p className="text-2xl font-bold text-slate-900 dark:text-white">{stat.value}</p>
                                        <p className="text-sm text-slate-500 dark:text-slate-400">{stat.label}</p>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>

                    {/* Dashboard Preview - Glassmorphism */}
                    <div className="mt-16 mx-auto max-w-5xl animate-in fade-in slide-in-from-bottom-24 duration-1000 delay-700">
                        <div className="relative rounded-2xl bg-white/60 dark:bg-slate-800/40 backdrop-blur-xl border border-white/50 dark:border-slate-700/50 shadow-[0_20px_80px_rgba(0,0,0,0.08)] overflow-hidden">
                            {/* Browser Bar */}
                            <div className="h-10 bg-slate-100/80 dark:bg-slate-800/80 border-b border-slate-200/50 dark:border-slate-700/50 flex items-center px-4 gap-2">
                                <div className="flex gap-1.5">
                                    <div className="w-3 h-3 rounded-full bg-red-400" />
                                    <div className="w-3 h-3 rounded-full bg-amber-400" />
                                    <div className="w-3 h-3 rounded-full bg-emerald-400" />
                                </div>
                                <div className="ml-4 flex-1 max-w-md h-6 bg-white/60 dark:bg-slate-700/60 rounded-md flex items-center px-3 text-xs text-slate-500">
                                    {typeof window !== 'undefined' ? window.location.host : 'societify.com'}/dashboard
                                </div>
                            </div>

                            {/* Dashboard Content Mock */}
                            <div className="aspect-[16/7] bg-gradient-to-br from-slate-50 to-white dark:from-slate-900 dark:to-slate-800 p-6 flex gap-6">
                                {/* Sidebar Mock */}
                                <div className="w-48 hidden md:flex flex-col gap-3">
                                    <div className="flex items-center gap-2 mb-4">
                                        <div className="h-8 w-8 rounded-lg bg-gradient-to-br from-cyan-500 to-teal-500 flex items-center justify-center text-white text-xs font-bold">S</div>
                                        <div className="h-3 w-20 bg-slate-200 dark:bg-slate-700 rounded-full" />
                                    </div>
                                    {[true, false, false, false].map((active, i) => (
                                        <div key={i} className={`h-9 rounded-lg flex items-center gap-2 px-3 ${active ? 'bg-cyan-50 dark:bg-cyan-900/20' : ''}`}>
                                            <div className={`h-4 w-4 rounded ${active ? 'bg-cyan-500' : 'bg-slate-200 dark:bg-slate-700'}`} />
                                            <div className={`h-2 rounded-full ${active ? 'w-16 bg-cyan-500/40' : 'w-12 bg-slate-200 dark:bg-slate-700'}`} />
                                        </div>
                                    ))}
                                </div>

                                {/* Main Content Mock */}
                                <div className="flex-1 space-y-4">
                                    <div className="flex justify-between items-center">
                                        <div className="space-y-1">
                                            <div className="h-4 w-32 bg-slate-200 dark:bg-slate-700 rounded-full" />
                                            <div className="h-2 w-48 bg-slate-100 dark:bg-slate-800 rounded-full" />
                                        </div>
                                        <div className="flex gap-2">
                                            <div className="h-8 w-8 rounded-full bg-slate-100 dark:bg-slate-700" />
                                            <div className="h-8 w-8 rounded-full bg-slate-100 dark:bg-slate-700" />
                                        </div>
                                    </div>
                                    
                                    <div className="grid grid-cols-3 gap-3">
                                        {['emerald', 'blue', 'amber'].map((color, i) => (
                                            <div key={i} className="p-4 rounded-xl bg-white dark:bg-slate-800 border border-slate-100 dark:border-slate-700 shadow-sm">
                                                <div className={`h-8 w-8 rounded-lg bg-${color}-100 dark:bg-${color}-900/30 mb-2 flex items-center justify-center`}>
                                                    <div className={`h-3 w-3 rounded bg-${color}-500`} />
                                                </div>
                                                <div className="h-2 w-10 bg-slate-100 dark:bg-slate-700 rounded-full mb-1" />
                                                <div className="h-4 w-16 bg-slate-200 dark:bg-slate-600 rounded-full" />
                                            </div>
                                        ))}
                                    </div>

                                    <div className="flex-1 p-4 rounded-xl bg-white dark:bg-slate-800 border border-slate-100 dark:border-slate-700 shadow-sm flex gap-4">
                                        <div className="flex-1 space-y-2">
                                            <div className="h-3 w-24 bg-slate-200 dark:bg-slate-700 rounded-full" />
                                            <div className="space-y-2">
                                                {[1, 2].map((i) => (
                                                    <div key={i} className="flex items-center gap-2">
                                                        <div className="h-6 w-6 rounded-full bg-slate-100 dark:bg-slate-700" />
                                                        <div className="h-2 flex-1 bg-slate-100 dark:bg-slate-800 rounded-full" />
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                        <div className="w-32 bg-slate-50 dark:bg-slate-900 rounded-lg p-3 flex items-end justify-center gap-1">
                                            {[25, 45, 35, 60, 40].map((h, i) => (
                                                <div key={i} className={`w-2 bg-cyan-${i === 3 ? '500' : '200'} dark:bg-cyan-${i === 3 ? '500' : '800'} rounded-t`} style={{ height: `${h}%` }} />
                                            ))}
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* Features Section - Glass Cards */}
            {cmsSettings?.enable_features_section && (
                <section id="features" className="py-24 lg:py-32 relative overflow-hidden bg-slate-50/50 dark:bg-slate-900/50">
                    <div className="container mx-auto px-4">
                        <div className="text-center max-w-3xl mx-auto mb-16">
                            <div className="inline-flex items-center gap-2 rounded-full bg-cyan-100/80 dark:bg-cyan-900/30 px-4 py-1.5 text-sm font-medium mb-6 text-cyan-700 dark:text-cyan-400">
                                <Sparkles className="h-4 w-4" />
                                Powerful Features
                            </div>
                            <h2 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 text-slate-900 dark:text-white">
                                Everything you need to run your society
                            </h2>
                            <p className="text-slate-600 dark:text-slate-400 text-lg">
                                Powerful tools designed to make society management effortless and transparent.
                            </p>
                        </div>

                        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                            {features && features.length > 0 ? features.map((feature, index) => (
                                <div key={index} className="group relative bg-white/80 dark:bg-slate-800/50 backdrop-blur-sm p-6 rounded-2xl border border-slate-200/50 dark:border-slate-700/50 shadow-sm hover:shadow-xl hover:shadow-slate-200/50 dark:hover:shadow-slate-900/50 transition-all duration-300 hover:-translate-y-1">
                                    <div className={`mb-4 w-12 h-12 rounded-xl bg-cyan-100 dark:bg-cyan-900/30 flex items-center justify-center`}>
                                        {renderIcon(feature.icon || 'Zap', "h-6 w-6 text-cyan-600 dark:text-cyan-400")}
                                    </div>
                                    <h3 className="text-xl font-semibold mb-2 text-slate-900 dark:text-white group-hover:text-cyan-600 dark:group-hover:text-cyan-400 transition-colors">{feature.title}</h3>
                                    <p className="text-slate-600 dark:text-slate-400">{feature.description}</p>
                                </div>
                            )) : (
                                <p className="col-span-full text-center text-muted-foreground">No features to display.</p>
                            )}
                        </div>
                    </div>
                </section>
            )}

            {/* User Roles Section - Glass Effect */}
            <section className="py-24 lg:py-32 relative overflow-hidden">
                <div className="container mx-auto px-4">
                    <div className="text-center max-w-3xl mx-auto mb-16">
                        <div className="inline-flex items-center gap-2 rounded-full bg-emerald-100/80 dark:bg-emerald-900/30 px-4 py-1.5 text-sm font-medium mb-6 text-emerald-700 dark:text-emerald-400">
                            <Users className="h-4 w-4" />
                            For Everyone
                        </div>
                        <h2 className="text-3xl md:text-4xl font-bold mb-6 text-slate-900 dark:text-white">Designed for Every Role</h2>
                        <p className="text-slate-600 dark:text-slate-400 text-lg">
                            Whether you're an Admin, Resident, or Guard, {appName} empowers you with tailored tools.
                        </p>
                    </div>

                    <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-5 max-w-5xl mx-auto">
                        {[
                            { role: 'Society Admin', desc: 'Full control over society, members, and billing.', icon: Building2 },
                            { role: 'Resident', desc: 'Pay bills, book facilities, and raise tickets.', icon: Users },
                            { role: 'Guard', desc: 'Visitor tracking with QR scanning & alerts.', icon: Shield },
                            { role: 'Accountant', desc: 'Manage finances, reports, and audits.', icon: BarChart3 },
                            { role: 'Super Admin', desc: 'Platform owner managing all societies.', icon: Globe },
                            { role: 'Staff', desc: 'Maintenance staff for ticket resolution.', icon: CheckCircle2 },
                        ].map((role, i) => (
                            <div key={i} className="bg-white/70 dark:bg-slate-800/40 backdrop-blur-sm p-5 rounded-2xl border border-slate-200/50 dark:border-slate-700/50 hover:border-cyan-300 dark:hover:border-cyan-700 transition-colors shadow-sm">
                                <div className="w-11 h-11 rounded-xl bg-gradient-to-br from-cyan-50 to-teal-50 dark:from-cyan-900/30 dark:to-teal-900/30 flex items-center justify-center text-cyan-600 dark:text-cyan-400 mb-4">
                                    <role.icon className="h-5 w-5" />
                                </div>
                                <h3 className="text-lg font-semibold mb-1 text-slate-900 dark:text-white">{role.role}</h3>
                                <p className="text-slate-600 dark:text-slate-400 text-sm">{role.desc}</p>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* How It Works - Clean Design */}
            <section className="py-24 lg:py-32 bg-slate-50/50 dark:bg-slate-900/50">
                <div className="container mx-auto px-4">
                    <div className="text-center max-w-3xl mx-auto mb-16">
                        <h2 className="text-3xl md:text-4xl font-bold mb-6 text-slate-900 dark:text-white">How It Works</h2>
                        <p className="text-slate-600 dark:text-slate-400 text-lg">
                            Get started with {appName} in three simple steps.
                        </p>
                    </div>

                    <div className="grid md:grid-cols-3 gap-8 max-w-4xl mx-auto relative">
                        {/* Connecting Line */}
                        <div className="hidden md:block absolute top-12 left-[20%] right-[20%] h-0.5 bg-gradient-to-r from-transparent via-cyan-300 dark:via-cyan-700 to-transparent" />

                        {[
                            { step: "01", title: "Register Your Society", desc: "Sign up and create your society profile." },
                            { step: "02", title: "Add Members & Units", desc: "Import member data and set up units." },
                            { step: "03", title: "Start Managing", desc: "Automate billing and engagement." },
                        ].map((item, index) => (
                            <div key={index} className="relative text-center">
                                <div className="w-20 h-20 mx-auto bg-white dark:bg-slate-800 border-4 border-cyan-200 dark:border-cyan-800 rounded-full flex items-center justify-center text-2xl font-bold text-cyan-600 dark:text-cyan-400 mb-6 shadow-lg relative z-10">
                                    {item.step}
                                </div>
                                <h3 className="text-xl font-semibold mb-2 text-slate-900 dark:text-white">{item.title}</h3>
                                <p className="text-slate-600 dark:text-slate-400">{item.desc}</p>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Testimonials - Glass Cards */}
            {cmsSettings?.enable_testimonials_section && (
                <section className="py-24 lg:py-32">
                    <div className="container mx-auto px-4">
                        <div className="text-center max-w-3xl mx-auto mb-16">
                            <h2 className="text-3xl md:text-4xl font-bold mb-6 text-slate-900 dark:text-white">Trusted by Societies Everywhere</h2>
                            <p className="text-slate-600 dark:text-slate-400 text-lg">
                                See what our users have to say about their experience.
                            </p>
                        </div>

                        <div className="grid md:grid-cols-3 gap-6 max-w-5xl mx-auto">
                            {testimonials && testimonials.length > 0 ? testimonials.map((testimonial, index) => (
                                <div key={index} className="bg-white/80 dark:bg-slate-800/50 backdrop-blur-sm p-6 rounded-2xl border border-slate-200/50 dark:border-slate-700/50 shadow-sm">
                                    <div className="flex gap-1 mb-4">
                                        {[...Array(testimonial.rating || 5)].map((_, i) => (
                                            <Star key={i} className="h-4 w-4 fill-amber-400 text-amber-400" />
                                        ))}
                                    </div>
                                    <p className="text-slate-700 dark:text-slate-300 mb-5 italic">"{testimonial.quote}"</p>
                                    <div>
                                        <p className="font-semibold text-slate-900 dark:text-white">{testimonial.author}</p>
                                        <p className="text-sm text-slate-500 dark:text-slate-400">{testimonial.role}</p>
                                    </div>
                                </div>
                            )) : (
                                <p className="col-span-full text-center text-muted-foreground">No testimonials available.</p>
                            )}
                        </div>
                    </div>
                </section>
            )}

            {/* FAQ Section - Minimal */}
            {cmsSettings?.enable_faq_section && (
                <section className="py-24 lg:py-32 bg-slate-50/50 dark:bg-slate-900/50">
                    <div className="container mx-auto px-4 max-w-3xl">
                        <div className="text-center mb-12">
                            <h2 className="text-3xl md:text-4xl font-bold mb-4 text-slate-900 dark:text-white">Frequently Asked Questions</h2>
                            <p className="text-slate-600 dark:text-slate-400">
                                Common questions about {appName}.
                            </p>
                        </div>

                        <Accordion type="single" collapsible className="w-full space-y-3">
                            {faqs && faqs.length > 0 ? faqs.map((faq, index) => (
                                <AccordionItem key={index} value={`item-${index}`} className="bg-white/80 dark:bg-slate-800/50 backdrop-blur-sm rounded-xl border border-slate-200/50 dark:border-slate-700/50 px-5">
                                    <AccordionTrigger className="text-left font-medium text-slate-900 dark:text-white hover:no-underline py-4">{faq.question}</AccordionTrigger>
                                    <AccordionContent className="text-slate-600 dark:text-slate-400 pb-4">
                                        {faq.answer}
                                    </AccordionContent>
                                </AccordionItem>
                            )) : (
                                <p className="text-center text-muted-foreground">No FAQs available.</p>
                            )}
                        </Accordion>
                    </div>
                </section>
            )}

            {/* CTA Section - Soft Gradient */}
            {cmsSettings?.enable_cta_section && (
                <section className="py-20">
                    <div className="container mx-auto px-4">
                        <div className="relative rounded-3xl overflow-hidden">
                            {/* Soft Gradient Background */}
                            <div className="absolute inset-0 bg-gradient-to-br from-cyan-500 via-teal-500 to-emerald-500" />
                            <div className="absolute inset-0 bg-[url('data:image/svg+xml,%3Csvg viewBox=%220 0 200 200%22 xmlns=%22http://www.w3.org/2000/svg%22%3E%3Cfilter id=%22noiseFilter%22%3E%3CfeTurbulence type=%22fractalNoise%22 baseFrequency=%220.65%22 numOctaves=%223%22 stitchTiles=%22stitch%22/%3E%3C/filter%3E%3Crect width=%22100%25%22 height=%22100%25%22 filter=%22url(%23noiseFilter)%22/%3E%3C/svg%3E')] opacity-10" />
                            
                            {/* Floating Orbs */}
                            <div className="absolute top-0 right-0 w-64 h-64 bg-white/10 rounded-full blur-3xl" />
                            <div className="absolute bottom-0 left-0 w-48 h-48 bg-white/10 rounded-full blur-3xl" />
                            
                            <div className="relative z-10 p-12 lg:p-16 text-center text-white">
                                <h2 className="text-3xl md:text-4xl font-bold mb-4">Ready to transform your society?</h2>
                                <p className="text-cyan-100 text-lg mb-8 max-w-xl mx-auto">
                                    Join thousands of societies using {appName} to create better living experiences.
                                </p>
                                {settings?.enable_registration === '1' && (
                                    <Link href="/register">
                                        <Button size="lg" className="h-12 px-8 text-base font-semibold rounded-full bg-white text-teal-700 hover:bg-slate-100 shadow-xl hover:shadow-2xl transition-all hover:scale-[1.02]">
                                            Get Started Now
                                            <ArrowRight className="ml-2 h-5 w-5" />
                                        </Button>
                                    </Link>
                                )}
                            </div>
                        </div>
                    </div>
                </section>
            )}
        </PublicLayout>
    );
}
