import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link } from '@inertiajs/react';
import { ArrowLeft, Building2, Calendar, Edit, Mail, Phone, Shield, User, Smartphone, CheckCircle2, XCircle } from 'lucide-react';
import { Label } from '@/components/ui/label';
import { PageHeader } from '@/components/ui/page-header';
import { GlassCard } from '@/components/ui/glass-card';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Users',
        href: '/super-admin/users',
    },
    {
        title: 'Details',
        href: '',
    },
];

interface User {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    status: string;
    fcm_token: string | null;
    directory_visible: boolean;
    society: {
        id: number;
        name: string;
        code: string;
        city: string;
    } | null;
    role: {
        id: number;
        name: string;
        display_name: string | null;
    };
    created_at: string;
    updated_at: string;
}

interface ShowProps {
    user: User;
}

export default function Show({ user }: ShowProps) {
    const getStatusColor = (status: string) => {
        switch (status.toLowerCase()) {
            case 'active':
                return 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20';
            case 'suspended':
                return 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20';
            case 'inactive':
                return 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20';
            default:
                return 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20';
        }
    };

    const getRoleColor = (roleName: string) => {
        if (roleName.includes('admin')) return 'bg-purple-500/10 text-purple-600 dark:text-purple-400 border-purple-500/20';
        if (roleName.includes('guard')) return 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20';
        return 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20';
    };

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title={`User Details - ${user.name}`} />

            {/* Animated Background Blobs */}
            <div className="fixed inset-0 -z-10 overflow-hidden pointer-events-none">
                <div className="absolute top-0 -left-4 w-96 h-96 bg-blue-300 dark:bg-blue-900 rounded-full mix-blend-multiply dark:mix-blend-soft-light filter blur-3xl opacity-20 animate-blob" />
                <div className="absolute top-0 -right-4 w-96 h-96 bg-cyan-300 dark:bg-cyan-900 rounded-full mix-blend-multiply dark:mix-blend-soft-light filter blur-3xl opacity-20 animate-blob animation-delay-2000" />
                <div className="absolute -bottom-8 left-20 w-96 h-96 bg-indigo-300 dark:bg-indigo-900 rounded-full mix-blend-multiply dark:mix-blend-soft-light filter blur-3xl opacity-20 animate-blob animation-delay-4000" />
            </div>

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-[1600px] mx-auto w-full">
                <PageHeader 
                    title="User Profile"
                    description={`Viewing details for ${user.name}`}
                    actions={
                        <div className="flex items-center gap-2">
                            <Link href="/super-admin/users">
                                <Button variant="outline" size="sm" className="gap-2">
                                    <ArrowLeft className="h-4 w-4" />
                                    Back to Users
                                </Button>
                            </Link>
                            <Link href={`/super-admin/users/${user.id}/edit`}>
                                <Button size="sm" className="gap-2 bg-gradient-to-r from-orange-500 to-amber-500 hover:from-orange-600 hover:to-amber-600 shadow-lg shadow-orange-500/25">
                                    <Edit className="h-4 w-4" />
                                    Edit User
                                </Button>
                            </Link>
                        </div>
                    }
                />

                <div className="grid gap-6 lg:grid-cols-3">
                    {/* Left Column: Profile Card */}
                    <div className="space-y-6">
                        <GlassCard intensity="low" className="p-0 overflow-hidden text-center relative group">
                            <div className="absolute top-0 left-0 w-full h-32 bg-gradient-to-br from-blue-500/20 via-cyan-500/20 to-indigo-500/20" />
                            <div className="pt-12 pb-8 px-6 relative">
                                <Avatar className="h-24 w-24 mx-auto border-4 border-background shadow-xl mb-4 group-hover:scale-105 transition-transform duration-300">
                                    <AvatarImage src={`https://ui-avatars.com/api/?name=${user.name}&background=random&size=128`} />
                                    <AvatarFallback className="text-2xl">{user.name.substring(0, 2).toUpperCase()}</AvatarFallback>
                                </Avatar>
                                <h2 className="text-2xl font-bold mb-2">{user.name}</h2>
                                <div className="flex items-center justify-center gap-2 mb-6">
                                    <Badge variant="outline" className={`${getRoleColor(user.role.name)} border capitalize shadow-sm`}>
                                        {user.role.display_name || user.role.name}
                                    </Badge>
                                    <Badge variant="outline" className={`${getStatusColor(user.status)} border capitalize shadow-sm`}>
                                        {user.status}
                                    </Badge>
                                </div>
                                
                                <div className="flex justify-center gap-6 text-sm">
                                    <div className="flex flex-col items-center p-3 rounded-xl bg-background/40 w-full">
                                        <span className="text-xs text-muted-foreground uppercase tracking-wider mb-1">Joined</span>
                                        <span className="font-semibold">{new Date(user.created_at).toLocaleDateString()}</span>
                                    </div>
                                </div>
                            </div>
                        </GlassCard>

                        {/* App Status Card */}
                        <GlassCard intensity="low" className="p-0 overflow-hidden">
                            <div className="p-6">
                                <div className="flex items-center gap-4 mb-4">
                                    <div className="p-2 rounded-lg bg-emerald-500/10 text-emerald-600 dark:text-emerald-400">
                                        <Smartphone className="h-5 w-5" />
                                    </div>
                                    <div>
                                        <h4 className="font-semibold text-emerald-900 dark:text-emerald-100">Mobile App Status</h4>
                                    </div>
                                </div>
                                {user.fcm_token ? (
                                    <div className="p-3 rounded-xl bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 flex items-center gap-2 text-sm font-medium">
                                        <CheckCircle2 className="h-4 w-4" />
                                        Device Connected
                                    </div>
                                ) : (
                                    <div className="p-3 rounded-xl bg-muted/50 text-muted-foreground flex items-center gap-2 text-sm font-medium">
                                        <XCircle className="h-4 w-4" />
                                        No Device Connected
                                    </div>
                                )}
                            </div>
                        </GlassCard>
                    </div>

                    {/* Right Column: Details */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Contact Information */}
                        <GlassCard intensity="low" className="p-0 overflow-hidden">
                            <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                                <div className="p-1.5 rounded-lg bg-blue-500/10 text-blue-600">
                                    <User className="h-4 w-4" />
                                </div>
                                <h3 className="font-semibold text-lg">Contact Information</h3>
                            </div>
                            <div className="p-6 grid gap-6 md:grid-cols-2">
                                <div className="space-y-1.5">
                                    <Label className="text-muted-foreground text-xs uppercase tracking-wider">Email Address</Label>
                                    <div className="flex items-center gap-2 font-medium p-2.5 rounded-lg bg-background/40 border border-border/50">
                                        <Mail className="h-4 w-4 text-muted-foreground" />
                                        {user.email || <span className="text-muted-foreground italic">Not provided</span>}
                                    </div>
                                </div>
                                <div className="space-y-1.5">
                                    <Label className="text-muted-foreground text-xs uppercase tracking-wider">Phone Number</Label>
                                    <div className="flex items-center gap-2 font-medium p-2.5 rounded-lg bg-background/40 border border-border/50">
                                        <Phone className="h-4 w-4 text-muted-foreground" />
                                        {user.phone}
                                    </div>
                                </div>
                                <div className="space-y-1.5">
                                    <Label className="text-muted-foreground text-xs uppercase tracking-wider">Directory Visibility</Label>
                                    <div className="flex items-center gap-2 font-medium">
                                        {user.directory_visible ? (
                                            <Badge variant="outline" className="bg-blue-500/10 text-blue-600 border-blue-500/20 shadow-sm">
                                                Public
                                            </Badge>
                                        ) : (
                                            <Badge variant="outline" className="bg-slate-500/10 text-slate-600 border-slate-500/20 shadow-sm">
                                                Private
                                            </Badge>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </GlassCard>

                        {/* Society Information */}
                        {user.society ? (
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-violet-500/10 text-violet-600">
                                        <Building2 className="h-4 w-4" />
                                    </div>
                                    <h3 className="font-semibold text-lg">Society Information</h3>
                                </div>
                                <div className="p-6 grid gap-6 md:grid-cols-2">
                                    <div className="space-y-1.5">
                                        <Label className="text-muted-foreground text-xs uppercase tracking-wider">Society Name</Label>
                                        <div className="font-medium text-lg">{user.society.name}</div>
                                    </div>
                                    <div className="space-y-1.5">
                                        <Label className="text-muted-foreground text-xs uppercase tracking-wider">Society Code</Label>
                                        <div className="font-medium font-mono bg-violet-500/10 text-violet-700 dark:text-violet-300 px-3 py-1 rounded-md w-fit border border-violet-500/20">
                                            {user.society.code}
                                        </div>
                                    </div>
                                    <div className="space-y-1.5">
                                        <Label className="text-muted-foreground text-xs uppercase tracking-wider">Location</Label>
                                        <div className="font-medium flex items-center gap-2">
                                            {user.society.city}
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>
                        ) : (
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="p-8 text-center text-muted-foreground flex flex-col items-center justify-center">
                                    <div className="p-4 rounded-full bg-muted/50 mb-4 ring-8 ring-muted/20">
                                        <Building2 className="h-8 w-8 text-muted-foreground/50" />
                                    </div>
                                    <p>This user is not associated with any society</p>
                                    <Badge variant="secondary" className="mt-2">System User</Badge>
                                </div>
                            </GlassCard>
                        )}

                        {/* System Info */}
                        <GlassCard intensity="low" className="p-0 overflow-hidden">
                            <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                                <div className="p-1.5 rounded-lg bg-indigo-500/10 text-indigo-600">
                                    <Shield className="h-4 w-4" />
                                </div>
                                <h3 className="font-semibold text-lg">System Information</h3>
                            </div>
                            <div className="p-6 grid gap-6 md:grid-cols-2">
                                <div className="space-y-1.5">
                                    <Label className="text-muted-foreground text-xs uppercase tracking-wider">Created At</Label>
                                    <div className="flex items-center gap-2 font-medium p-2.5 rounded-lg bg-background/40 border border-border/50">
                                        <Calendar className="h-4 w-4 text-muted-foreground" />
                                        {new Date(user.created_at).toLocaleString()}
                                    </div>
                                </div>
                                <div className="space-y-1.5">
                                    <Label className="text-muted-foreground text-xs uppercase tracking-wider">Last Updated</Label>
                                    <div className="flex items-center gap-2 font-medium p-2.5 rounded-lg bg-background/40 border border-border/50">
                                        <Calendar className="h-4 w-4 text-muted-foreground" />
                                        {new Date(user.updated_at).toLocaleString()}
                                    </div>
                                </div>
                            </div>
                        </GlassCard>
                    </div>
                </div>
            </div>
        </SuperAdminLayout>
    );
}
