import { Button } from '@/components/ui/button';
import { CardContent } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Users, Plus, Search, Edit, Trash2, Eye, Shield, Building2, Phone, Mail, ChevronDown, ChevronRight, CheckCircle, XCircle, AlertTriangle } from 'lucide-react';
import { useState } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from '@/components/ui/collapsible';
import { PageHeader } from '@/components/ui/page-header';
import { GlassCard } from '@/components/ui/glass-card';
import { 
    ModernTable, 
    ModernTableHeader, 
    ModernTableBody, 
    ModernTableHead, 
    ModernTableRow, 
    ModernTableCell 
} from '@/components/ui/modern-table';

interface User {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    status: string;
    fcm_token: string | null;
    directory_visible: boolean;
    society_id: number | null;
    role: {
        id: number;
        name: string;
        display_name: string | null;
    };
    created_at: string;
}

interface Society {
    id: number;
    name: string;
    users: User[];
}

interface Role {
    id: number;
    name: string;
    display_name: string | null;
}

interface IndexProps {
    groupedSocieties: {
        data: Society[];
        links: any;
        meta: any;
    };
    systemUsers: User[];
    societies: { id: number; name: string }[];
    roles: Role[];
    filters: {
        search?: string;
        society_id?: string;
        role_id?: string;
        status?: string;
    };
}

export default function Index({ groupedSocieties, systemUsers, societies, roles, filters }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [societyId, setSocietyId] = useState(filters.society_id || 'all');
    const [roleId, setRoleId] = useState(filters.role_id || 'all');
    const [status, setStatus] = useState(filters.status || 'all');
    
    // State for open accordions - default to all open if filtered, otherwise empty
    const [openSocieties, setOpenSocieties] = useState<number[]>([]);

    const toggleSociety = (id: number) => {
        setOpenSocieties(prev => 
            prev.includes(id) ? prev.filter(sid => sid !== id) : [...prev, id]
        );
    };

    const handleSearch = () => {
        const filterParams: any = { search };
        if (societyId && societyId !== 'all') {
            filterParams.society_id = societyId;
        }
        if (roleId && roleId !== 'all') {
            filterParams.role_id = roleId;
        }
        if (status && status !== 'all') {
            filterParams.status = status;
        }

        router.get('/super-admin/users', filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this user?')) {
            router.delete(`/super-admin/users/${id}`, {
                preserveScroll: true,
            });
        }
    };

    return (
        <SuperAdminLayout breadcrumbs={[
            { title: 'Home', href: '/super-admin/dashboard' },
            { title: 'Users', href: '/super-admin/users' }
        ]}>
            <Head title="Users Management" />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-[1600px] mx-auto w-full">
                <PageHeader 
                    title="Users Management"
                    description="Manage system users, roles, and permissions across all societies."
                    actions={
                        <Link href="/super-admin/users/create">
                            <Button size="sm" className="gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25">
                                <Plus className="h-4 w-4" />
                                Add New User
                            </Button>
                        </Link>
                    }
                />

                {/* Filters */}
                <GlassCard intensity="low" className="p-6">
                    <div className="grid gap-4 md:grid-cols-4 lg:grid-cols-5 items-end">
                        <div className="md:col-span-2">
                            <Label htmlFor="search">Search</Label>
                            <div className="relative mt-1">
                                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                <Input
                                    id="search"
                                    placeholder="Search by name, email, or phone..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="pl-9 bg-background/50"
                                />
                            </div>
                        </div>
                        <div>
                            <Label htmlFor="society">Society</Label>
                            <Select value={societyId} onValueChange={setSocietyId}>
                                <SelectTrigger id="society" className="mt-1 bg-background/50">
                                    <SelectValue placeholder="All Societies" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Societies</SelectItem>
                                    {societies.map((society) => (
                                        <SelectItem key={society.id} value={society.id.toString()}>
                                            {society.name}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div>
                            <Label htmlFor="role">Role</Label>
                            <Select value={roleId} onValueChange={setRoleId}>
                                <SelectTrigger id="role" className="mt-1 bg-background/50">
                                    <SelectValue placeholder="All Roles" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Roles</SelectItem>
                                    {roles.map((role) => (
                                        <SelectItem key={role.id} value={role.id.toString()}>
                                            {role.display_name || role.name}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div>
                            <Button onClick={handleSearch} className="w-full bg-secondary/80 hover:bg-secondary text-secondary-foreground">
                                Apply Filters
                            </Button>
                        </div>
                    </div>
                </GlassCard>

                {/* System Users */}
                {systemUsers.length > 0 && (
                    <GlassCard intensity="low" className="p-0 overflow-hidden border-l-4 border-l-primary/50">
                        <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                            <Shield className="h-5 w-5 text-primary" />
                            <h3 className="font-semibold text-lg">System Administrators</h3>
                            <Badge variant="secondary" className="ml-2 bg-primary/10 text-primary hover:bg-primary/20">{systemUsers.length}</Badge>
                        </div>
                        <div className="relative w-full overflow-auto">
                            <ModernTable>
                                <ModernTableHeader className="bg-muted/10">
                                    <ModernTableRow>
                                        <ModernTableHead>User</ModernTableHead>
                                        <ModernTableHead>Contact</ModernTableHead>
                                        <ModernTableHead>Role</ModernTableHead>
                                        <ModernTableHead>Status</ModernTableHead>
                                        <ModernTableHead className="text-right">Actions</ModernTableHead>
                                    </ModernTableRow>
                                </ModernTableHeader>
                                <ModernTableBody>
                                    {systemUsers.map((user) => (
                                        <ModernTableRow key={user.id}>
                                            <ModernTableCell>
                                                <div className="flex items-center gap-3">
                                                    <Avatar className="h-9 w-9 border-2 border-white/20 shadow-sm">
                                                        <AvatarImage src={`https://ui-avatars.com/api/?name=${user.name}&background=random`} />
                                                        <AvatarFallback>{user.name.substring(0, 2).toUpperCase()}</AvatarFallback>
                                                    </Avatar>
                                                    <div>
                                                        <div className="font-medium text-sm flex items-center gap-2">
                                                            {user.name}
                                                            {!user.directory_visible && (
                                                                <Badge variant="outline" className="text-[10px] px-1 py-0 h-4 border-dashed border-muted-foreground/50 text-muted-foreground">
                                                                    Private
                                                                </Badge>
                                                            )}
                                                        </div>
                                                        {user.fcm_token && (
                                                            <div className="text-[10px] text-emerald-600 dark:text-emerald-400 flex items-center gap-1">
                                                                <span className="h-1.5 w-1.5 rounded-full bg-emerald-500 animate-pulse shadow-[0_0_8px_rgba(16,185,129,0.6)]" />
                                                                App Connected
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <div className="space-y-1">
                                                    {user.email && (
                                                        <div className="flex items-center gap-2 text-muted-foreground text-xs hover:text-foreground transition-colors">
                                                            <Mail className="h-3 w-3" />
                                                            {user.email}
                                                        </div>
                                                    )}
                                                    <div className="flex items-center gap-2 text-muted-foreground text-xs hover:text-foreground transition-colors">
                                                        <Phone className="h-3 w-3" />
                                                        {user.phone}
                                                    </div>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                {(() => {
                                                    let colorClass = 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20';
                                                    if (user.role.name.includes('admin')) colorClass = 'bg-cyan-500/10 text-cyan-600 dark:text-cyan-400 border-cyan-500/20';
                                                    if (user.role.name.includes('guard')) colorClass = 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20';
                                                    return (
                                                        <Badge variant="outline" className={`${colorClass} border capitalize text-xs`}>
                                                            {user.role.display_name || user.role.name}
                                                        </Badge>
                                                    );
                                                })()}
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                {(() => {
                                                    const statusColors: Record<string, string> = {
                                                        active: 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20',
                                                        suspended: 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20', 
                                                        inactive: 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20'
                                                    };
                                                    const icons: Record<string, any> = {
                                                        active: CheckCircle,
                                                        suspended: AlertTriangle,
                                                        inactive: XCircle
                                                    };
                                                    const statusKey = user.status.toLowerCase();
                                                    const StatusIcon = icons[statusKey] || XCircle;
                                                    const colorClass = statusColors[statusKey] || statusColors.inactive;

                                                    return (
                                                        <Badge variant="outline" className={`${colorClass} border capitalize text-xs gap-1`}>
                                                            <StatusIcon className="h-3 w-3" />
                                                            {user.status}
                                                        </Badge>
                                                    );
                                                })()}
                                            </ModernTableCell>
                                            <ModernTableCell className="text-right">
                                                <div className="flex items-center justify-end gap-1">
                                                    <Link href={`/super-admin/users/${user.id}`}>
                                                        <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-blue-600 hover:bg-blue-100 dark:hover:bg-blue-900/30">
                                                            <Eye className="h-4 w-4" />
                                                        </Button>
                                                    </Link>
                                                    <Link href={`/super-admin/users/${user.id}/edit`}>
                                                        <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-orange-600 hover:bg-orange-100 dark:hover:bg-orange-900/30">
                                                            <Edit className="h-4 w-4" />
                                                        </Button>
                                                    </Link>
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        className="h-8 w-8 hover:text-red-600 hover:bg-red-100 dark:hover:bg-red-900/30"
                                                        onClick={() => handleDelete(user.id)}
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </Button>
                                                </div>
                                            </ModernTableCell>
                                        </ModernTableRow>
                                    ))}
                                </ModernTableBody>
                            </ModernTable>
                        </div>
                    </GlassCard>
                )}

                {/* Society Users */}
                <div className="space-y-4">
                    {groupedSocieties.data.map((society) => (
                        <GlassCard key={society.id} intensity="low" className="p-0 overflow-hidden hover:shadow-lg transition-all duration-300">
                            <Collapsible
                                open={openSocieties.includes(society.id)}
                                onOpenChange={() => toggleSociety(society.id)}
                            >
                                <CollapsibleTrigger className="w-full">
                                    <div className="flex items-center justify-between px-6 py-4 bg-transparent hover:bg-muted/30 transition-colors">
                                        <div className="flex items-center gap-4">
                                            <div className="p-2.5 rounded-xl bg-gradient-to-br from-indigo-500/10 to-purple-500/10 border border-white/10 shadow-sm">
                                                <Building2 className="h-5 w-5 text-indigo-600 dark:text-indigo-400" />
                                            </div>
                                            <div className="text-left">
                                                <h3 className="font-semibold text-lg leading-none mb-1">{society.name}</h3>
                                                <p className="text-sm text-muted-foreground flex items-center gap-2">
                                                    <Users className="h-3.5 w-3.5" />
                                                    {society.users.length} Users
                                                </p>
                                            </div>
                                        </div>
                                        <div className="flex items-center justify-center w-8 h-8 rounded-full bg-muted/20 hover:bg-muted/40 transition-colors">
                                            {openSocieties.includes(society.id) ? (
                                                <ChevronDown className="h-4 w-4" />
                                            ) : (
                                                <ChevronRight className="h-4 w-4" />
                                            )}
                                        </div>
                                    </div>
                                </CollapsibleTrigger>
                                <CollapsibleContent>
                                    <div className="border-t border-border/50 animate-in slide-in-from-top-1 duration-200">
                                        <div className="relative w-full overflow-auto">
                                            <ModernTable>
                                                <ModernTableHeader className="bg-muted/10 h-10">
                                                    <ModernTableRow>
                                                        <ModernTableHead>User</ModernTableHead>
                                                        <ModernTableHead>Contact</ModernTableHead>
                                                        <ModernTableHead>Role</ModernTableHead>
                                                        <ModernTableHead>Status</ModernTableHead>
                                                        <ModernTableHead className="text-right">Actions</ModernTableHead>
                                                    </ModernTableRow>
                                                </ModernTableHeader>
                                                <ModernTableBody>
                                                    {society.users.map((user) => (
                                                        <ModernTableRow key={user.id}>
                                                            <ModernTableCell>
                                                                <div className="flex items-center gap-3">
                                                                    <Avatar className="h-9 w-9 border-2 border-white/20 shadow-sm">
                                                                        <AvatarImage src={`https://ui-avatars.com/api/?name=${user.name}&background=random`} />
                                                                        <AvatarFallback>{user.name.substring(0, 2).toUpperCase()}</AvatarFallback>
                                                                    </Avatar>
                                                                    <div>
                                                                        <div className="font-medium text-sm flex items-center gap-2">
                                                                            {user.name}
                                                                            {!user.directory_visible && (
                                                                                <Badge variant="outline" className="text-[10px] px-1 py-0 h-4 border-dashed border-muted-foreground/50 text-muted-foreground">
                                                                                    Private
                                                                                </Badge>
                                                                            )}
                                                                        </div>
                                                                        {user.fcm_token && (
                                                                            <div className="text-[10px] text-emerald-600 dark:text-emerald-400 flex items-center gap-1">
                                                                                <span className="h-1.5 w-1.5 rounded-full bg-emerald-500 animate-pulse shadow-[0_0_8px_rgba(16,185,129,0.6)]" />
                                                                                App Connected
                                                                            </div>
                                                                        )}
                                                                    </div>
                                                                </div>
                                                            </ModernTableCell>
                                                            <ModernTableCell>
                                                                <div className="space-y-1">
                                                                    {user.email && (
                                                                        <div className="flex items-center gap-2 text-muted-foreground text-xs hover:text-foreground transition-colors">
                                                                            <Mail className="h-3 w-3" />
                                                                            {user.email}
                                                                        </div>
                                                                    )}
                                                                    <div className="flex items-center gap-2 text-muted-foreground text-xs hover:text-foreground transition-colors">
                                                                        <Phone className="h-3 w-3" />
                                                                        {user.phone}
                                                                    </div>
                                                                </div>
                                                            </ModernTableCell>
                                                            <ModernTableCell>
                                                                {(() => {
                                                                    let colorClass = 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20';
                                                                    if (user.role.name.includes('admin')) colorClass = 'bg-cyan-500/10 text-cyan-600 dark:text-cyan-400 border-cyan-500/20';
                                                                    if (user.role.name.includes('guard')) colorClass = 'bg-orange-500/10 text-orange-600 dark:text-orange-400 border-orange-500/20';
                                                                    return (
                                                                        <Badge variant="outline" className={`${colorClass} border capitalize text-xs`}>
                                                                            {user.role.display_name || user.role.name}
                                                                        </Badge>
                                                                    );
                                                                })()}
                                                            </ModernTableCell>
                                                            <ModernTableCell>
                                                                {(() => {
                                                                    const statusColors: Record<string, string> = {
                                                                        active: 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20',
                                                                        suspended: 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20', 
                                                                        inactive: 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20'
                                                                    };
                                                                    const icons: Record<string, any> = {
                                                                        active: CheckCircle,
                                                                        suspended: AlertTriangle,
                                                                        inactive: XCircle
                                                                    };
                                                                    const statusKey = user.status.toLowerCase();
                                                                    const StatusIcon = icons[statusKey] || XCircle;
                                                                    const colorClass = statusColors[statusKey] || statusColors.inactive;

                                                                    return (
                                                                        <Badge variant="outline" className={`${colorClass} border capitalize text-xs gap-1`}>
                                                                            <StatusIcon className="h-3 w-3" />
                                                                            {user.status}
                                                                        </Badge>
                                                                    );
                                                                })()}
                                                            </ModernTableCell>
                                                            <ModernTableCell className="text-right">
                                                                <div className="flex items-center justify-end gap-1">
                                                                    <Link href={`/super-admin/users/${user.id}`}>
                                                                        <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-blue-600 hover:bg-blue-100 dark:hover:bg-blue-900/30">
                                                                            <Eye className="h-4 w-4" />
                                                                        </Button>
                                                                    </Link>
                                                                    <Link href={`/super-admin/users/${user.id}/edit`}>
                                                                        <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-orange-600 hover:bg-orange-100 dark:hover:bg-orange-900/30">
                                                                            <Edit className="h-4 w-4" />
                                                                        </Button>
                                                                    </Link>
                                                                    <Button
                                                                        variant="ghost"
                                                                        size="icon"
                                                                        className="h-8 w-8 hover:text-red-600 hover:bg-red-100 dark:hover:bg-red-900/30"
                                                                        onClick={() => handleDelete(user.id)}
                                                                    >
                                                                        <Trash2 className="h-4 w-4" />
                                                                    </Button>
                                                                </div>
                                                            </ModernTableCell>
                                                        </ModernTableRow>
                                                    ))}
                                                </ModernTableBody>
                                            </ModernTable>
                                        </div>
                                    </div>
                                </CollapsibleContent>
                            </Collapsible>
                        </GlassCard>
                    ))}

                    {groupedSocieties.data.length === 0 && systemUsers.length === 0 && (
                        <GlassCard className="flex flex-col items-center justify-center py-16 text-muted-foreground text-center">
                            <div className="p-4 rounded-full bg-muted/50 mb-4 ring-8 ring-muted/20">
                                <Users className="h-8 w-8 text-muted-foreground/50" />
                            </div>
                            <h3 className="text-lg font-medium mb-1">No users found</h3>
                            <p className="text-sm max-w-sm mx-auto">Try adjusting your search or filters to find what you're looking for.</p>
                        </GlassCard>
                    )}
                </div>

                {/* Pagination */}
                {groupedSocieties.links && groupedSocieties.links.length > 3 && (
                    <div className="flex items-center justify-between pt-4">
                        <div className="text-sm text-muted-foreground bg-background/50 px-3 py-1 rounded-full border border-border/50">
                            Showing <span className="font-medium text-foreground">{groupedSocieties.meta.from}</span> to <span className="font-medium text-foreground">{groupedSocieties.meta.to}</span> of <span className="font-medium text-foreground">{groupedSocieties.meta.total}</span> societies
                        </div>
                        <div className="flex gap-1">
                            {groupedSocieties.links.map((link: any, index: number) => (
                                <Link
                                    key={index}
                                    href={link.url || '#'}
                                    className={`px-3 py-1.5 rounded-lg text-sm font-medium transition-all ${
                                        link.active
                                            ? 'bg-primary text-primary-foreground shadow-md shadow-primary/20'
                                            : 'bg-background/50 hover:bg-background/80 text-muted-foreground hover:text-foreground border border-transparent hover:border-border/50'
                                    } ${!link.url && 'opacity-50 cursor-not-allowed hover:bg-transparent hover:text-muted-foreground'}`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                />
                            ))}
                        </div>
                    </div>
                )}
            </div>
        </SuperAdminLayout>
    );
}
