import { Button } from '@/components/ui/button';
import { CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { Head, Link, useForm } from '@inertiajs/react';
import { ArrowLeft, Building2, Lock, Mail, Phone, Save, Shield, User, UserPlus, CheckCircle, XCircle } from 'lucide-react';
import { FormEventHandler } from 'react';
import InputError from '@/components/input-error';
import { Switch } from '@/components/ui/switch';
import { PageHeader } from '@/components/ui/page-header';
import { GlassCard } from '@/components/ui/glass-card';

interface Society {
    id: number;
    name: string;
}

interface Role {
    id: number;
    name: string;
    display_name: string | null;
}

interface CreateProps {
    societies: Society[];
    roles: Role[];
}

export default function Create({ societies, roles }: CreateProps) {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        email: '',
        phone: '',
        password: '',
        password_confirmation: '',
        role_id: '',
        society_id: '',
        status: 'active',
        directory_visible: true,
    });

    const handleSubmit: FormEventHandler = (e) => {
        e.preventDefault();
        post('/super-admin/users');
    };

    return (
        <SuperAdminLayout breadcrumbs={[
            { title: 'Home', href: '/super-admin/dashboard' },
            { title: 'Users', href: '/super-admin/users' },
            { title: 'Create', href: '/super-admin/users/create' },
        ]}>
            <Head title="Create User" />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-[1600px] mx-auto w-full">
                <PageHeader 
                    title="Create User"
                    description="Add a new user to the system and assign roles."
                    actions={
                        <Link href="/super-admin/users">
                            <Button variant="outline" size="sm" className="gap-2">
                                <ArrowLeft className="h-4 w-4" />
                                Back to Users
                            </Button>
                        </Link>
                    }
                />

                <form onSubmit={handleSubmit} className="flex-1">
                    <div className="grid gap-6 lg:grid-cols-3">
                        {/* Main Form */}
                        <div className="lg:col-span-2 space-y-6">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-blue-500/10 text-blue-600">
                                        <User className="h-4 w-4" />
                                    </div>
                                    <h3 className="font-semibold text-lg">User Information</h3>
                                </div>
                                <div className="p-6 space-y-6">
                                    {/* Name */}
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Full Name *</Label>
                                        <div className="relative">
                                            <User className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                            <Input
                                                id="name"
                                                value={data.name}
                                                onChange={(e) => setData('name', e.target.value)}
                                                className="pl-9 bg-background/50"
                                                placeholder="Enter full name"
                                                required
                                            />
                                        </div>
                                        <InputError message={errors.name} />
                                    </div>

                                    {/* Contact Info */}
                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="email">Email Address</Label>
                                            <div className="relative">
                                                <Mail className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                                <Input
                                                    id="email"
                                                    type="email"
                                                    value={data.email}
                                                    onChange={(e) => setData('email', e.target.value)}
                                                    className="pl-9 bg-background/50"
                                                    placeholder="Enter email address"
                                                />
                                            </div>
                                            <InputError message={errors.email} />
                                        </div>

                                        <div className="space-y-2">
                                            <Label htmlFor="phone">Phone Number *</Label>
                                            <div className="relative">
                                                <Phone className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                                <Input
                                                    id="phone"
                                                    value={data.phone}
                                                    onChange={(e) => setData('phone', e.target.value)}
                                                    className="pl-9 bg-background/50"
                                                    placeholder="Enter phone number"
                                                    required
                                                />
                                            </div>
                                            <InputError message={errors.phone} />
                                        </div>
                                    </div>

                                    {/* Role & Society */}
                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="role_id">Role *</Label>
                                            <Select
                                                value={data.role_id}
                                                onValueChange={(value) => setData('role_id', value)}
                                            >
                                                <SelectTrigger id="role_id" className="bg-background/50">
                                                    <SelectValue placeholder="Select Role" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {roles.map((role) => (
                                                        <SelectItem key={role.id} value={role.id.toString()}>
                                                            {role.display_name || role.name}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors.role_id} />
                                        </div>

                                        <div className="space-y-2">
                                            <Label htmlFor="society_id">Society</Label>
                                            <Select
                                                value={data.society_id}
                                                onValueChange={(value) => setData('society_id', value)}
                                            >
                                                <SelectTrigger id="society_id" className="bg-background/50">
                                                    <SelectValue placeholder="Select Society (Optional)" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {societies.map((society) => (
                                                        <SelectItem key={society.id} value={society.id.toString()}>
                                                            {society.name}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                            <p className="text-[10px] text-muted-foreground">Optional: Select only if user belongs to a specific society.</p>
                                            <InputError message={errors.society_id} />
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>

                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-orange-500/10 text-orange-600">
                                        <Lock className="h-4 w-4" />
                                    </div>
                                    <h3 className="font-semibold text-lg">Security</h3>
                                </div>
                                <div className="p-6 space-y-6">
                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="password">Password *</Label>
                                            <div className="relative">
                                                <Lock className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                                <Input
                                                    id="password"
                                                    type="password"
                                                    value={data.password}
                                                    onChange={(e) => setData('password', e.target.value)}
                                                    className="pl-9 bg-background/50"
                                                    placeholder="Enter password"
                                                    required
                                                />
                                            </div>
                                            <InputError message={errors.password} />
                                        </div>

                                        <div className="space-y-2">
                                            <Label htmlFor="password_confirmation">Confirm Password *</Label>
                                            <div className="relative">
                                                <Lock className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                                <Input
                                                    id="password_confirmation"
                                                    type="password"
                                                    value={data.password_confirmation}
                                                    onChange={(e) => setData('password_confirmation', e.target.value)}
                                                    className="pl-9 bg-background/50"
                                                    placeholder="Confirm password"
                                                    required
                                                />
                                            </div>
                                            <InputError message={errors.password_confirmation} />
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>
                        </div>

                        {/* Sidebar / Settings */}
                        <div className="space-y-6">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="px-6 py-4 border-b border-border/50 bg-muted/30 flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-indigo-500/10 text-indigo-600">
                                        <Shield className="h-4 w-4" />
                                    </div>
                                    <h3 className="font-semibold text-lg">Account Settings</h3>
                                </div>
                                <div className="p-6 space-y-6">
                                    {/* Status */}
                                    <div className="space-y-2">
                                        <Label htmlFor="status">Account Status</Label>
                                        <Select
                                            value={data.status}
                                            onValueChange={(value) => setData('status', value)}
                                        >
                                            <SelectTrigger id="status" className="bg-background/50">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="active">
                                                    <div className="flex items-center gap-2">
                                                        <CheckCircle className="h-3.5 w-3.5 text-emerald-500" />
                                                        Active
                                                    </div>
                                                </SelectItem>
                                                <SelectItem value="inactive">
                                                    <div className="flex items-center gap-2">
                                                        <XCircle className="h-3.5 w-3.5 text-slate-500" />
                                                        Inactive
                                                    </div>
                                                </SelectItem>
                                                <SelectItem value="suspended">
                                                    <div className="flex items-center gap-2">
                                                        <XCircle className="h-3.5 w-3.5 text-rose-500" />
                                                        Suspended
                                                    </div>
                                                </SelectItem>
                                            </SelectContent>
                                        </Select>
                                        <InputError message={errors.status} />
                                    </div>

                                    {/* Directory Visibility */}
                                    <div className="flex items-center justify-between rounded-xl border border-border/50 bg-background/30 p-4">
                                        <div className="space-y-0.5">
                                            <Label className="text-base">Directory Visibility</Label>
                                            <p className="text-xs text-muted-foreground">
                                                Show in public directory
                                            </p>
                                        </div>
                                        <Switch
                                            checked={data.directory_visible}
                                            onCheckedChange={(checked) => setData('directory_visible', checked)}
                                        />
                                    </div>

                                    <div className="pt-4 border-t border-border/50">
                                        <Button 
                                            type="submit" 
                                            disabled={processing}
                                            className="w-full gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25"
                                        >
                                            {processing ? (
                                                <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent" />
                                            ) : (
                                                <Save className="h-4 w-4" />
                                            )}
                                            Create User
                                        </Button>
                                    </div>
                                </div>
                            </GlassCard>

                            {/* Info Card */}
                            <GlassCard intensity="low" className="p-6 bg-blue-500/5 border-blue-500/10">
                                <div className="flex items-start gap-4">
                                    <div className="p-2 rounded-lg bg-blue-500/10 text-blue-600 dark:text-blue-400">
                                        <Shield className="h-5 w-5" />
                                    </div>
                                    <div>
                                        <h4 className="font-semibold text-blue-900 dark:text-blue-100">Role Permissions</h4>
                                        <p className="text-sm text-blue-700/80 dark:text-blue-300/80 mt-1 leading-relaxed">
                                            The selected role determines access levels. Admin roles have full access to society features, while guard roles are limited to visitor management.
                                        </p>
                                    </div>
                                </div>
                            </GlassCard>
                        </div>
                    </div>
                </form>
            </div>
        </SuperAdminLayout>
    );
}
