import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link } from '@inertiajs/react';
import { 
    ArrowLeft, 
    Edit, 
    CheckCircle2, 
    XCircle, 
    Package, 
    Users, 
    Building2, 
    HardDrive, 
    CreditCard, 
    Shield, 
    Zap, 
    Calendar,
    Crown,
    HelpCircle,
    Mail,
    MessageCircle,
    Smartphone,
    FileJson,
    LayoutGrid,
    DollarSign
} from 'lucide-react';
import { Separator } from '@/components/ui/separator';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';

interface SubscriptionPlan {
    id: number;
    name: string;
    display_name: string;
    description: string | null;
    validity_days: number | null;
    monthly_price: number;
    quarterly_price: number | null;
    annual_price: number | null;
    max_users: number;
    max_units: number;
    max_buildings: number;
    storage_gb: number;
    sms_enabled: boolean;
    whatsapp_enabled: boolean;
    email_enabled: boolean;
    fcm_enabled: boolean;
    advanced_reports: boolean;
    api_access: boolean;
    custom_branding: boolean;
    priority_support: boolean;
    white_label: boolean;
    is_active: boolean;
    sort_order: number;
    subscriptions_count?: number;
    created_at: string;
    updated_at: string;
}

interface ShowProps {
    plan: SubscriptionPlan;
}

export default function Show({ plan }: ShowProps) {
    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Subscription Plans', href: '/super-admin/subscription-plans' },
        { title: 'View', href: `/super-admin/subscription-plans/${plan.id}` },
    ];

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    const features = [
        { key: 'email_enabled', label: 'Email Notifications', icon: Mail },
        { key: 'sms_enabled', label: 'SMS Notifications', icon: MessageCircle },
        { key: 'whatsapp_enabled', label: 'WhatsApp Integration', icon: MessageCircle },
        { key: 'fcm_enabled', label: 'Mobile App Access', icon: Smartphone },
        { key: 'advanced_reports', label: 'Advanced Reports', icon: FileJson },
        { key: 'api_access', label: 'API Access', icon: FileJson },
        { key: 'custom_branding', label: 'Custom Branding', icon: Crown },
        { key: 'priority_support', label: 'Priority Support', icon: HelpCircle },
        { key: 'white_label', label: 'White Label', icon: Shield },
    ];

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title={`${plan.display_name} - Subscription Plan`} />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-7xl mx-auto w-full">
                <PageHeader 
                    title={plan.display_name}
                    description={`Detailed view of the ${plan.name} subscription plan`}
                    actions={
                        <div className="flex items-center gap-2">
                            <Link href="/super-admin/subscription-plans">
                                <Button variant="outline" size="sm" className="gap-2">
                                    <ArrowLeft className="h-4 w-4" />
                                    Back
                                </Button>
                            </Link>
                            <Link href={`/super-admin/subscription-plans/${plan.id}/edit`}>
                                <Button size="sm" className="gap-2 bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 shadow-lg shadow-emerald-500/25">
                                    <Edit className="h-4 w-4" />
                                    Edit Plan
                                </Button>
                            </Link>
                        </div>
                    }
                />

                <div className="grid gap-6 lg:grid-cols-3">
                    {/* Left Column: Summary & Pricing */}
                    <div className="space-y-6">
                        <GlassCard className="p-0 overflow-hidden h-full">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-teal-500/10 text-teal-600 dark:text-teal-400">
                                        <DollarSign className="h-4 w-4" />
                                    </div>
                                    Pricing & Status
                                </h3>
                            </div>
                            <div className="p-6 space-y-6">
                                <div className="flex items-center justify-between">
                                    <span className="text-sm text-muted-foreground">Status</span>
                                    <Badge variant={plan.is_active ? 'default' : 'secondary'} className={plan.is_active ? 'bg-emerald-500/15 text-emerald-700 dark:text-emerald-400 hover:bg-emerald-500/25 border-emerald-500/20' : ''}>
                                        {plan.is_active ? 'Active' : 'Inactive'}
                                    </Badge>
                                </div>

                                <div className="flex items-center justify-between">
                                    <span className="text-sm text-muted-foreground">Validity</span>
                                    <span className="font-medium flex items-center gap-2">
                                        <Calendar className="h-4 w-4 text-muted-foreground" />
                                        {plan.validity_days ? `${plan.validity_days} Days` : 'Unlimited'}
                                    </span>
                                </div>
                                
                                <div className="space-y-2 p-4 rounded-xl bg-muted/30 border border-border/50">
                                    <span className="text-sm text-muted-foreground">Monthly Price</span>
                                    <div className="flex items-baseline gap-1">
                                        <span className="text-3xl font-bold text-emerald-600 dark:text-emerald-400">
                                            {formatCurrency(plan.monthly_price)}
                                        </span>
                                        <span className="text-muted-foreground">/mo</span>
                                    </div>
                                </div>

                                <Separator />

                                <div className="space-y-4">
                                    <div className="flex justify-between items-center">
                                        <span className="text-sm text-muted-foreground">Quarterly</span>
                                        <div className="text-right">
                                            <span className="font-semibold block">{plan.quarterly_price ? formatCurrency(plan.quarterly_price) : '-'}</span>
                                            {plan.quarterly_price && plan.monthly_price > 0 && (
                                                <span className="text-xs text-emerald-600 dark:text-emerald-400 font-medium">
                                                    Save {formatCurrency(plan.monthly_price * 3 - plan.quarterly_price)}
                                                </span>
                                            )}
                                        </div>
                                    </div>
                                    <div className="flex justify-between items-center">
                                        <span className="text-sm text-muted-foreground">Annual</span>
                                        <div className="text-right">
                                            <span className="font-semibold block">{plan.annual_price ? formatCurrency(plan.annual_price) : '-'}</span>
                                            {plan.annual_price && plan.monthly_price > 0 && (
                                                <span className="text-xs text-emerald-600 dark:text-emerald-400 font-medium">
                                                    Save {formatCurrency(plan.monthly_price * 12 - plan.annual_price)}
                                                </span>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {plan.description && (
                                    <div className="pt-4 border-t border-border/50">
                                        <p className="text-sm text-muted-foreground italic">
                                            "{plan.description}"
                                        </p>
                                    </div>
                                )}
                            </div>
                        </GlassCard>
                    </div>

                    {/* Middle Column: Limits */}
                    <div className="space-y-6">
                        <GlassCard className="p-0 overflow-hidden h-full">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-cyan-500/10 text-cyan-600 dark:text-cyan-400">
                                        <Shield className="h-4 w-4" />
                                    </div>
                                    Resource Limits
                                </h3>
                            </div>
                            <div className="p-6 space-y-6">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="p-4 rounded-xl bg-background/50 border border-border/50 flex flex-col items-center justify-center text-center gap-2">
                                        <div className="p-2 rounded-full bg-blue-500/10 text-blue-600 dark:text-blue-400">
                                            <Users className="h-5 w-5" />
                                        </div>
                                        <div>
                                            <span className="text-xs font-medium uppercase tracking-wider text-muted-foreground block mb-1">Users</span>
                                            <p className="text-xl font-bold">
                                                {plan.max_users === 9999 ? '∞' : plan.max_users}
                                            </p>
                                        </div>
                                    </div>
                                    <div className="p-4 rounded-xl bg-background/50 border border-border/50 flex flex-col items-center justify-center text-center gap-2">
                                        <div className="p-2 rounded-full bg-purple-500/10 text-purple-600 dark:text-purple-400">
                                            <Package className="h-5 w-5" />
                                        </div>
                                        <div>
                                            <span className="text-xs font-medium uppercase tracking-wider text-muted-foreground block mb-1">Units</span>
                                            <p className="text-xl font-bold">
                                                {plan.max_units === 9999 ? '∞' : plan.max_units}
                                            </p>
                                        </div>
                                    </div>
                                    <div className="p-4 rounded-xl bg-background/50 border border-border/50 flex flex-col items-center justify-center text-center gap-2">
                                        <div className="p-2 rounded-full bg-orange-500/10 text-orange-600 dark:text-orange-400">
                                            <Building2 className="h-5 w-5" />
                                        </div>
                                        <div>
                                            <span className="text-xs font-medium uppercase tracking-wider text-muted-foreground block mb-1">Buildings</span>
                                            <p className="text-xl font-bold">
                                                {plan.max_buildings === 9999 ? '∞' : plan.max_buildings}
                                            </p>
                                        </div>
                                    </div>
                                    <div className="p-4 rounded-xl bg-background/50 border border-border/50 flex flex-col items-center justify-center text-center gap-2">
                                        <div className="p-2 rounded-full bg-indigo-500/10 text-indigo-600 dark:text-indigo-400">
                                            <HardDrive className="h-5 w-5" />
                                        </div>
                                        <div>
                                            <span className="text-xs font-medium uppercase tracking-wider text-muted-foreground block mb-1">Storage</span>
                                            <p className="text-xl font-bold">
                                                {plan.storage_gb} <span className="text-sm font-normal text-muted-foreground">GB</span>
                                            </p>
                                        </div>
                                    </div>
                                </div>

                                {plan.subscriptions_count !== undefined && (
                                    <div className="mt-6 p-4 rounded-xl bg-gradient-to-br from-emerald-500/5 to-teal-500/5 border border-emerald-500/10">
                                        <div className="flex items-center justify-between">
                                            <span className="text-sm font-medium text-emerald-700 dark:text-emerald-300">Active Subscriptions</span>
                                            <Badge variant="outline" className="bg-emerald-500/10 text-emerald-700 dark:text-emerald-300 border-emerald-500/20">
                                                {plan.subscriptions_count}
                                            </Badge>
                                        </div>
                                    </div>
                                )}
                            </div>
                        </GlassCard>
                    </div>

                    {/* Right Column: Features */}
                    <div className="space-y-6">
                        <GlassCard className="p-0 overflow-hidden h-full">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-indigo-500/10 text-indigo-600 dark:text-indigo-400">
                                        <Zap className="h-4 w-4" />
                                    </div>
                                    Included Features
                                </h3>
                            </div>
                            <div className="p-6">
                                <div className="space-y-1">
                                    {features.map((feature) => (
                                        <div key={feature.key} className="flex items-center justify-between p-3 rounded-lg hover:bg-muted/50 transition-colors">
                                            <div className="flex items-center gap-3">
                                                <feature.icon className={`h-4 w-4 ${plan[feature.key as keyof SubscriptionPlan] ? 'text-foreground' : 'text-muted-foreground'}`} />
                                                <span className={`text-sm ${plan[feature.key as keyof SubscriptionPlan] ? 'text-foreground font-medium' : 'text-muted-foreground line-through decoration-muted-foreground/50'}`}>
                                                    {feature.label}
                                                </span>
                                            </div>
                                            {plan[feature.key as keyof SubscriptionPlan] ? (
                                                <CheckCircle2 className="h-5 w-5 text-emerald-500" />
                                            ) : (
                                                <XCircle className="h-5 w-5 text-muted-foreground/20" />
                                            )}
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </GlassCard>
                    </div>
                </div>
            </div>
        </SuperAdminLayout>
    );
}
