import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { 
    CreditCard, 
    Plus, 
    Search, 
    Edit, 
    Trash2, 
    Eye, 
    Package, 
    Users, 
    Building2, 
    HardDrive,
    CheckCircle,
    XCircle,
    Smartphone,
    Mail,
    MessageCircle,
    FileJson
} from 'lucide-react';
import { useState } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { 
    ModernTable, 
    ModernTableHeader, 
    ModernTableBody, 
    ModernTableRow, 
    ModernTableHead, 
    ModernTableCell 
} from '@/components/ui/modern-table';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Subscription Plans',
        href: '/super-admin/subscription-plans',
    },
];

interface SubscriptionPlan {
    id: number;
    name: string;
    display_name: string;
    description: string | null;
    monthly_price: number;
    quarterly_price: number | null;
    annual_price: number | null;
    max_users: number;
    max_units: number;
    max_buildings: number;
    storage_gb: number;
    is_active: boolean;
    sort_order: number;
    subscriptions_count?: number;
    created_at: string;
    email_enabled: boolean;
    sms_enabled: boolean;
    whatsapp_enabled: boolean;
    fcm_enabled: boolean;
    advanced_reports: boolean;
    api_access: boolean;
}

interface IndexProps {
    plans: {
        data: SubscriptionPlan[];
        links: any;
        meta: any;
    };
    filters: {
        search?: string;
        is_active?: string;
    };
}

export default function Index({ plans, filters }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [isActive, setIsActive] = useState(filters.is_active || '');

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) {
            filterParams.search = search;
        }
        if (isActive && isActive !== 'all') {
            filterParams.is_active = isActive === 'true';
        }
        router.get('/super-admin/subscription-plans', filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this subscription plan?')) {
            router.delete(`/super-admin/subscription-plans/${id}`, {
                preserveScroll: true,
            });
        }
    };

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title="Subscription Plans" />

            <div className="flex h-full flex-1 flex-col gap-6 p-8 overflow-x-hidden">
                <PageHeader 
                    title="Subscription Plans"
                    description="Manage pricing tiers, limits, and feature access"
                    actions={
                        <Link href="/super-admin/subscription-plans/create">
                            <Button size="sm" className="gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25">
                                <Plus className="h-4 w-4" />
                                Add Plan
                            </Button>
                        </Link>
                    }
                />

                <GlassCard className="flex flex-col gap-6 p-6 py-6">
                    {/* Filters */}
                    <div className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1">
                            <Label htmlFor="search" className="sr-only">Search</Label>
                            <div className="relative">
                                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                <Input
                                    id="search"
                                    placeholder="Search plans..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="pl-9 bg-background/50 border-input/50"
                                />
                            </div>
                        </div>
                        <div className="w-full md:w-48">
                            <Label htmlFor="is_active" className="sr-only">Status</Label>
                            <Select
                                value={isActive || 'all'}
                                onValueChange={(value) => setIsActive(value === 'all' ? '' : value)}
                            >
                                <SelectTrigger id="is_active" className="bg-background/50 border-input/50">
                                    <SelectValue placeholder="All Status" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Status</SelectItem>
                                    <SelectItem value="true">Active</SelectItem>
                                    <SelectItem value="false">Inactive</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        <Button 
                            onClick={handleSearch} 
                            variant="secondary"
                            className="shadow-sm"
                        >
                            Apply
                        </Button>
                    </div>

                    {/* Table */}
                    <ModernTable>
                        <ModernTableHeader>
                            <ModernTableRow>
                                <ModernTableHead>Plan Details</ModernTableHead>
                                <ModernTableHead>Pricing</ModernTableHead>
                                <ModernTableHead>Limits</ModernTableHead>
                                <ModernTableHead>Features</ModernTableHead>
                                <ModernTableHead>Status</ModernTableHead>
                                <ModernTableHead className="text-right">Actions</ModernTableHead>
                            </ModernTableRow>
                        </ModernTableHeader>
                        <ModernTableBody>
                            {plans.data.length > 0 ? (
                                plans.data.map((plan) => (
                                    <ModernTableRow key={plan.id}>
                                        <ModernTableCell>
                                            <div className="flex items-center gap-3">
                                                <div className="flex h-10 w-10 shrink-0 items-center justify-center rounded-lg bg-primary/10 text-primary">
                                                    <Package className="h-5 w-5" />
                                                </div>
                                                <div>
                                                    <p className="font-semibold text-foreground">{plan.display_name}</p>
                                                    <p className="text-xs text-muted-foreground font-mono">{plan.name}</p>
                                                </div>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="space-y-0.5">
                                                <div className="font-medium text-sm">₹{plan.monthly_price}<span className="text-xs text-muted-foreground">/mo</span></div>
                                                {plan.annual_price && (
                                                    <div className="text-xs text-muted-foreground">₹{plan.annual_price}/yr</div>
                                                )}
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="space-y-1 text-xs text-muted-foreground">
                                                <div className="flex items-center gap-1.5">
                                                    <Users className="h-3 w-3" />
                                                    <span>{plan.max_users === 9999 ? 'Unl.' : plan.max_users} Users</span>
                                                </div>
                                                <div className="flex items-center gap-1.5">
                                                    <Building2 className="h-3 w-3" />
                                                    <span>{plan.max_units === 9999 ? 'Unl.' : plan.max_units} Units</span>
                                                </div>
                                                <div className="flex items-center gap-1.5">
                                                    <HardDrive className="h-3 w-3" />
                                                    <span>{plan.storage_gb} GB</span>
                                                </div>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="flex flex-wrap gap-1.5 max-w-[150px]">
                                                {plan.email_enabled && (
                                                    <div className="p-1 rounded bg-blue-500/10 text-blue-600 dark:text-blue-400" title="Email">
                                                        <Mail className="h-3 w-3" />
                                                    </div>
                                                )}
                                                {plan.sms_enabled && (
                                                    <div className="p-1 rounded bg-green-500/10 text-green-600 dark:text-green-400" title="SMS">
                                                        <MessageCircle className="h-3 w-3" />
                                                    </div>
                                                )}
                                                {plan.fcm_enabled && (
                                                    <div className="p-1 rounded bg-purple-500/10 text-purple-600 dark:text-purple-400" title="App Notifications">
                                                        <Smartphone className="h-3 w-3" />
                                                    </div>
                                                )}
                                                {plan.api_access && (
                                                    <div className="p-1 rounded bg-orange-500/10 text-orange-600 dark:text-orange-400" title="API Access">
                                                        <FileJson className="h-3 w-3" />
                                                    </div>
                                                )}
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            {plan.is_active ? (
                                                <Badge variant="outline" className="bg-emerald-500/10 text-emerald-600 border-emerald-500/20">
                                                    Active
                                                </Badge>
                                            ) : (
                                                <Badge variant="outline" className="bg-slate-500/10 text-slate-600 border-slate-500/20">
                                                    Inactive
                                                </Badge>
                                            )}
                                        </ModernTableCell>
                                        <ModernTableCell className="text-right">
                                            <div className="flex items-center justify-end gap-1">
                                                <Link href={`/super-admin/subscription-plans/${plan.id}`}>
                                                    <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-primary">
                                                        <Eye className="h-4 w-4" />
                                                    </Button>
                                                </Link>
                                                <Link href={`/super-admin/subscription-plans/${plan.id}/edit`}>
                                                    <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-blue-500">
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                </Link>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-8 w-8 hover:text-red-500"
                                                    onClick={() => handleDelete(plan.id)}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                ))
                            ) : (
                                <ModernTableRow>
                                    <ModernTableCell colSpan={6} className="h-24 text-center">
                                        <div className="flex flex-col items-center justify-center text-muted-foreground">
                                            <Package className="h-8 w-8 mb-2 opacity-20" />
                                            <p>No subscription plans found</p>
                                        </div>
                                    </ModernTableCell>
                                </ModernTableRow>
                            )}
                        </ModernTableBody>
                    </ModernTable>

                    {/* Pagination */}
                    {plans.links && plans.links.length > 3 && (
                        <div className="flex items-center justify-between border-t border-border/50 pt-4">
                            <div className="text-sm text-muted-foreground">
                                Showing <span className="font-medium text-foreground">{plans.meta.from}</span> to <span className="font-medium text-foreground">{plans.meta.to}</span> of <span className="font-medium text-foreground">{plans.meta.total}</span> entries
                            </div>
                            <div className="flex gap-1">
                                {plans.links.map((link: any, index: number) => (
                                    <Link
                                        key={index}
                                        href={link.url || '#'}
                                        className={`px-3 py-1 rounded-md text-xs font-medium transition-colors ${
                                            link.active
                                                ? 'bg-primary text-primary-foreground shadow-sm'
                                                : 'text-muted-foreground hover:bg-muted hover:text-foreground'
                                        }`}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        </div>
                    )}
                </GlassCard>
            </div>
        </SuperAdminLayout>
    );
}
