import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, useForm } from '@inertiajs/react';
import { 
    ArrowLeft, 
    CreditCard, 
    DollarSign, 
    HardDrive, 
    LayoutGrid, 
    Package, 
    Save, 
    Settings, 
    Shield, 
    Users,
    Smartphone,
    Mail,
    MessageCircle,
    FileJson,
    Crown,
    HelpCircle
} from 'lucide-react';
import { FormEventHandler } from 'react';
import { Switch } from '@/components/ui/switch';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';

interface SubscriptionPlan {
    id: number;
    name: string;
    display_name: string;
    description: string | null;
    validity_days: number | null;
    monthly_price: number;
    quarterly_price: number | null;
    annual_price: number | null;
    max_users: number;
    max_units: number;
    max_buildings: number;
    storage_gb: number;
    sms_enabled: boolean;
    whatsapp_enabled: boolean;
    email_enabled: boolean;
    fcm_enabled: boolean;
    advanced_reports: boolean;
    api_access: boolean;
    custom_branding: boolean;
    priority_support: boolean;
    white_label: boolean;
    is_active: boolean;
    sort_order: number;
}

interface EditProps {
    plan: SubscriptionPlan;
    errors?: any;
}

export default function Edit({ plan, errors }: EditProps) {
    const { data, setData, put, processing, errors: formErrors } = useForm({
        name: plan.name,
        display_name: plan.display_name,
        description: plan.description || '',
        validity_days: plan.validity_days?.toString() || '',
        monthly_price: plan.monthly_price.toString(),
        quarterly_price: plan.quarterly_price?.toString() || '',
        annual_price: plan.annual_price?.toString() || '',
        max_users: plan.max_users.toString(),
        max_units: plan.max_units.toString(),
        max_buildings: plan.max_buildings.toString(),
        storage_gb: plan.storage_gb.toString(),
        sort_order: plan.sort_order.toString(),
        sms_enabled: plan.sms_enabled,
        whatsapp_enabled: plan.whatsapp_enabled,
        email_enabled: plan.email_enabled,
        fcm_enabled: plan.fcm_enabled,
        advanced_reports: plan.advanced_reports,
        api_access: plan.api_access,
        custom_branding: plan.custom_branding,
        priority_support: plan.priority_support,
        white_label: plan.white_label,
        is_active: plan.is_active,
    });

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Subscription Plans', href: '/super-admin/subscription-plans' },
        { title: 'Edit', href: `/super-admin/subscription-plans/${plan.id}/edit` },
    ];

    const handleSubmit: FormEventHandler = (e) => {
        e.preventDefault();
        put(`/super-admin/subscription-plans/${plan.id}`);
    };

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title={`Edit ${plan.display_name}`} />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-7xl mx-auto w-full">
                <PageHeader 
                    title="Edit Plan"
                    description={`Update subscription details for ${plan.display_name}`}
                    actions={
                        <Link href="/super-admin/subscription-plans">
                            <Button variant="outline" size="sm" className="gap-2">
                                <ArrowLeft className="h-4 w-4" />
                                Back to List
                            </Button>
                        </Link>
                    }
                />

                <form onSubmit={handleSubmit} className="grid gap-6 lg:grid-cols-3">
                    {/* Main Form */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Plan Details */}
                        <GlassCard className="p-0 overflow-hidden">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-emerald-500/10 text-emerald-600 dark:text-emerald-400">
                                        <LayoutGrid className="h-4 w-4" />
                                    </div>
                                    Plan Information
                                </h3>
                            </div>
                            <div className="p-6 space-y-6">
                                <div className="grid gap-6 md:grid-cols-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Internal Name *</Label>
                                        <Input
                                            id="name"
                                            value={data.name}
                                            onChange={(e) => setData('name', e.target.value)}
                                            placeholder="e.g., basic_plan"
                                            className="font-mono bg-background/50"
                                            required
                                        />
                                        <InputError message={formErrors.name} />
                                        <p className="text-xs text-muted-foreground">Lowercase, no spaces</p>
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="display_name">Display Name *</Label>
                                        <Input
                                            id="display_name"
                                            value={data.display_name}
                                            onChange={(e) => setData('display_name', e.target.value)}
                                            placeholder="e.g., Basic Plan"
                                            className="bg-background/50"
                                            required
                                        />
                                        <InputError message={formErrors.display_name} />
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="description">Description</Label>
                                    <Textarea
                                        id="description"
                                        value={data.description}
                                        onChange={(e) => setData('description', e.target.value)}
                                        className="bg-background/50 min-h-[100px]"
                                        placeholder="Describe the plan features..."
                                    />
                                    <InputError message={formErrors.description} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="validity_days">Validity (Days)</Label>
                                    <Input
                                        id="validity_days"
                                        type="number"
                                        min="1"
                                        value={data.validity_days}
                                        onChange={(e) => setData('validity_days', e.target.value)}
                                        placeholder="e.g., 30, 365"
                                        className="bg-background/50"
                                    />
                                    <InputError message={formErrors.validity_days} />
                                    <p className="text-xs text-muted-foreground">
                                        Leave empty for unlimited duration
                                    </p>
                                </div>
                            </div>
                        </GlassCard>

                        {/* Pricing */}
                        <GlassCard className="p-0 overflow-hidden">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-teal-500/10 text-teal-600 dark:text-teal-400">
                                        <DollarSign className="h-4 w-4" />
                                    </div>
                                    Pricing Configuration
                                </h3>
                            </div>
                            <div className="p-6 grid gap-6 md:grid-cols-3">
                                <div className="space-y-2">
                                    <Label htmlFor="monthly_price">Monthly Price (₹) *</Label>
                                    <div className="relative">
                                        <span className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground">₹</span>
                                        <Input
                                            id="monthly_price"
                                            type="number"
                                            step="0.01"
                                            value={data.monthly_price}
                                            onChange={(e) => setData('monthly_price', e.target.value)}
                                            className="pl-7 bg-background/50"
                                            placeholder="0.00"
                                            required
                                        />
                                    </div>
                                    <InputError message={formErrors.monthly_price} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="quarterly_price">Quarterly Price (₹)</Label>
                                    <div className="relative">
                                        <span className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground">₹</span>
                                        <Input
                                            id="quarterly_price"
                                            type="number"
                                            step="0.01"
                                            value={data.quarterly_price}
                                            onChange={(e) => setData('quarterly_price', e.target.value)}
                                            className="pl-7 bg-background/50"
                                            placeholder="Optional"
                                        />
                                    </div>
                                    <InputError message={formErrors.quarterly_price} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="annual_price">Annual Price (₹)</Label>
                                    <div className="relative">
                                        <span className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground">₹</span>
                                        <Input
                                            id="annual_price"
                                            type="number"
                                            step="0.01"
                                            value={data.annual_price}
                                            onChange={(e) => setData('annual_price', e.target.value)}
                                            className="pl-7 bg-background/50"
                                            placeholder="Optional"
                                        />
                                    </div>
                                    <InputError message={formErrors.annual_price} />
                                </div>
                            </div>
                        </GlassCard>

                        {/* Limits */}
                        <GlassCard className="p-0 overflow-hidden">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-cyan-500/10 text-cyan-600 dark:text-cyan-400">
                                        <Shield className="h-4 w-4" />
                                    </div>
                                    Resource Limits
                                </h3>
                            </div>
                            <div className="p-6 grid gap-6 md:grid-cols-2">
                                <div className="space-y-2">
                                    <Label htmlFor="max_users">Max Users *</Label>
                                    <div className="relative">
                                        <Users className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                        <Input
                                            id="max_users"
                                            type="number"
                                            value={data.max_users}
                                            onChange={(e) => setData('max_users', e.target.value)}
                                            className="pl-9 bg-background/50"
                                            required
                                        />
                                    </div>
                                    <InputError message={formErrors.max_users} />
                                    <p className="text-xs text-muted-foreground">9999 for unlimited</p>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="max_units">Max Units *</Label>
                                    <div className="relative">
                                        <Package className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                        <Input
                                            id="max_units"
                                            type="number"
                                            value={data.max_units}
                                            onChange={(e) => setData('max_units', e.target.value)}
                                            className="pl-9 bg-background/50"
                                            required
                                        />
                                    </div>
                                    <InputError message={formErrors.max_units} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="max_buildings">Max Buildings *</Label>
                                    <div className="relative">
                                        <Package className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                        <Input
                                            id="max_buildings"
                                            type="number"
                                            value={data.max_buildings}
                                            onChange={(e) => setData('max_buildings', e.target.value)}
                                            className="pl-9 bg-background/50"
                                            required
                                        />
                                    </div>
                                    <InputError message={formErrors.max_buildings} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="storage_gb">Storage (GB) *</Label>
                                    <div className="relative">
                                        <HardDrive className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                        <Input
                                            id="storage_gb"
                                            type="number"
                                            value={data.storage_gb}
                                            onChange={(e) => setData('storage_gb', e.target.value)}
                                            className="pl-9 bg-background/50"
                                            required
                                        />
                                    </div>
                                    <InputError message={formErrors.storage_gb} />
                                </div>
                            </div>
                        </GlassCard>
                    </div>

                    {/* Sidebar / Settings */}
                    <div className="space-y-6">
                        <GlassCard className="p-0 overflow-hidden sticky top-6">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="p-1.5 rounded-md bg-indigo-500/10 text-indigo-600 dark:text-indigo-400">
                                        <Settings className="h-4 w-4" />
                                    </div>
                                    Plan Settings
                                </h3>
                            </div>
                            <div className="p-6 space-y-6">
                                {/* Status */}
                                <div className="flex items-center justify-between rounded-lg border border-border/50 bg-muted/20 p-4">
                                    <div className="space-y-0.5">
                                        <Label className="text-base">Active Status</Label>
                                        <p className="text-xs text-muted-foreground">
                                            Plan visibility to users
                                        </p>
                                    </div>
                                    <Switch
                                        checked={data.is_active}
                                        onCheckedChange={(checked) => setData('is_active', checked)}
                                    />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="sort_order">Sort Order</Label>
                                    <Input
                                        id="sort_order"
                                        type="number"
                                        value={data.sort_order}
                                        onChange={(e) => setData('sort_order', e.target.value)}
                                        className="bg-background/50"
                                    />
                                    <InputError message={formErrors.sort_order} />
                                </div>

                                <div className="space-y-4 pt-4 border-t border-border/50">
                                    <h4 className="font-semibold text-sm text-foreground uppercase tracking-wider">Features</h4>
                                    
                                    <div className="space-y-3">
                                        {[
                                            { id: 'email_enabled', label: 'Email Notifications', icon: Mail },
                                            { id: 'sms_enabled', label: 'SMS Notifications', icon: MessageCircle },
                                            { id: 'whatsapp_enabled', label: 'WhatsApp Integration', icon: MessageCircle },
                                            { id: 'fcm_enabled', label: 'Mobile App Access', icon: Smartphone },
                                            { id: 'advanced_reports', label: 'Advanced Reports', icon: FileJson },
                                            { id: 'api_access', label: 'API Access', icon: FileJson },
                                            { id: 'custom_branding', label: 'Custom Branding', icon: Crown },
                                            { id: 'priority_support', label: 'Priority Support', icon: HelpCircle },
                                            { id: 'white_label', label: 'White Label', icon: Shield },
                                        ].map((feature) => (
                                            <div key={feature.id} className="flex items-center space-x-3 p-2 rounded-lg hover:bg-muted/50 transition-colors">
                                                <Checkbox
                                                    id={feature.id}
                                                    checked={data[feature.id as keyof typeof data] as boolean}
                                                    onCheckedChange={(checked) => setData(feature.id as keyof typeof data, checked as boolean)}
                                                />
                                                <div className="flex items-center gap-2 cursor-pointer flex-1" onClick={() => setData(feature.id as keyof typeof data, !data[feature.id as keyof typeof data])}>
                                                    <feature.icon className="h-4 w-4 text-muted-foreground" />
                                                    <Label htmlFor={feature.id} className="font-normal cursor-pointer flex-1 user-select-none">
                                                        {feature.label}
                                                    </Label>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>

                                <Button 
                                    type="submit" 
                                    disabled={processing}
                                    className="w-full bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 shadow-lg shadow-emerald-500/25"
                                >
                                    <Save className="mr-2 h-4 w-4" />
                                    Update Plan
                                </Button>
                            </div>
                        </GlassCard>
                    </div>
                </form>
            </div>
        </SuperAdminLayout>
    );
}
