import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link } from '@inertiajs/react';
import { ArrowLeft, Building2, Edit, User, MapPin, Calendar, Crown, Shield, Settings } from 'lucide-react';

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Societies', href: '/super-admin/societies' },
    { title: 'View', href: '#' },
];

interface Society {
    id: number;
    name: string;
    registration_no: string;
    address: string;
    city: string;
    state: string;
    pincode: string;
    status: string;
    logo?: string | null;
    gst_no?: string | null;
    pan_no?: string | null;
    phone?: string | null;
    email?: string | null;
    code?: string | null;
    rules?: string | null;
    admin?: {
        id: number;
        name: string;
        email: string | null;
        phone: string;
        role: {
            id: number;
            name: string;
            display_name: string | null;
        };
    } | null;
    subscription?: {
        id: number;
        plan_display_name: string;
        status: string;
        billing_cycle: string;
        start_date: string;
        end_date?: string | null;
    } | null;
    settings?: {
        timezone: string;
        currency: string;
        date_format: string;
        language: string;
    } | null;
    created_at: string;
    updated_at: string;
}

interface ShowProps {
    society: Society;
}

export default function Show({ society }: ShowProps) {
    const getStatusColor = (status: string) => {
        switch (status.toLowerCase()) {
            case 'active':
                return 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20';
            case 'pending':
                return 'bg-amber-500/10 text-amber-600 dark:text-amber-400 border-amber-500/20';
            case 'suspended':
                return 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20';
            default:
                return 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20';
        }
    };

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title={`${society.name} - Society Details`} />
            
            {/* Animated Background Blobs */}
            <div className="fixed inset-0 -z-10 overflow-hidden pointer-events-none">
                <div className="absolute top-0 -left-4 w-96 h-96 bg-cyan-300 dark:bg-cyan-900 rounded-full mix-blend-multiply dark:mix-blend-soft-light filter blur-3xl opacity-20 animate-blob" />
                <div className="absolute top-0 -right-4 w-96 h-96 bg-teal-300 dark:bg-teal-900 rounded-full mix-blend-multiply dark:mix-blend-soft-light filter blur-3xl opacity-20 animate-blob animation-delay-2000" />
                <div className="absolute -bottom-8 left-20 w-96 h-96 bg-cyan-300 dark:bg-cyan-900 rounded-full mix-blend-multiply dark:mix-blend-soft-light filter blur-3xl opacity-20 animate-blob animation-delay-4000" />
            </div>

            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto p-8">
                {/* Header Section with Glass Effect */}
                <div className="relative">
                    <div className="absolute inset-0 bg-gradient-to-r from-sky-500/5 via-sky-500/5 to-teal-500/5 rounded-2xl blur-2xl" />
                    <div className="relative flex items-center justify-between p-6 rounded-2xl border border-border/50 bg-card/50 backdrop-blur-sm">
                        <div className="flex items-center gap-4">
                            <Link href="/super-admin/societies">
                                <Button variant="ghost" size="sm" className="gap-2">
                                    <ArrowLeft className="h-4 w-4" />
                                    Back
                                </Button>
                            </Link>
                            <div className="h-8 w-px bg-border" />
                            <div>
                                <div className="flex items-center gap-3 mb-1">
                                    <div className="p-2 rounded-xl bg-gradient-to-br from-sky-500 to-teal-500 shadow-lg shadow-cyan-500/50">
                                        <Building2 className="h-5 w-5 text-white" />
                                    </div>
                                    <h1 className="text-3xl font-bold bg-gradient-to-r from-sky-600 via-sky-600 to-teal-600 dark:from-sky-400 dark:via-sky-400 dark:to-teal-400 bg-clip-text text-transparent">
                                        {society.name}
                                    </h1>
                                    <Badge variant="outline" className={`${getStatusColor(society.status)} border capitalize`}>
                                        {society.status}
                                    </Badge>
                                </div>
                                <p className="text-muted-foreground ml-12">
                                    Registration: {society.registration_no}
                                </p>
                            </div>
                        </div>
                        <Link href={`/super-admin/societies/${society.id}/edit`}>
                            <Button className="gap-2 bg-gradient-to-r from-sky-600 to-teal-600 hover:from-sky-700 hover:to-teal-700 shadow-lg shadow-cyan-500/50">
                                <Edit className="h-4 w-4" />
                                Edit Society
                            </Button>
                        </Link>
                    </div>
                </div>

                {/* Main Content Grid */}
                <div className="grid gap-6 lg:grid-cols-3">
                    {/* Left Column - Basic, Contact & Tax Info */}
                    <div className="lg:col-span-1 space-y-6">
                        {/* Basic Information */}
                        <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                            <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-sky-500 via-sky-500 to-teal-500" />
                            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-sky-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                            <CardHeader className="relative pb-3">
                                <CardTitle className="text-lg flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-gradient-to-br from-sky-500 to-teal-500">
                                        <Building2 className="h-4 w-4 text-white" />
                                    </div>
                                    Basic Information
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="relative space-y-3">
                                <div>
                                    <p className="text-xs text-muted-foreground mb-1">Society Name</p>
                                    <p className="font-semibold">{society.name}</p>
                                </div>
                                <div className="pt-2 border-t border-border/50">
                                    <p className="text-xs text-muted-foreground mb-1">Registration Number</p>
                                    <p className="font-medium text-sm">{society.registration_no}</p>
                                </div>
                                {society.code && (
                                    <div className="pt-2 border-t border-border/50">
                                        <p className="text-xs text-muted-foreground mb-1">Society Code</p>
                                        <p className="font-medium text-sm font-mono">{society.code}</p>
                                    </div>
                                )}
                                <div className="pt-2 border-t border-border/50">
                                    <p className="text-xs text-muted-foreground mb-1">Status</p>
                                    <Badge variant="outline" className={`${getStatusColor(society.status)} border capitalize`}>
                                        {society.status}
                                    </Badge>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Contact Information */}
                        <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                            <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-emerald-500 via-green-500 to-teal-500" />
                            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-emerald-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                            <CardHeader className="relative pb-3">
                                <CardTitle className="text-lg flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-gradient-to-br from-emerald-500 to-teal-500">
                                        <MapPin className="h-4 w-4 text-white" />
                                    </div>
                                    Contact Information
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="relative space-y-3">
                                <div>
                                    <p className="text-xs text-muted-foreground mb-1">Address</p>
                                    <p className="font-medium text-sm leading-relaxed">
                                        {society.address}<br/>
                                        {society.city}, {society.state} - {society.pincode}
                                    </p>
                                </div>
                                {society.phone && (
                                    <div className="pt-2 border-t border-border/50">
                                        <p className="text-xs text-muted-foreground mb-1">Phone</p>
                                        <p className="font-medium text-sm">{society.phone}</p>
                                    </div>
                                )}
                                {society.email && (
                                    <div className="pt-2 border-t border-border/50">
                                        <p className="text-xs text-muted-foreground mb-1">Email</p>
                                        <p className="font-medium text-sm break-all">{society.email}</p>
                                    </div>
                                )}
                            </CardContent>
                        </Card>

                        {/* Tax Information */}
                        <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                            <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-slate-500 via-gray-500 to-zinc-500" />
                            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-slate-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                            <CardHeader className="relative pb-3">
                                <CardTitle className="text-lg flex items-center gap-2">
                                    <div className="p-1.5 rounded-lg bg-gradient-to-br from-slate-500 to-zinc-500">
                                        <Shield className="h-4 w-4 text-white" />
                                    </div>
                                    Tax Information
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="relative space-y-3">
                                {society.gst_no && (
                                    <div>
                                        <p className="text-xs text-muted-foreground mb-1">GST Number</p>
                                        <p className="font-medium text-sm font-mono">{society.gst_no}</p>
                                    </div>
                                )}
                                {society.pan_no && (
                                    <div className={society.gst_no ? "pt-2 border-t border-border/50" : ""}>
                                        <p className="text-xs text-muted-foreground mb-1">PAN Number</p>
                                        <p className="font-medium text-sm font-mono">{society.pan_no}</p>
                                    </div>
                                )}
                                {!society.gst_no && !society.pan_no && (
                                    <p className="text-sm text-muted-foreground italic">No tax information available</p>
                                )}
                            </CardContent>
                        </Card>
                    </div>

                    {/* Right Column - Admin, Subscription, Settings, Rules */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Society Admin */}
                        {society.admin && (
                            <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-cyan-500 via-blue-500 to-indigo-500" />
                                <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-cyan-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                                <CardHeader className="relative pb-3">
                                    <CardTitle className="text-lg flex items-center gap-2">
                                        <div className="p-1.5 rounded-lg bg-gradient-to-br from-cyan-500 to-indigo-500">
                                            <User className="h-4 w-4 text-white" />
                                        </div>
                                        Society Admin
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="relative">
                                    <div className="grid gap-4 sm:grid-cols-3">
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Name</p>
                                            <p className="font-semibold">{society.admin.name}</p>
                                        </div>
                                        {society.admin.email && (
                                            <div>
                                                <p className="text-xs text-muted-foreground mb-1">Email</p>
                                                <p className="font-medium text-sm break-all">{society.admin.email}</p>
                                            </div>
                                        )}
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Phone</p>
                                            <p className="font-medium text-sm">{society.admin.phone}</p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                        )}

                        {/* Subscription */}
                        {society.subscription && (
                            <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-cyan-500 via-pink-500 to-rose-500" />
                                <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-cyan-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                                <CardHeader className="relative pb-3">
                                    <CardTitle className="text-lg flex items-center gap-2">
                                        <div className="p-1.5 rounded-lg bg-gradient-to-br from-cyan-500 to-rose-500">
                                            <Crown className="h-4 w-4 text-white" />
                                        </div>
                                        Subscription Details
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="relative">
                                    <div className="grid gap-4 sm:grid-cols-4">
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Plan</p>
                                            <p className="font-semibold">{society.subscription.plan_display_name}</p>
                                        </div>
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Status</p>
                                            <Badge variant="outline" className={`${getStatusColor(society.subscription.status)} border capitalize`}>
                                                {society.subscription.status}
                                            </Badge>
                                        </div>
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Billing Cycle</p>
                                            <p className="font-medium text-sm capitalize">{society.subscription.billing_cycle}</p>
                                        </div>
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Start Date</p>
                                            <p className="font-medium text-sm">
                                                {new Date(society.subscription.start_date).toLocaleDateString()}
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                        )}

                        {/* System Settings */}
                        {society.settings && (
                            <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-indigo-500 via-sky-500 to-pink-500" />
                                <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-indigo-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                                <CardHeader className="relative pb-3">
                                    <CardTitle className="text-lg flex items-center gap-2">
                                        <div className="p-1.5 rounded-lg bg-gradient-to-br from-indigo-500 to-pink-500">
                                            <Settings className="h-4 w-4 text-white" />
                                        </div>
                                        System Settings
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="relative">
                                    <div className="grid gap-4 sm:grid-cols-4">
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Timezone</p>
                                            <p className="font-medium text-sm">{society.settings.timezone}</p>
                                        </div>
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Currency</p>
                                            <p className="font-medium text-sm">{society.settings.currency}</p>
                                        </div>
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Date Format</p>
                                            <p className="font-medium text-sm">{society.settings.date_format}</p>
                                        </div>
                                        <div>
                                            <p className="text-xs text-muted-foreground mb-1">Language</p>
                                            <p className="font-medium text-sm capitalize">{society.settings.language}</p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                        )}

                        {/* Rules */}
                        {society.rules && (
                            <Card className="border-0 shadow-xl bg-gradient-to-br from-card to-card/80 backdrop-blur-sm overflow-hidden group hover:shadow-2xl transition-all duration-300">
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-rose-500 via-red-500 to-orange-500" />
                                <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-rose-500/10 to-transparent rounded-full -mr-16 -mt-16 group-hover:scale-150 transition-transform duration-700" />
                                <CardHeader className="relative pb-3">
                                    <CardTitle className="text-lg flex items-center gap-2">
                                        <div className="p-1.5 rounded-lg bg-gradient-to-br from-rose-500 to-orange-500">
                                            <Calendar className="h-4 w-4 text-white" />
                                        </div>
                                        Society Rules & Regulations
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="relative">
                                    <p className="whitespace-pre-wrap text-sm leading-relaxed">{society.rules}</p>
                                </CardContent>
                            </Card>
                        )}
                    </div>
                </div>
            </div>

            <style>{`
                @keyframes blob {
                    0%, 100% {
                        transform: translate(0px, 0px) scale(1);
                    }
                    33% {
                        transform: translate(30px, -50px) scale(1.1);
                    }
                    66% {
                        transform: translate(-20px, 20px) scale(0.9);
                    }
                }
                .animate-blob {
                    animation: blob 7s infinite;
                }
                .animation-delay-2000 {
                    animation-delay: 2s;
                }
                .animation-delay-4000 {
                    animation-delay: 4s;
                }
            `}</style>
        </SuperAdminLayout>
    );
}
