import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { 
    Building2, 
    Plus, 
    Search, 
    Edit, 
    Trash2, 
    Eye,
    Users,
    CheckCircle,
    Clock,
    Filter,
    ArrowUpRight
} from 'lucide-react';
import { useState } from 'react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { 
    ModernTable, 
    ModernTableHeader, 
    ModernTableBody, 
    ModernTableRow, 
    ModernTableHead, 
    ModernTableCell 
} from '@/components/ui/modern-table';
import { CardContent, CardHeader, CardTitle } from '@/components/ui/card';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Societies',
        href: '/super-admin/societies',
    },
];

interface Society {
    id: number;
    name: string;
    registration_no: string;
    address: string;
    city: string;
    state: string;
    pincode: string;
    status: string;
    code?: string;
    users_count: number;
    platformUser?: {
        id: number;
        name: string;
        email: string | null;
        phone: string;
        role: {
            id: number;
            name: string;
            display_name: string | null;
        };
    } | null;
    primaryAdmin?: {
        id: number;
        name: string;
        email: string | null;
        phone: string;
        role: {
            id: number;
            name: string;
            display_name: string | null;
        };
    } | null;
    admin?: {
        id: number;
        name: string;
        email: string | null;
        phone: string;
        role: {
            id: number;
            name: string;
            display_name: string | null;
        };
    } | null;
    platform_fee?: {
        amount: number;
        status: string;
        frequency: string;
    } | null;
}

interface Stats {
    total_societies: number;
    active_societies: number;
    pending_societies: number;
    suspended_societies: number;
    total_users: number;
    pending_fees: number;
}

interface IndexProps {
    societies: {
        data: Society[];
        links: any;
        meta: any;
    };
    filters: {
        search?: string;
        status?: string;
    };
    stats: Stats;
}

export default function Index({ societies, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || '');

    const handleSearch = () => {
        const filters: any = { search };
        if (status && status !== 'all') {
            filters.status = status;
        }
        router.get('/super-admin/societies', filters, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this society?')) {
            router.delete(`/super-admin/societies/${id}`, {
                preserveScroll: true,
            });
        }
    };

    const handleStatusChange = (societyId: number, newStatus: string) => {
        router.put(`/super-admin/societies/${societyId}`, 
            { status: newStatus },
            {
                preserveScroll: true,
                preserveState: true,
                only: ['societies'],
            }
        );
    };

    const getStatusColor = (status: string) => {
        switch (status.toLowerCase()) {
            case 'active':
                return 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20';
            case 'pending':
                return 'bg-amber-500/10 text-amber-600 dark:text-amber-400 border-amber-500/20';
            case 'suspended':
                return 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20';
            default:
                return 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20';
        }
    };

    const statsCards = [
        {
            title: 'Total Societies',
            value: stats.total_societies,
            subtitle: `${stats.active_societies} active`,
            icon: Building2,
            iconClass: 'text-primary',
            iconBg: 'bg-primary/10',
            progress: stats.total_societies > 0 ? (stats.active_societies / stats.total_societies) * 100 : 0,
        },
        {
            title: 'Active Societies',
            value: stats.active_societies,
            subtitle: 'Currently operational',
            icon: CheckCircle,
            iconClass: 'text-emerald-500',
            iconBg: 'bg-emerald-500/10',
            progress: stats.total_societies > 0 ? (stats.active_societies / stats.total_societies) * 100 : 0,
        },
        {
            title: 'Pending Approval',
            value: stats.pending_societies,
            subtitle: 'Awaiting verification',
            icon: Clock,
            iconClass: 'text-amber-500',
            iconBg: 'bg-amber-500/10',
            progress: stats.total_societies > 0 ? (stats.pending_societies / stats.total_societies) * 100 : 0,
        },
        {
            title: 'Total Members',
            value: stats.total_users,
            subtitle: 'Across all societies',
            icon: Users,
            iconClass: 'text-blue-500',
            iconBg: 'bg-blue-500/10',
            progress: 75,
        },
    ];

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title="Societies" />
            
            <div className="flex h-full flex-1 flex-col gap-6 p-8 overflow-x-hidden">
                <PageHeader 
                    title="Societies Management"
                    description="Manage and monitor all registered societies"
                    actions={
                        <Link href="/super-admin/societies/create">
                            <Button size="sm" className="gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25">
                                <Plus className="h-4 w-4" />
                                Add New Society
                            </Button>
                        </Link>
                    }
                />

                {/* Stats Cards Grid */}
                <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
                    {statsCards.map((stat, index) => (
                        <GlassCard 
                            key={index}
                            className="group relative overflow-hidden transition-all duration-300 hover:-translate-y-1 py-6"
                            intensity="low"
                        >
                            <CardHeader className="relative pb-2">
                                <div className="flex items-start justify-between">
                                    <div className="space-y-1">
                                        <CardTitle className="text-sm font-medium text-muted-foreground">
                                            {stat.title}
                                        </CardTitle>
                                        <div className="text-3xl font-bold tracking-tight">
                                            {stat.value}
                                        </div>
                                    </div>
                                    <div className={`p-2.5 rounded-xl ${stat.iconBg} transition-colors group-hover:scale-110 duration-300`}>
                                        <stat.icon className={`h-5 w-5 ${stat.iconClass}`} />
                                    </div>
                                </div>
                            </CardHeader>
                            <CardContent>
                                <p className="text-xs text-muted-foreground mb-3">
                                    {stat.subtitle}
                                </p>
                                <div className="h-1.5 w-full bg-slate-100 dark:bg-slate-800 rounded-full overflow-hidden">
                                    <div 
                                        className={`h-full rounded-full transition-all duration-1000 ${stat.iconClass.replace('text-', 'bg-')}`}
                                        style={{ width: `${stat.progress}%` }}
                                    />
                                </div>
                            </CardContent>
                        </GlassCard>
                    ))}
                </div>

                {/* Filters & Table */}
                <GlassCard className="flex flex-col gap-6 p-6 py-6">
                    {/* Filters */}
                    <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
                        <div className="flex items-center gap-2 text-primary font-medium">
                            <Filter className="h-4 w-4" />
                            Filters
                        </div>
                        <div className="flex flex-1 w-full md:w-auto gap-3">
                            <div className="relative flex-1 md:max-w-xs">
                                <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
                                <Input
                                    type="text"
                                    placeholder="Search societies..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="pl-9 bg-background/50 border-input/50"
                                />
                            </div>
                            <Select value={status || 'all'} onValueChange={(value) => setStatus(value === 'all' ? '' : value)}>
                                <SelectTrigger className="w-[140px] bg-background/50 border-input/50">
                                    <SelectValue placeholder="All Status" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Status</SelectItem>
                                    <SelectItem value="active">Active</SelectItem>
                                    <SelectItem value="pending">Pending</SelectItem>
                                    <SelectItem value="suspended">Suspended</SelectItem>
                                    <SelectItem value="inactive">Inactive</SelectItem>
                                </SelectContent>
                            </Select>
                            <Button 
                                onClick={handleSearch}
                                variant="secondary"
                                className="shadow-sm"
                            >
                                Apply
                            </Button>
                        </div>
                    </div>

                    {/* Table */}
                    <ModernTable>
                        <ModernTableHeader>
                            <ModernTableRow>
                                <ModernTableHead>Society</ModernTableHead>
                                <ModernTableHead>Info</ModernTableHead>
                                <ModernTableHead>Admin</ModernTableHead>
                                <ModernTableHead>Members</ModernTableHead>
                                <ModernTableHead>Status</ModernTableHead>
                                <ModernTableHead className="text-right">Actions</ModernTableHead>
                            </ModernTableRow>
                        </ModernTableHeader>
                        <ModernTableBody>
                            {societies.data.length > 0 ? (
                                societies.data.map((society) => (
                                    <ModernTableRow key={society.id}>
                                        <ModernTableCell className="font-medium">
                                            <div className="flex items-center gap-3">
                                                <div className="flex h-10 w-10 shrink-0 items-center justify-center rounded-lg bg-primary/10 text-primary">
                                                    <Building2 className="h-5 w-5" />
                                                </div>
                                                <div>
                                                    <p className="font-semibold text-foreground">{society.name}</p>
                                                    <p className="text-xs text-muted-foreground">{society.registration_no}</p>
                                                </div>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="text-sm">
                                                <p>{society.city || '-'}</p>
                                                <p className="text-xs text-muted-foreground">{society.state}</p>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            {society.admin ? (
                                                <div className="text-sm">
                                                    <p className="font-medium">{society.admin.name}</p>
                                                    <a href={`mailto:${society.admin.email}`} className="text-xs text-muted-foreground hover:text-primary transition-colors">
                                                        {society.admin.email || society.admin.phone}
                                                    </a>
                                                </div>
                                            ) : (
                                                <span className="text-sm text-muted-foreground italic">No Admin</span>
                                            )}
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="flex items-center gap-1.5">
                                                <Users className="h-4 w-4 text-muted-foreground" />
                                                <span className="text-sm font-medium">{society.users_count}</span>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <Select 
                                                value={society.status} 
                                                onValueChange={(value) => handleStatusChange(society.id, value)}
                                            >
                                                <SelectTrigger className="w-[110px] h-8 border-0 p-0 bg-transparent focus:ring-0">
                                                    <Badge variant="outline" className={`${getStatusColor(society.status)} border capitalize hover:bg-transparent`}>
                                                        {society.status}
                                                    </Badge>
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="active">Active</SelectItem>
                                                    <SelectItem value="pending">Pending</SelectItem>
                                                    <SelectItem value="suspended">Suspended</SelectItem>
                                                    <SelectItem value="inactive">Inactive</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </ModernTableCell>
                                        <ModernTableCell className="text-right">
                                            <div className="flex items-center justify-end gap-1">
                                                <Link href={`/super-admin/societies/${society.id}`}>
                                                    <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-primary">
                                                        <Eye className="h-4 w-4" />
                                                    </Button>
                                                </Link>
                                                <Link href={`/super-admin/societies/${society.id}/edit`}>
                                                    <Button variant="ghost" size="icon" className="h-8 w-8 hover:text-blue-500">
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                </Link>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-8 w-8 hover:text-red-500"
                                                    onClick={() => handleDelete(society.id)}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                ))
                            ) : (
                                <ModernTableRow>
                                    <ModernTableCell colSpan={6} className="h-24 text-center">
                                        <div className="flex flex-col items-center justify-center text-muted-foreground">
                                            <Building2 className="h-8 w-8 mb-2 opacity-20" />
                                            <p>No societies found</p>
                                        </div>
                                    </ModernTableCell>
                                </ModernTableRow>
                            )}
                        </ModernTableBody>
                    </ModernTable>

                    {/* Pagination */}
                    {societies.links && societies.links.length > 3 && (
                        <div className="flex items-center justify-between border-t border-border/50 pt-4">
                            <div className="text-sm text-muted-foreground">
                                Showing <span className="font-medium text-foreground">{societies.meta.from}</span> to <span className="font-medium text-foreground">{societies.meta.to}</span> of <span className="font-medium text-foreground">{societies.meta.total}</span> entries
                            </div>
                            <div className="flex gap-1">
                                {societies.links.map((link: any, index: number) => (
                                    <Link
                                        key={index}
                                        href={link.url || '#'}
                                        className={`px-3 py-1 rounded-md text-xs font-medium transition-colors ${
                                            link.active
                                                ? 'bg-primary text-primary-foreground shadow-sm'
                                                : 'text-muted-foreground hover:bg-muted hover:text-foreground'
                                        }`}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        </div>
                    )}
                </GlassCard>
            </div>
        </SuperAdminLayout>
    );
}
