import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import {
    Collapsible,
    CollapsibleContent,
    CollapsibleTrigger,
} from '@/components/ui/collapsible';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Form, Head, Link, useForm } from '@inertiajs/react';
import { 
    ChevronDown, 
    ChevronRight, 
    ArrowLeft,
    Check,
    CreditCard,
    Shield,
    Bell,
    Save,
    Trash2
} from 'lucide-react';
import { useState } from 'react';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';

interface SubscriptionPlan {
    id: number;
    name: string;
    display_name: string;
    monthly_price: number;
    max_users: number;
    storage_gb: number;
}

interface Society {
    id: number;
    name: string;
    registration_no: string;
    address: string;
    city: string;
    state: string;
    pincode: string;
    status: string;
    gst_no?: string;
    pan_no?: string;
    phone?: string;
    email?: string;
    subscription?: {
        plan_id: number;
        status: string;
        billing_cycle: string;
    };
    settings?: any;
    admin?: {
        name: string;
        email: string;
        phone: string;
    };
}

interface EditProps {
    society: Society;
    subscriptionPlans: SubscriptionPlan[];
    errors: any;
}

export default function Edit({ society, subscriptionPlans = [], errors }: EditProps) {
    const [subscriptionOpen, setSubscriptionOpen] = useState(false);
    const [settingsOpen, setSettingsOpen] = useState(false);

    const { data, setData, put, processing, delete: destroy } = useForm({
        name: society.name || '',
        registration_no: society.registration_no || '',
        address: society.address || '',
        city: society.city || '',
        state: society.state || '',
        pincode: society.pincode || '',
        status: society.status || 'active',
        gst_no: society.gst_no || '',
        pan_no: society.pan_no || '',
        phone: society.phone || '',
        email: society.email || '',
        admin: {
            name: society.admin?.name || '',
            email: society.admin?.email || '',
            phone: society.admin?.phone || '',
            password: '', // Password is optional on edit
        },
        subscription: {
            plan_id: society.subscription?.plan_id?.toString() || '',
            status: society.subscription?.status || 'active',
            billing_cycle: society.subscription?.billing_cycle || 'monthly',
        },
        settings: society.settings || {},
    });

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Societies', href: '/super-admin/societies' },
        { title: society.name, href: `/super-admin/societies/${society.id}` },
        { title: 'Edit', href: `/super-admin/societies/${society.id}/edit` },
    ];

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        put(`/super-admin/societies/${society.id}`);
    };

    const handleDelete = () => {
        if (confirm('Are you sure you want to delete this society? This action cannot be undone.')) {
            destroy(`/super-admin/societies/${society.id}`);
        }
    };

    const selectedPlanId = data.subscription.plan_id;

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title={`Edit ${society.name}`} />
            
            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-7xl mx-auto w-full">
                <PageHeader 
                    title={`Edit ${society.name}`}
                    description="Update society details, subscription, and settings"
                    actions={
                        <div className="flex gap-2">
                             <Button 
                                variant="destructive" 
                                size="sm" 
                                className="gap-2"
                                onClick={handleDelete}
                                type="button"
                            >
                                <Trash2 className="h-4 w-4" />
                                Delete
                            </Button>
                            <Link href="/super-admin/societies">
                                <Button variant="outline" size="sm" className="gap-2">
                                    <ArrowLeft className="h-4 w-4" />
                                    Back
                                </Button>
                            </Link>
                        </div>
                    }
                />

                <form onSubmit={handleSubmit} className="space-y-6">
                    <div className="grid gap-6">
                        {/* Basic Info & Location */}
                        <GlassCard className="p-0 overflow-hidden">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="h-2 w-2 rounded-full bg-primary" />
                                    Society Information
                                </h3>
                            </div>
                            <div className="p-6 space-y-6">
                                <div className="grid gap-6 md:grid-cols-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Society Name *</Label>
                                        <Input 
                                            id="name" 
                                            value={data.name} 
                                            onChange={(e) => setData('name', e.target.value)}
                                            required 
                                        />
                                        <InputError message={errors.name} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="registration_no">Registration Number *</Label>
                                        <Input 
                                            id="registration_no" 
                                            value={data.registration_no} 
                                            onChange={(e) => setData('registration_no', e.target.value)}
                                            required 
                                        />
                                        <InputError message={errors.registration_no} />
                                    </div>
                                </div>

                                <div className="grid gap-6 md:grid-cols-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="city">City *</Label>
                                        <Input 
                                            id="city" 
                                            value={data.city} 
                                            onChange={(e) => setData('city', e.target.value)}
                                            required 
                                        />
                                        <InputError message={errors.city} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="state">State *</Label>
                                        <Input 
                                            id="state" 
                                            value={data.state} 
                                            onChange={(e) => setData('state', e.target.value)}
                                            required 
                                        />
                                        <InputError message={errors.state} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="pincode">Pincode *</Label>
                                        <Input 
                                            id="pincode" 
                                            value={data.pincode} 
                                            onChange={(e) => setData('pincode', e.target.value)}
                                            required 
                                            maxLength={10} 
                                        />
                                        <InputError message={errors.pincode} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="status">Status *</Label>
                                        <Select 
                                            value={data.status} 
                                            onValueChange={(value) => setData('status', value)}
                                        >
                                            <SelectTrigger id="status"><SelectValue /></SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="pending">Pending</SelectItem>
                                                <SelectItem value="active">Active</SelectItem>
                                                <SelectItem value="suspended">Suspended</SelectItem>
                                                <SelectItem value="inactive">Inactive</SelectItem>
                                            </SelectContent>
                                        </Select>
                                        <InputError message={errors.status} />
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="address">Address *</Label>
                                    <textarea
                                        id="address"
                                        value={data.address}
                                        onChange={(e) => setData('address', e.target.value)}
                                        required
                                        rows={3}
                                        className="flex min-h-[80px] w-full rounded-md border border-input bg-background/50 px-3 py-2 text-sm shadow-sm placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring disabled:cursor-not-allowed disabled:opacity-50"
                                    />
                                    <InputError message={errors.address} />
                                </div>
                            </div>
                        </GlassCard>

                        {/* Admin Details */}
                        <GlassCard className="p-0 overflow-hidden">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="h-2 w-2 rounded-full bg-blue-500" />
                                    Primary Administrator (Update)
                                </h3>
                            </div>
                            <div className="p-6 grid gap-6 md:grid-cols-2">
                                <div className="space-y-2">
                                    <Label htmlFor="admin_name">Admin Name *</Label>
                                    <Input 
                                        id="admin_name" 
                                        value={data.admin.name} 
                                        onChange={(e) => setData('admin', { ...data.admin, name: e.target.value })}
                                        required 
                                    />
                                    <InputError message={errors?.['admin.name']} />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="admin_email">Admin Email *</Label>
                                    <Input 
                                        id="admin_email" 
                                        type="email"
                                        value={data.admin.email} 
                                        onChange={(e) => setData('admin', { ...data.admin, email: e.target.value })}
                                        required 
                                    />
                                    <InputError message={errors?.['admin.email']} />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="admin_phone">Admin Phone *</Label>
                                    <Input 
                                        id="admin_phone" 
                                        type="tel"
                                        value={data.admin.phone} 
                                        onChange={(e) => setData('admin', { ...data.admin, phone: e.target.value })}
                                        required 
                                    />
                                    <InputError message={errors?.['admin.phone']} />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="admin_password">New Password (Optional)</Label>
                                    <Input 
                                        id="admin_password" 
                                        type="password"
                                        value={data.admin.password} 
                                        onChange={(e) => setData('admin', { ...data.admin, password: e.target.value })}
                                        placeholder="Leave blank to keep current"
                                    />
                                    <InputError message={errors?.['admin.password']} />
                                </div>
                            </div>
                        </GlassCard>

                        {/* Additional Details */}
                        <GlassCard className="p-0 overflow-hidden">
                            <div className="border-b bg-muted/40 p-4">
                                <h3 className="font-semibold flex items-center gap-2">
                                    <div className="h-2 w-2 rounded-full bg-orange-500" />
                                    Additional Details
                                </h3>
                            </div>
                            <div className="p-6 space-y-6">
                                <div className="grid gap-6 md:grid-cols-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="gst_no">GST Number</Label>
                                        <Input 
                                            id="gst_no" 
                                            value={data.gst_no} 
                                            onChange={(e) => setData('gst_no', e.target.value)} 
                                        />
                                        <InputError message={errors.gst_no} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="pan_no">PAN Number</Label>
                                        <Input 
                                            id="pan_no" 
                                            value={data.pan_no} 
                                            onChange={(e) => setData('pan_no', e.target.value)} 
                                        />
                                        <InputError message={errors.pan_no} />
                                    </div>
                                </div>
                                <div className="grid gap-6 md:grid-cols-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="phone">Society Support Phone</Label>
                                        <Input 
                                            id="phone" 
                                            type="tel"
                                            value={data.phone} 
                                            onChange={(e) => setData('phone', e.target.value)} 
                                        />
                                        <InputError message={errors.phone} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="email">Society Support Email</Label>
                                        <Input 
                                            id="email" 
                                            type="email"
                                            value={data.email} 
                                            onChange={(e) => setData('email', e.target.value)} 
                                        />
                                        <InputError message={errors.email} />
                                    </div>
                                </div>
                            </div>
                        </GlassCard>

                         {/* Subscription Plan */}
                         <GlassCard className="p-0 overflow-hidden">
                                <Collapsible open={subscriptionOpen} onOpenChange={setSubscriptionOpen}>
                                    <CollapsibleTrigger className="flex w-full items-center justify-between p-4 border-b bg-muted/40 hover:bg-muted/60 transition-colors">
                                        <h3 className="font-semibold flex items-center gap-2">
                                            <div className="h-2 w-2 rounded-full bg-emerald-500" />
                                            Subscription Plan
                                        </h3>
                                        {subscriptionOpen ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                                    </CollapsibleTrigger>
                                    <CollapsibleContent>
                                        <div className="p-6 space-y-6">
                                            <div>
                                                <Label className="mb-4 block text-base">Select Plan *</Label>
                                                <div className="grid gap-4 md:grid-cols-3">
                                                    {subscriptionPlans.map((plan) => (
                                                        <div
                                                            key={plan.id}
                                                            onClick={() => setData('subscription', { ...data.subscription, plan_id: plan.id.toString() })}
                                                            className={`relative cursor-pointer rounded-xl border-2 p-5 transition-all duration-200 hover:shadow-md ${
                                                                selectedPlanId === plan.id.toString()
                                                                    ? 'border-primary bg-primary/5 dark:bg-primary/10'
                                                                    : 'border-border bg-background/50 hover:border-primary/50'
                                                            }`}
                                                        >
                                                            {selectedPlanId === plan.id.toString() && (
                                                                <div className="absolute right-3 top-3 h-5 w-5 rounded-full bg-primary text-white flex items-center justify-center">
                                                                    <Check className="h-3 w-3" />
                                                                </div>
                                                            )}
                                                            
                                                            <h4 className="font-bold text-lg mb-1">{plan.display_name}</h4>
                                                            <div className="flex items-baseline gap-1 mb-2">
                                                                <span className="text-2xl font-bold">₹{plan.monthly_price}</span>
                                                                <span className="text-sm text-muted-foreground">/mo</span>
                                                            </div>
                                                            
                                                            <div className="space-y-2 text-xs text-muted-foreground mt-4">
                                                                <div className="flex items-center gap-2">
                                                                    <div className="h-1.5 w-1.5 rounded-full bg-primary" />
                                                                    {plan.max_users === 9999 ? 'Unlimited' : plan.max_users} Users
                                                                </div>
                                                                <div className="flex items-center gap-2">
                                                                    <div className="h-1.5 w-1.5 rounded-full bg-primary" />
                                                                    {plan.storage_gb} GB Storage
                                                                </div>
                                                            </div>
                                                        </div>
                                                    ))}
                                                </div>
                                                <InputError message={errors?.['subscription.plan_id']} />
                                            </div>

                                            {/* Additional Fields hidden until plan selected */}
                                            {selectedPlanId && (
                                                <div className="bg-muted/30 rounded-xl p-6 border border-border/50 animate-in fade-in slide-in-from-top-4 duration-300">
                                                    <div className="grid gap-6 md:grid-cols-2">
                                                        <div className="space-y-2">
                                                            <Label>Billing Cycle</Label>
                                                            <Select 
                                                                value={data.subscription.billing_cycle} 
                                                                onValueChange={(val) => setData('subscription', { ...data.subscription, billing_cycle: val })} 
                                                            >
                                                                <SelectTrigger><SelectValue /></SelectTrigger>
                                                                <SelectContent>
                                                                    <SelectItem value="monthly">Monthly</SelectItem>
                                                                    <SelectItem value="quarterly">Quarterly</SelectItem>
                                                                    <SelectItem value="annual">Annual</SelectItem>
                                                                </SelectContent>
                                                            </Select>
                                                        </div>
                                                        <div className="space-y-2">
                                                            <Label>Subscription Status</Label>
                                                            <Select 
                                                                value={data.subscription.status} 
                                                                onValueChange={(val) => setData('subscription', { ...data.subscription, status: val })}
                                                            >
                                                                <SelectTrigger><SelectValue /></SelectTrigger>
                                                                <SelectContent>
                                                                    <SelectItem value="trial">Trial</SelectItem>
                                                                    <SelectItem value="active">Active</SelectItem>
                                                                    <SelectItem value="cancelled">Cancelled</SelectItem>
                                                                    <SelectItem value="expired">Expired</SelectItem>
                                                                </SelectContent>
                                                            </Select>
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>
                                    </CollapsibleContent>
                                </Collapsible>
                            </GlassCard>

                        {/* Settings */}
                        <GlassCard className="p-0 overflow-hidden">
                            <Collapsible open={settingsOpen} onOpenChange={setSettingsOpen}>
                                <CollapsibleTrigger className="flex w-full items-center justify-between p-4 border-b bg-muted/40 hover:bg-muted/60 transition-colors">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="h-2 w-2 rounded-full bg-purple-500" />
                                        Advanced Settings
                                    </h3>
                                    {settingsOpen ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                                </CollapsibleTrigger>
                                <CollapsibleContent>
                                    <div className="p-6 space-y-8">
                                        <div className="space-y-4">
                                            <div className="flex items-center gap-2 font-medium text-sm text-primary border-b pb-2">
                                                <CreditCard className="h-4 w-4" />
                                                Payment Gateways
                                            </div>
                                            <div className="grid gap-4 md:grid-cols-2">
                                                {[
                                                    { id: 'razorpay', label: 'Razorpay' },
                                                    { id: 'stripe', label: 'Stripe' },
                                                ].map((gw) => (
                                                    <div key={gw.id} className="flex items-center space-x-2">
                                                        <Checkbox 
                                                            id={`settings.${gw.id}_enabled`} 
                                                            checked={data.settings?.[`${gw.id}_enabled`]}
                                                            onCheckedChange={(checked) => setData('settings', { ...data.settings, [`${gw.id}_enabled`]: checked })}
                                                        />
                                                        <Label htmlFor={`settings.${gw.id}_enabled`}>Enable {gw.label}</Label>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>

                                        <div className="space-y-4">
                                            <div className="flex items-center gap-2 font-medium text-sm text-primary border-b pb-2">
                                                <Bell className="h-4 w-4" />
                                                Notifications
                                            </div>
                                            <div className="grid gap-4 md:grid-cols-2">
                                                {[
                                                    { id: 'email', label: 'Email Alerts' },
                                                    { id: 'sms', label: 'SMS Alerts' },
                                                    { id: 'fcm', label: 'Push Notifications (FCM)' },
                                                ].map((notif) => (
                                                    <div key={notif.id} className="flex items-center space-x-2">
                                                        <Checkbox 
                                                            id={`settings.${notif.id}_enabled`} 
                                                            checked={data.settings?.[`${notif.id}_enabled`]}
                                                            onCheckedChange={(checked) => setData('settings', { ...data.settings, [`${notif.id}_enabled`]: checked })}
                                                        />
                                                        <Label htmlFor={`settings.${notif.id}_enabled`}>{notif.label}</Label>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>

                                        <div className="space-y-4">
                                            <div className="flex items-center gap-2 font-medium text-sm text-primary border-b pb-2">
                                                <Shield className="h-4 w-4" />
                                                Modules
                                            </div>
                                            <div className="grid gap-4 md:grid-cols-2">
                                                {[
                                                    { id: 'visitor_management', label: 'Visitor Management' },
                                                    { id: 'maintenance_tickets', label: 'Maintenance Tickets' },
                                                    { id: 'complaints', label: 'Complaints' },
                                                    { id: 'events', label: 'Events & Notices' },
                                                ].map((module) => (
                                                    <div key={module.id} className="flex items-center space-x-2">
                                                        <Checkbox 
                                                            id={`settings.enable_${module.id}`} 
                                                            checked={data.settings?.[`enable_${module.id}`]}
                                                            onCheckedChange={(checked) => setData('settings', { ...data.settings, [`enable_${module.id}`]: checked })}
                                                        />
                                                        <Label htmlFor={`settings.enable_${module.id}`}>{module.label}</Label>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    </div>
                                </CollapsibleContent>
                            </Collapsible>
                        </GlassCard>

                        <div className="flex justify-end gap-4 pt-4">
                            <Link href="/super-admin/societies">
                                <Button variant="outline" type="button">Cancel</Button>
                            </Link>
                            <Button type="submit" className="bg-gradient-to-r from-primary to-primary/80 gap-2" disabled={processing}>
                                <Save className="h-4 w-4" />
                                {processing ? 'Saving...' : 'Save Changes'}
                            </Button>
                        </div>
                    </div>
                </form>
            </div>
        </SuperAdminLayout>
    );
}
