import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import {
    Collapsible,
    CollapsibleContent,
    CollapsibleTrigger,
} from '@/components/ui/collapsible';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Form, Head, Link } from '@inertiajs/react';
import { 
    ChevronDown, 
    ChevronRight, 
    Plus, 
    ArrowLeft,
    Check,
    CreditCard,
    Shield,
    Bell
} from 'lucide-react';
import { useState } from 'react';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { CardContent, CardHeader, CardTitle } from '@/components/ui/card';

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Societies', href: '/super-admin/societies' },
    { title: 'Create', href: '/super-admin/societies/create' },
];

interface SubscriptionPlan {
    id: number;
    name: string;
    display_name: string;
    description: string | null;
    monthly_price: number;
    quarterly_price: number | null;
    annual_price: number | null;
    max_users: number;
    max_units: number;
    max_buildings: number;
    storage_gb: number;
    sms_enabled: boolean;
    whatsapp_enabled: boolean;
    email_enabled: boolean;
    fcm_enabled: boolean;
    advanced_reports: boolean;
    api_access: boolean;
    custom_branding: boolean;
    priority_support: boolean;
    white_label: boolean;
}

interface CreateProps {
    subscriptionPlans?: SubscriptionPlan[];
    errors?: any;
}

export default function Create({ subscriptionPlans = [], errors }: CreateProps) {
    const [subscriptionOpen, setSubscriptionOpen] = useState(false);
    const [settingsOpen, setSettingsOpen] = useState(false);
    const [selectedPlanId, setSelectedPlanId] = useState<string>('');
    const [billingCycle, setBillingCycle] = useState<string>('monthly');
    const [startDate, setStartDate] = useState<string>(new Date().toISOString().split('T')[0]);

    const selectedPlan = subscriptionPlans.find((p) => p.id.toString() === selectedPlanId);

    const calculateEndDate = (start: string, cycle: string): string => {
        if (!start) return '';
        const startDateObj = new Date(start);
        let endDateObj = new Date(startDateObj);
        
        switch (cycle) {
            case 'monthly':
                endDateObj.setMonth(endDateObj.getMonth() + 1);
                break;
            case 'quarterly':
                endDateObj.setMonth(endDateObj.getMonth() + 3);
                break;
            case 'annual':
                endDateObj.setFullYear(endDateObj.getFullYear() + 1);
                break;
        }
        
        return endDateObj.toISOString().split('T')[0];
    };

    const endDate = calculateEndDate(startDate, billingCycle);

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title="Create Society" />
            
            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-7xl mx-auto w-full">
                <PageHeader 
                    title="Create New Society"
                    description="Add a new society to the system with subscription plan"
                    actions={
                        <Link href="/super-admin/societies">
                            <Button variant="outline" size="sm" className="gap-2">
                                <ArrowLeft className="h-4 w-4" />
                                Back to List
                            </Button>
                        </Link>
                    }
                />

                <Form
                    action="/super-admin/societies"
                    method="post"
                    className="space-y-6"
                >
                    {({ processing }) => (
                        <div className="grid gap-6">
                            {/* Always include plan_id field (required for validation) */}
                            <input type="hidden" name="subscription[plan_id]" value={selectedPlanId || ''} />
                            {/* Basic Info & Location */}
                            <GlassCard className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="h-2 w-2 rounded-full bg-primary" />
                                        Society Information
                                    </h3>
                                </div>
                                <div className="p-6 space-y-6">
                                    <div className="grid gap-6 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="name">Society Name *</Label>
                                            <Input id="name" name="name" required placeholder="Enter society name" />
                                            <InputError message={errors.name} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="registration_no">Registration Number *</Label>
                                            <Input id="registration_no" name="registration_no" required placeholder="e.g., MH/RES/2025/001" />
                                            <InputError message={errors.registration_no} />
                                        </div>
                                    </div>

                                    <div className="grid gap-6 md:grid-cols-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="city">City *</Label>
                                            <Input id="city" name="city" required placeholder="Enter city" />
                                            <InputError message={errors.city} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="state">State *</Label>
                                            <Input id="state" name="state" required placeholder="Enter state" />
                                            <InputError message={errors.state} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="pincode">Pincode *</Label>
                                            <Input id="pincode" name="pincode" required maxLength={10} placeholder="Enter pincode" />
                                            <InputError message={errors.pincode} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="status">Status *</Label>
                                            <Select name="status" defaultValue="pending" required>
                                                <SelectTrigger id="status"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="pending">Pending</SelectItem>
                                                    <SelectItem value="active">Active</SelectItem>
                                                    <SelectItem value="suspended">Suspended</SelectItem>
                                                    <SelectItem value="inactive">Inactive</SelectItem>
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors.status} />
                                        </div>
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="address">Address *</Label>
                                        <textarea
                                            id="address"
                                            name="address"
                                            required
                                            placeholder="Enter full address"
                                            rows={3}
                                            className="flex min-h-[80px] w-full rounded-md border border-input bg-background/50 px-3 py-2 text-sm shadow-sm placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring disabled:cursor-not-allowed disabled:opacity-50"
                                        />
                                        <InputError message={errors.address} />
                                    </div>
                                </div>
                            </GlassCard>

                            {/* Admin Details */}
                            <GlassCard className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="h-2 w-2 rounded-full bg-blue-500" />
                                        Primary Administrator
                                    </h3>
                                </div>
                                <div className="p-6 grid gap-6 md:grid-cols-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="admin_name">Admin Name *</Label>
                                        <Input id="admin_name" name="admin[name]" required placeholder="Enter admin name" />
                                        <InputError message={errors?.['admin.name']} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="admin_email">Admin Email *</Label>
                                        <Input id="admin_email" name="admin[email]" type="email" required placeholder="Enter admin email" />
                                        <InputError message={errors?.['admin.email']} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="admin_phone">Admin Phone *</Label>
                                        <Input id="admin_phone" name="admin[phone]" type="tel" required placeholder="Enter admin phone number" />
                                        <InputError message={errors?.['admin.phone']} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="admin_password">Admin Password *</Label>
                                        <Input id="admin_password" name="admin[password]" type="password" required placeholder="Enter admin password" />
                                        <InputError message={errors?.['admin.password']} />
                                        <p className="text-xs text-muted-foreground">Minimum 8 characters</p>
                                    </div>
                                </div>
                            </GlassCard>

                            {/* Additional Details */}
                            <GlassCard className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="h-2 w-2 rounded-full bg-orange-500" />
                                        Additional Details
                                    </h3>
                                </div>
                                <div className="p-6 space-y-6">
                                    <div className="grid gap-6 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="gst_no">GST Number</Label>
                                            <Input id="gst_no" name="gst_no" placeholder="Optional" />
                                            <InputError message={errors.gst_no} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="pan_no">PAN Number</Label>
                                            <Input id="pan_no" name="pan_no" placeholder="Optional" />
                                            <InputError message={errors.pan_no} />
                                        </div>
                                    </div>
                                    <div className="grid gap-6 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="phone">Society Support Phone</Label>
                                            <Input id="phone" name="phone" type="tel" placeholder="Optional" />
                                            <InputError message={errors.phone} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="email">Society Support Email</Label>
                                            <Input id="email" name="email" type="email" placeholder="Optional" />
                                            <InputError message={errors.email} />
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>

                            {/* Subscription Plan */}
                            <GlassCard className="p-0 overflow-hidden">
                                <Collapsible open={subscriptionOpen} onOpenChange={setSubscriptionOpen}>
                                    <CollapsibleTrigger className="flex w-full items-center justify-between p-4 border-b bg-muted/40 hover:bg-muted/60 transition-colors">
                                        <h3 className="font-semibold flex items-center gap-2">
                                            <div className="h-2 w-2 rounded-full bg-emerald-500" />
                                            Subscription Plan
                                        </h3>
                                        {subscriptionOpen ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                                    </CollapsibleTrigger>
                                    <CollapsibleContent>
                                        <div className="p-6 space-y-6">
                                            <div>
                                                <Label className="mb-4 block text-base">Select Plan *</Label>
                                                <div className="grid gap-4 md:grid-cols-3">
                                                    {subscriptionPlans.map((plan) => (
                                                        <div
                                                            key={plan.id}
                                                            onClick={() => setSelectedPlanId(plan.id.toString())}
                                                            className={`relative cursor-pointer rounded-xl border-2 p-5 transition-all duration-200 hover:shadow-md ${
                                                                selectedPlanId === plan.id.toString()
                                                                    ? 'border-primary bg-primary/5 dark:bg-primary/10'
                                                                    : 'border-border bg-background/50 hover:border-primary/50'
                                                            }`}
                                                        >
                                                            {selectedPlanId === plan.id.toString() && (
                                                                <div className="absolute right-3 top-3 h-5 w-5 rounded-full bg-primary text-white flex items-center justify-center">
                                                                    <Check className="h-3 w-3" />
                                                                </div>
                                                            )}
                                                            
                                                            <h4 className="font-bold text-lg mb-1">{plan.display_name}</h4>
                                                            <div className="flex items-baseline gap-1 mb-2">
                                                                <span className="text-2xl font-bold">₹{plan.monthly_price}</span>
                                                                <span className="text-sm text-muted-foreground">/mo</span>
                                                            </div>
                                                            
                                                            <div className="space-y-2 text-xs text-muted-foreground mt-4">
                                                                <div className="flex items-center gap-2">
                                                                    <div className="h-1.5 w-1.5 rounded-full bg-primary" />
                                                                    {plan.max_users === 9999 ? 'Unlimited' : plan.max_users} Users
                                                                </div>
                                                                <div className="flex items-center gap-2">
                                                                    <div className="h-1.5 w-1.5 rounded-full bg-primary" />
                                                                    {plan.storage_gb} GB Storage
                                                                </div>
                                                            </div>
                                                        </div>
                                                    ))}
                                                </div>
                                                {!selectedPlanId && (
                                                    <p className="text-sm text-destructive mt-2">Please select a subscription plan</p>
                                                )}
                                                <InputError message={errors?.['subscription.plan_id']} />
                                            </div>

                                            {/* Additional Fields hidden until plan selected */}
                                            {selectedPlan && (
                                                <div className="bg-muted/30 rounded-xl p-6 border border-border/50 animate-in fade-in slide-in-from-top-4 duration-300">
                                                    <div className="grid gap-6 md:grid-cols-2">
                                                        <div className="space-y-2">
                                                            <Label>Billing Cycle</Label>
                                                            <Select name="subscription[billing_cycle]" value={billingCycle} onValueChange={setBillingCycle} required>
                                                                <SelectTrigger><SelectValue /></SelectTrigger>
                                                                <SelectContent>
                                                                    <SelectItem value="monthly">Monthly</SelectItem>
                                                                    <SelectItem value="quarterly">Quarterly</SelectItem>
                                                                    <SelectItem value="annual">Annual</SelectItem>
                                                                </SelectContent>
                                                            </Select>
                                                        </div>
                                                        <div className="space-y-2">
                                                            <Label>Start Date</Label>
                                                            <Input type="date" name="subscription[start_date]" required value={startDate} onChange={(e) => setStartDate(e.target.value)} />
                                                        </div>
                                                        <div className="space-y-2">
                                                            <Label>End Date (Auto)</Label>
                                                            <Input value={endDate} readOnly className="bg-muted" />
                                                            <input type="hidden" name="subscription[end_date]" value={endDate} />
                                                        </div>
                                                        <div className="space-y-2">
                                                            <Label>Subscription Status</Label>
                                                            <Select name="subscription[status]" defaultValue="trial" required>
                                                                <SelectTrigger><SelectValue /></SelectTrigger>
                                                                <SelectContent>
                                                                    <SelectItem value="trial">Trial</SelectItem>
                                                                    <SelectItem value="active">Active</SelectItem>
                                                                </SelectContent>
                                                            </Select>
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>
                                    </CollapsibleContent>
                                </Collapsible>
                            </GlassCard>

                            {/* Settings */}
                            <GlassCard className="p-0 overflow-hidden">
                                <Collapsible open={settingsOpen} onOpenChange={setSettingsOpen}>
                                    <CollapsibleTrigger className="flex w-full items-center justify-between p-4 border-b bg-muted/40 hover:bg-muted/60 transition-colors">
                                        <h3 className="font-semibold flex items-center gap-2">
                                            <div className="h-2 w-2 rounded-full bg-purple-500" />
                                            Advanced Settings
                                        </h3>
                                        {settingsOpen ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                                    </CollapsibleTrigger>
                                    <CollapsibleContent>
                                        <div className="p-6 space-y-8">
                                            <div className="space-y-4">
                                                <div className="flex items-center gap-2 font-medium text-sm text-primary border-b pb-2">
                                                    <CreditCard className="h-4 w-4" />
                                                    Payment Gateways
                                                </div>
                                                <div className="grid gap-4 md:grid-cols-2">
                                                    {[
                                                        { id: 'razorpay', label: 'Razorpay' },
                                                        { id: 'stripe', label: 'Stripe' },
                                                    ].map((gw) => (
                                                        <div key={gw.id} className="flex items-center space-x-2">
                                                            <Checkbox id={`settings.${gw.id}_enabled`} name={`settings[${gw.id}_enabled]`} />
                                                            <Label htmlFor={`settings.${gw.id}_enabled`}>Enable {gw.label}</Label>
                                                        </div>
                                                    ))}
                                                </div>
                                            </div>

                                            <div className="space-y-4">
                                                <div className="flex items-center gap-2 font-medium text-sm text-primary border-b pb-2">
                                                    <Bell className="h-4 w-4" />
                                                    Notifications
                                                </div>
                                                <div className="grid gap-4 md:grid-cols-2">
                                                    {[
                                                        { id: 'email', label: 'Email Alerts', default: true },
                                                        { id: 'sms', label: 'SMS Alerts', default: false },
                                                        { id: 'fcm', label: 'Push Notifications (FCM)', default: true },
                                                    ].map((notif) => (
                                                        <div key={notif.id} className="flex items-center space-x-2">
                                                            <Checkbox 
                                                                id={`settings.${notif.id}_enabled`} 
                                                                name={`settings[${notif.id}_enabled]`} 
                                                                defaultChecked={notif.default}
                                                            />
                                                            <Label htmlFor={`settings.${notif.id}_enabled`}>{notif.label}</Label>
                                                        </div>
                                                    ))}
                                                </div>
                                            </div>

                                            <div className="space-y-4">
                                                <div className="flex items-center gap-2 font-medium text-sm text-primary border-b pb-2">
                                                    <Shield className="h-4 w-4" />
                                                    Modules
                                                </div>
                                                <div className="grid gap-4 md:grid-cols-2">
                                                    {[
                                                        { id: 'visitor_management', label: 'Visitor Management', default: true },
                                                        { id: 'maintenance_tickets', label: 'Maintenance Tickets', default: true },
                                                        { id: 'complaints', label: 'Complaints', default: true },
                                                        { id: 'events', label: 'Events & Notices', default: true },
                                                    ].map((module) => (
                                                        <div key={module.id} className="flex items-center space-x-2">
                                                            <Checkbox 
                                                                id={`settings.enable_${module.id}`} 
                                                                name={`settings[enable_${module.id}]`} 
                                                                defaultChecked={module.default}
                                                            />
                                                            <Label htmlFor={`settings.enable_${module.id}`}>{module.label}</Label>
                                                        </div>
                                                    ))}
                                                </div>
                                            </div>
                                        </div>
                                    </CollapsibleContent>
                                </Collapsible>
                            </GlassCard>

                            <div className="flex justify-end gap-4 pt-4">
                                <Link href="/super-admin/societies">
                                    <Button variant="outline" type="button">Cancel</Button>
                                </Link>
                                <Button type="submit" className="bg-gradient-to-r from-primary to-primary/80" disabled={processing || !selectedPlanId}>
                                    {processing ? 'Creating...' : 'Create Society'}
                                </Button>
                            </div>
                        </div>
                    )}
                </Form>
            </div>
        </SuperAdminLayout>
    );
}
