import { Button } from '@/components/ui/button';
import { CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Textarea } from '@/components/ui/textarea';
import SuperAdminLayout from '@/layouts/super-admin-layout';
import { Head, useForm, router } from '@inertiajs/react';
import { FormEventHandler } from 'react';
import { 
    Save, 
    Globe, 
    Palette, 
    Layout, 
    CreditCard, 
    Mail, 
    ShieldCheck, 
    Bell, 
    Upload, 
    Trash2, 
    CheckCircle, 
    XCircle, 
    Monitor, 
    Users,
    Type,
    Image,
    MousePointerClick,
    HelpCircle,
    FileText
} from 'lucide-react';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import InputError from '@/components/input-error';

interface SettingsProps {
    settings: Record<string, any[]>;
}

export default function Settings({ settings }: SettingsProps) {
    // Flatten settings for form default values
    const defaultValues: Record<string, any> = {};
    Object.values(settings).flat().forEach((setting) => {
        defaultValues[setting.key] = setting.value;
    });

    const { data, setData, post, processing, errors } = useForm<Record<string, any>>(defaultValues);

    const handleSubmit: FormEventHandler = (e) => {
        e.preventDefault();
        // @ts-ignore
        post('/super-admin/settings', {
            preserveScroll: true,
        });
    };

    const handleFileUpload = (key: string, file: File) => {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('key', key);
        router.post('/super-admin/settings/upload', formData, {
            forceFormData: true,
            preserveScroll: true,
        });
    };

    return (
        <SuperAdminLayout breadcrumbs={[
            { title: 'System', href: '#' },
            { title: 'Settings', href: '/super-admin/settings' },
        ]}>
            <Head title="System Settings" />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-[1600px] mx-auto w-full">
                <PageHeader 
                    title="System Settings"
                    description="Configure your application preferences, integrations, and content."
                    actions={
                        <Button 
                            onClick={handleSubmit} 
                            disabled={processing} 
                            size="sm"
                            className="gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25"
                        >
                            {processing ? <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent" /> : <Save className="h-4 w-4" />}
                            Save Changes
                        </Button>
                    }
                />

                <form onSubmit={handleSubmit} className="flex-1">
                    <Tabs defaultValue="general" className="w-full space-y-6">
                        <TabsList className="grid w-full grid-cols-2 md:grid-cols-4 lg:grid-cols-8 h-auto p-1 bg-muted/40 backdrop-blur-sm rounded-xl border border-border/50">
                            {[
                                { value: 'general', icon: Globe, label: 'General' },
                                { value: 'branding', icon: Palette, label: 'Branding' },
                                { value: 'features', icon: Layout, label: 'Features' },
                                { value: 'payment', icon: CreditCard, label: 'Payments' },
                                { value: 'firebase', icon: Bell, label: 'Push Notif.' },
                                { value: 'email', icon: Mail, label: 'Email' },
                                { value: 'legal', icon: ShieldCheck, label: 'Legal' },
                                { value: 'frontend', icon: Monitor, label: 'Frontend' },
                            ].map((tab) => (
                                <TabsTrigger 
                                    key={tab.value} 
                                    value={tab.value} 
                                    className="gap-2 py-2.5 data-[state=active]:bg-background/80 data-[state=active]:backdrop-blur-sm data-[state=active]:text-primary data-[state=active]:shadow-sm rounded-lg transition-all"
                                >
                                    <tab.icon className="h-4 w-4" />
                                    <span className="hidden lg:inline text-xs font-medium">{tab.label}</span>
                                    <span className="lg:hidden text-xs font-medium">{tab.label}</span>
                                </TabsTrigger>
                            ))}
                        </TabsList>

                        {/* General & SEO Tab */}
                        <TabsContent value="general">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="p-1.5 rounded-md bg-blue-500/10 text-blue-600">
                                            <Globe className="h-4 w-4" />
                                        </div>
                                        SEO Settings
                                    </h3>
                                    <p className="text-sm text-muted-foreground ml-9">Manage search engine optimization settings for public pages.</p>
                                </div>
                                <div className="p-6 space-y-6">
                                    <div className="space-y-2">
                                        <Label htmlFor="meta_title">Meta Title</Label>
                                        <div className="relative">
                                            <Type className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                            <Input
                                                id="meta_title"
                                                value={data.meta_title || ''}
                                                onChange={(e) => setData('meta_title', e.target.value)}
                                                className="pl-9 bg-background/50"
                                            />
                                        </div>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="meta_description">Meta Description</Label>
                                        <div className="relative">
                                            <FileText className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                                            <Textarea
                                                id="meta_description"
                                                value={data.meta_description || ''}
                                                onChange={(e) => setData('meta_description', e.target.value)}
                                                className="pl-9 bg-background/50 min-h-[100px]"
                                            />
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>
                        </TabsContent>

                        {/* Branding Tab */}
                        <TabsContent value="branding">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="p-1.5 rounded-md bg-purple-500/10 text-purple-600">
                                            <Palette className="h-4 w-4" />
                                        </div>
                                        Branding
                                    </h3>
                                    <p className="text-sm text-muted-foreground ml-9">Customize the look and feel of your application.</p>
                                </div>
                                <div className="p-6 space-y-8">
                                    <div className="space-y-2 max-w-md">
                                        <Label htmlFor="app_name">Application Name</Label>
                                        <div className="relative">
                                            <Type className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                            <Input
                                                id="app_name"
                                                value={data.app_name || ''}
                                                onChange={(e) => setData('app_name', e.target.value)}
                                                className="pl-9 bg-background/50"
                                            />
                                        </div>
                                    </div>
                                    
                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                                        {/* Logo Upload */}
                                        <div className="space-y-4">
                                            <Label className="flex items-center gap-2">
                                                <Image className="h-4 w-4 text-muted-foreground" />
                                                App Logo (Light Mode)
                                            </Label>
                                            <div className="p-4 border-2 border-dashed border-border rounded-xl bg-background/50 text-center space-y-4 hover:border-primary/50 transition-colors">
                                                <div className="w-32 h-20 mx-auto flex items-center justify-center bg-white rounded-lg shadow-sm">
                                                    {data.app_logo ? (
                                                        <img src={data.app_logo} alt="Logo" className="w-full h-full object-contain p-2" />
                                                    ) : (
                                                        <Palette className="w-8 h-8 text-muted-foreground" />
                                                    )}
                                                </div>
                                                <div className="flex justify-center gap-2">
                                                    <Button
                                                        type="button"
                                                        variant="outline"
                                                        size="sm"
                                                        className="h-8 text-xs"
                                                        onClick={() => document.getElementById('app_logo_upload')?.click()}
                                                    >
                                                        Upload
                                                    </Button>
                                                    {data.app_logo && (
                                                        <Button
                                                            type="button"
                                                            variant="ghost"
                                                            size="sm"
                                                            className="h-8 text-xs text-destructive hover:text-destructive"
                                                            onClick={() => setData('app_logo', '')}
                                                        >
                                                            Remove
                                                        </Button>
                                                    )}
                                                </div>
                                                <input
                                                    type="file"
                                                    accept="image/*"
                                                    id="app_logo_upload"
                                                    className="hidden"
                                                    onChange={(e) => e.target.files?.[0] && handleFileUpload('app_logo', e.target.files[0])}
                                                />
                                            </div>
                                            <p className="text-xs text-muted-foreground text-center">For light backgrounds</p>
                                        </div>

                                        {/* Dark Mode Logo Upload */}
                                        <div className="space-y-4">
                                            <Label className="flex items-center gap-2">
                                                <Image className="h-4 w-4 text-muted-foreground" />
                                                App Logo (Dark Mode)
                                            </Label>
                                            <div className="p-4 border-2 border-dashed border-border rounded-xl bg-background/50 text-center space-y-4 hover:border-primary/50 transition-colors">
                                                <div className="w-32 h-20 mx-auto flex items-center justify-center bg-slate-900 rounded-lg shadow-sm">
                                                    {data.app_logo_dark ? (
                                                        <img src={data.app_logo_dark} alt="Dark Logo" className="w-full h-full object-contain p-2" />
                                                    ) : (
                                                        <Palette className="w-8 h-8 text-slate-400" />
                                                    )}
                                                </div>
                                                <div className="flex justify-center gap-2">
                                                    <Button
                                                        type="button"
                                                        variant="outline"
                                                        size="sm"
                                                        className="h-8 text-xs"
                                                        onClick={() => document.getElementById('app_logo_dark_upload')?.click()}
                                                    >
                                                        Upload
                                                    </Button>
                                                    {data.app_logo_dark && (
                                                        <Button
                                                            type="button"
                                                            variant="ghost"
                                                            size="sm"
                                                            className="h-8 text-xs text-destructive hover:text-destructive"
                                                            onClick={() => setData('app_logo_dark', '')}
                                                        >
                                                            Remove
                                                        </Button>
                                                    )}
                                                </div>
                                                <input
                                                    type="file"
                                                    accept="image/*"
                                                    id="app_logo_dark_upload"
                                                    className="hidden"
                                                    onChange={(e) => e.target.files?.[0] && handleFileUpload('app_logo_dark', e.target.files[0])}
                                                />
                                            </div>
                                            <p className="text-xs text-muted-foreground text-center">For dark backgrounds</p>
                                        </div>
                                        
                                        {/* Favicon Upload */}
                                        <div className="space-y-4">
                                            <Label className="flex items-center gap-2">
                                                <Image className="h-4 w-4 text-muted-foreground" />
                                                Favicon
                                            </Label>
                                            <div className="p-4 border-2 border-dashed border-border rounded-xl bg-background/50 text-center space-y-4 hover:border-primary/50 transition-colors">
                                                <div className="w-20 h-20 mx-auto flex items-center justify-center bg-transparent rounded-lg">
                                                    {data.app_favicon ? (
                                                        <img src={data.app_favicon} alt="Favicon" className="w-full h-full object-contain" />
                                                    ) : (
                                                        <Globe className="w-8 h-8 text-muted-foreground" />
                                                    )}
                                                </div>
                                                <div className="flex justify-center gap-2">
                                                    <Button
                                                        type="button"
                                                        variant="outline"
                                                        size="sm"
                                                        className="h-8 text-xs"
                                                        onClick={() => document.getElementById('app_favicon_upload')?.click()}
                                                    >
                                                        Upload
                                                    </Button>
                                                    {data.app_favicon && (
                                                        <Button
                                                            type="button"
                                                            variant="ghost"
                                                            size="sm"
                                                            className="h-8 text-xs text-destructive hover:text-destructive"
                                                            onClick={() => setData('app_favicon', '')}
                                                        >
                                                            Remove
                                                        </Button>
                                                    )}
                                                </div>
                                                <input
                                                    type="file"
                                                    accept="image/*"
                                                    id="app_favicon_upload"
                                                    className="hidden"
                                                    onChange={(e) => e.target.files?.[0] && handleFileUpload('app_favicon', e.target.files[0])}
                                                />
                                            </div>
                                            <p className="text-xs text-muted-foreground text-center">32x32px or 64x64px</p>
                                        </div>
                                    </div>
                                    
                                    <div className="space-y-2 max-w-sm">
                                        <Label htmlFor="primary_color">Primary Color</Label>
                                        <div className="flex gap-3">
                                            <Input
                                                id="primary_color"
                                                type="color"
                                                className="w-12 h-10 p-1 cursor-pointer"
                                                value={data.primary_color || '#10b981'}
                                                onChange={(e) => setData('primary_color', e.target.value)}
                                            />
                                            <Input
                                                type="text"
                                                value={data.primary_color || ''}
                                                onChange={(e) => setData('primary_color', e.target.value)}
                                                className="flex-1 bg-background/50 font-mono uppercase"
                                                placeholder="#000000"
                                            />
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>
                        </TabsContent>

                        {/* Features Tab */}
                        <TabsContent value="features">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="p-1.5 rounded-md bg-pink-500/10 text-pink-600">
                                            <Layout className="h-4 w-4" />
                                        </div>
                                        Feature Flags
                                    </h3>
                                    <p className="text-sm text-muted-foreground ml-9">Enable or disable specific features of the application.</p>
                                </div>
                                <div className="p-6 space-y-4">
                                    {[
                                        { key: 'enable_landing_page', label: 'Public Landing Page', desc: 'Show the landing page at the root URL.' },
                                        { key: 'enable_pricing_page', label: 'Public Pricing Page', desc: 'Show the pricing page to public visitors.' },
                                        { key: 'enable_registration', label: 'New Registrations', desc: 'Allow new societies to register.' },
                                    ].map((feature) => (
                                        <div key={feature.key} className="flex items-center justify-between p-4 rounded-xl border border-border/50 bg-background/30 hover:bg-background/50 transition-colors">
                                            <div className="space-y-0.5">
                                                <Label className="text-base">{feature.label}</Label>
                                                <p className="text-xs text-muted-foreground">{feature.desc}</p>
                                            </div>
                                            <Switch
                                                checked={data[feature.key] === '1'}
                                                onCheckedChange={(checked) => setData(feature.key, checked ? '1' : '0')}
                                                className="data-[state=checked]:bg-primary"
                                            />
                                        </div>
                                    ))}
                                </div>
                            </GlassCard>
                        </TabsContent>

                        {/* Payment Gateways Tab */}
                        <TabsContent value="payment">
                            <div className="grid gap-6">
                                {/* Stripe */}
                                <GlassCard intensity="low" className="p-0 overflow-hidden">
                                    <div className="border-b bg-muted/40 p-4 flex items-center justify-between">
                                        <div>
                                            <h3 className="font-semibold flex items-center gap-2">
                                                <div className="p-1.5 rounded-md bg-indigo-500/10 text-indigo-600">
                                                    <CreditCard className="h-4 w-4" />
                                                </div>
                                                Stripe
                                            </h3>
                                            <p className="text-sm text-muted-foreground ml-9">Configure Stripe payment gateway.</p>
                                        </div>
                                        <Switch
                                            checked={data.stripe_enabled === '1'}
                                            onCheckedChange={(checked) => setData('stripe_enabled', checked ? '1' : '0')}
                                            className="data-[state=checked]:bg-indigo-600"
                                        />
                                    </div>
                                    {data.stripe_enabled === '1' && (
                                        <div className="p-6 space-y-4 animate-in slide-in-from-top-2 duration-200">
                                            <div className="space-y-2">
                                                <Label htmlFor="stripe_key">Publishable Key</Label>
                                                <Input
                                                    id="stripe_key"
                                                    value={data.stripe_key || ''}
                                                    onChange={(e) => setData('stripe_key', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="stripe_secret">Secret Key</Label>
                                                <Input
                                                    id="stripe_secret"
                                                    type="password"
                                                    value={data.stripe_secret || ''}
                                                    onChange={(e) => setData('stripe_secret', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                            </div>
                                        </div>
                                    )}
                                </GlassCard>

                                {/* Razorpay */}
                                <GlassCard intensity="low" className="p-0 overflow-hidden">
                                    <div className="border-b bg-muted/40 p-4 flex items-center justify-between">
                                        <div>
                                            <h3 className="font-semibold flex items-center gap-2">
                                                <div className="p-1.5 rounded-md bg-blue-500/10 text-blue-600">
                                                    <CreditCard className="h-4 w-4" />
                                                </div>
                                                Razorpay
                                            </h3>
                                            <p className="text-sm text-muted-foreground ml-9">Configure Razorpay payment gateway.</p>
                                        </div>
                                        <Switch
                                            checked={data.razorpay_enabled === '1'}
                                            onCheckedChange={(checked) => setData('razorpay_enabled', checked ? '1' : '0')}
                                            className="data-[state=checked]:bg-blue-600"
                                        />
                                    </div>
                                    {data.razorpay_enabled === '1' && (
                                        <div className="p-6 space-y-4 animate-in slide-in-from-top-2 duration-200">
                                            <div className="space-y-2">
                                                <Label htmlFor="razorpay_key">Key ID</Label>
                                                <Input
                                                    id="razorpay_key"
                                                    value={data.razorpay_key || ''}
                                                    onChange={(e) => setData('razorpay_key', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="razorpay_secret">Key Secret</Label>
                                                <Input
                                                    id="razorpay_secret"
                                                    type="password"
                                                    value={data.razorpay_secret || ''}
                                                    onChange={(e) => setData('razorpay_secret', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="razorpay_webhook_secret">Webhook Secret</Label>
                                                <Input
                                                    id="razorpay_webhook_secret"
                                                    type="password"
                                                    value={data.razorpay_webhook_secret || ''}
                                                    onChange={(e) => setData('razorpay_webhook_secret', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                                <p className="text-xs text-muted-foreground">
                                                    Used to verify Razorpay webhooks for SaaS subscription payments.
                                                </p>
                                            </div>
                                        </div>
                                    )}
                                </GlassCard>

                                {/* PayPal */}
                                <GlassCard intensity="low" className="p-0 overflow-hidden">
                                    <div className="border-b bg-muted/40 p-4 flex items-center justify-between">
                                        <div>
                                            <h3 className="font-semibold flex items-center gap-2">
                                                <div className="p-1.5 rounded-md bg-sky-500/10 text-sky-600">
                                                    <CreditCard className="h-4 w-4" />
                                                </div>
                                                PayPal
                                            </h3>
                                            <p className="text-sm text-muted-foreground ml-9">Configure PayPal payment gateway.</p>
                                        </div>
                                        <Switch
                                            checked={data.paypal_enabled === '1'}
                                            onCheckedChange={(checked) => setData('paypal_enabled', checked ? '1' : '0')}
                                            className="data-[state=checked]:bg-sky-600"
                                        />
                                    </div>
                                    {data.paypal_enabled === '1' && (
                                        <div className="p-6 space-y-4 animate-in slide-in-from-top-2 duration-200">
                                            <div className="space-y-2">
                                                <Label htmlFor="paypal_client_id">Client ID</Label>
                                                <Input
                                                    id="paypal_client_id"
                                                    value={data.paypal_client_id || ''}
                                                    onChange={(e) => setData('paypal_client_id', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="paypal_secret">Secret</Label>
                                                <Input
                                                    id="paypal_secret"
                                                    type="password"
                                                    value={data.paypal_secret || ''}
                                                    onChange={(e) => setData('paypal_secret', e.target.value)}
                                                    className="bg-background/50 font-mono"
                                                />
                                            </div>
                                        </div>
                                    )}
                                </GlassCard>
                            </div>
                        </TabsContent>

                        {/* Firebase Push Notifications Tab */}
                        <TabsContent value="firebase">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4 flex items-center justify-between">
                                    <div>
                                        <h3 className="font-semibold flex items-center gap-2">
                                            <div className="p-1.5 rounded-md bg-orange-500/10 text-orange-600">
                                                <Bell className="h-4 w-4" />
                                            </div>
                                            Firebase Cloud Messaging
                                        </h3>
                                        <p className="text-sm text-muted-foreground ml-9">Configure push notifications for mobile app.</p>
                                    </div>
                                    {data.firebase_enabled === '1' ? (
                                        <div className="flex items-center gap-2 px-3 py-1.5 rounded-full bg-emerald-500/10 text-emerald-600 text-xs font-medium border border-emerald-500/20">
                                            <CheckCircle className="h-3 w-3" />
                                            Enabled
                                        </div>
                                    ) : (
                                        <div className="flex items-center gap-2 px-3 py-1.5 rounded-full bg-amber-500/10 text-amber-600 text-xs font-medium border border-amber-500/20">
                                            <XCircle className="h-3 w-3" />
                                            Not Configured
                                        </div>
                                    )}
                                </div>
                                <div className="p-6 space-y-6">
                                    {/* Status Display */}
                                    {data.firebase_project_id && (
                                        <div className="p-4 rounded-xl border border-emerald-500/30 bg-emerald-500/5">
                                            <div className="flex items-center gap-3">
                                                <CheckCircle className="h-5 w-5 text-emerald-600" />
                                                <div>
                                                    <p className="font-medium text-emerald-700 dark:text-emerald-400">Firebase Connected</p>
                                                    <p className="text-sm text-muted-foreground">Project ID: <span className="font-mono">{data.firebase_project_id}</span></p>
                                                </div>
                                            </div>
                                        </div>
                                    )}

                                    {/* Upload Instructions */}
                                    <div className="p-4 rounded-xl border border-border/50 bg-background/30">
                                        <h4 className="font-medium mb-2 text-sm">How to get Firebase Service Account Key:</h4>
                                        <ol className="text-sm text-muted-foreground space-y-1 list-decimal list-inside">
                                            <li>Go to <a href="https://console.firebase.google.com" target="_blank" rel="noopener noreferrer" className="text-primary hover:underline">Firebase Console</a></li>
                                            <li>Select your project → Project Settings → Service accounts</li>
                                            <li>Click "Generate new private key"</li>
                                            <li>Download the JSON file and upload it below</li>
                                        </ol>
                                    </div>

                                    {/* File Upload */}
                                    <div className="space-y-4">
                                        <Label>Service Account Key (JSON)</Label>
                                        <div className="flex flex-col sm:flex-row gap-3">
                                            <input
                                                type="file"
                                                accept=".json"
                                                id="firebase_credentials"
                                                className="hidden"
                                                onChange={(e) => {
                                                    const file = e.target.files?.[0];
                                                    if (file) {
                                                        const formData = new FormData();
                                                        formData.append('firebase_credentials', file);
                                                        router.post('/super-admin/settings/firebase/upload', formData, {
                                                            forceFormData: true,
                                                            preserveScroll: true,
                                                        });
                                                    }
                                                }}
                                            />
                                            <Button
                                                type="button"
                                                variant="outline"
                                                className="gap-2 flex-1"
                                                onClick={() => document.getElementById('firebase_credentials')?.click()}
                                            >
                                                <Upload className="h-4 w-4" />
                                                {data.firebase_enabled === '1' ? 'Update Credentials' : 'Upload Credentials'}
                                            </Button>
                                            {data.firebase_enabled === '1' && (
                                                <Button
                                                    type="button"
                                                    variant="destructive"
                                                    className="gap-2"
                                                    onClick={() => {
                                                        if (confirm('Are you sure you want to remove Firebase credentials? Push notifications will be disabled.')) {
                                                            router.delete('/super-admin/settings/firebase/remove', {
                                                                preserveScroll: true,
                                                            });
                                                        }
                                                    }}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                    Remove
                                                </Button>
                                            )}
                                        </div>
                                        <p className="text-xs text-muted-foreground">
                                            The credentials file is stored securely and never exposed publicly.
                                        </p>
                                    </div>

                                    {/* Test Notification */}
                                    {data.firebase_enabled === '1' && (
                                        <div className="pt-4 border-t border-border/50">
                                            <div className="flex items-center justify-between">
                                                <div>
                                                    <h4 className="font-medium text-sm">Push Notification Features</h4>
                                                    <p className="text-xs text-muted-foreground">Enabled notification types:</p>
                                                </div>
                                            </div>
                                            <div className="mt-3 grid grid-cols-1 md:grid-cols-2 gap-3">
                                                {[
                                                    'Visitor arrival notifications to residents',
                                                    'Visitor approval/rejection notifications to guards',
                                                    'Bill generation notifications',
                                                    'Announcement notifications'
                                                ].map((note, i) => (
                                                    <div key={i} className="flex items-center gap-2 text-xs">
                                                        <CheckCircle className="h-3 w-3 text-emerald-600" />
                                                        <span>{note}</span>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </GlassCard>
                        </TabsContent>

                        {/* Email Configuration Tab */}
                        <TabsContent value="email">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="p-1.5 rounded-md bg-teal-500/10 text-teal-600">
                                            <Mail className="h-4 w-4" />
                                        </div>
                                        SMTP Configuration
                                    </h3>
                                    <p className="text-sm text-muted-foreground ml-9">Configure email settings for sending notifications.</p>
                                </div>
                                <div className="p-6 space-y-6">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_host">SMTP Host</Label>
                                            <Input
                                                id="mail_host"
                                                value={data.mail_host || ''}
                                                onChange={(e) => setData('mail_host', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="smtp.example.com"
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_port">SMTP Port</Label>
                                            <Input
                                                id="mail_port"
                                                value={data.mail_port || ''}
                                                onChange={(e) => setData('mail_port', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="587"
                                            />
                                        </div>
                                    </div>
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_username">Username</Label>
                                            <Input
                                                id="mail_username"
                                                value={data.mail_username || ''}
                                                onChange={(e) => setData('mail_username', e.target.value)}
                                                className="bg-background/50"
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_password">Password</Label>
                                            <Input
                                                id="mail_password"
                                                type="password"
                                                value={data.mail_password || ''}
                                                onChange={(e) => setData('mail_password', e.target.value)}
                                                className="bg-background/50"
                                            />
                                        </div>
                                    </div>
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_encryption">Encryption</Label>
                                            <Input
                                                id="mail_encryption"
                                                value={data.mail_encryption || 'tls'}
                                                onChange={(e) => setData('mail_encryption', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="tls"
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_mailer">Mailer</Label>
                                            <Input
                                                id="mail_mailer"
                                                value={data.mail_mailer || 'smtp'}
                                                onChange={(e) => setData('mail_mailer', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="smtp"
                                            />
                                        </div>
                                    </div>
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_from_address">From Address</Label>
                                            <Input
                                                id="mail_from_address"
                                                value={data.mail_from_address || ''}
                                                onChange={(e) => setData('mail_from_address', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="noreply@example.com"
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="mail_from_name">From Name</Label>
                                            <Input
                                                id="mail_from_name"
                                                value={data.mail_from_name || ''}
                                                onChange={(e) => setData('mail_from_name', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="App Name"
                                            />
                                        </div>
                                    </div>
                                </div>
                            </GlassCard>
                        </TabsContent>

                        {/* Legal Pages Tab */}
                        <TabsContent value="legal">
                            <GlassCard intensity="low" className="p-0 overflow-hidden">
                                <div className="border-b bg-muted/40 p-4">
                                    <h3 className="font-semibold flex items-center gap-2">
                                        <div className="p-1.5 rounded-md bg-emerald-500/10 text-emerald-600">
                                            <ShieldCheck className="h-4 w-4" />
                                        </div>
                                        Legal Pages
                                    </h3>
                                    <p className="text-sm text-muted-foreground ml-9">Manage content for legal and informational pages (HTML allowed).</p>
                                </div>
                                <div className="p-6 space-y-6">
                                    {[
                                        { key: 'privacy_policy', label: 'Privacy Policy' },
                                        { key: 'terms_of_service', label: 'Terms of Service' },
                                        { key: 'refund_policy', label: 'Refund Policy' },
                                        { key: 'about_us', label: 'About Us' }
                                    ].map((page) => (
                                        <div key={page.key} className="space-y-2">
                                            <Label htmlFor={page.key}>{page.label}</Label>
                                            <Textarea
                                                id={page.key}
                                                value={data[page.key] || ''}
                                                onChange={(e) => setData(page.key, e.target.value)}
                                                className="bg-background/50 min-h-[150px] font-mono text-sm"
                                                placeholder={`HTML content for ${page.label}...`}
                                            />
                                        </div>
                                    ))}
                                </div>
                            </GlassCard>
                        </TabsContent>

                        {/* Frontend CMS Tab */}
                        <TabsContent value="frontend">
                            <div className="space-y-6">
                                {/* CMS Management Shortcuts */}
                                <div className="grid gap-4 md:grid-cols-3">
                                    {[
                                        { title: 'Features', icon: Layout, url: '/super-admin/cms/features', desc: 'Add/Edit homepage features', color: 'text-cyan-500' },
                                        { title: 'FAQs', icon: HelpCircle, url: '/super-admin/cms/faqs', desc: 'Update questions & answers', color: 'text-blue-500' },
                                        { title: 'Testimonials', icon: Users, url: '/super-admin/cms/testimonials', desc: 'Add client testimonials', color: 'text-pink-500' },
                                    ].map((item) => (
                                        <GlassCard 
                                            key={item.title} 
                                            className="cursor-pointer hover:border-primary/50 hover:bg-white/70 dark:hover:bg-black/50 transition-all group py-6" 
                                            onClick={() => router.visit(item.url)}
                                        >
                                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                                <CardTitle className="text-sm font-medium text-muted-foreground group-hover:text-primary transition-colors">{item.title}</CardTitle>
                                                <item.icon className={`h-4 w-4 ${item.color}`} />
                                            </CardHeader>
                                            <CardContent>
                                                <div className="text-2xl font-bold">Manage {item.title}</div>
                                                <p className="text-xs text-muted-foreground">{item.desc}</p>
                                            </CardContent>
                                        </GlassCard>
                                    ))}
                                </div>

                                {/* Hero Section Config */}
                                <GlassCard intensity="low" className="p-0 overflow-hidden">
                                    <div className="border-b bg-muted/40 p-4">
                                        <h3 className="font-semibold flex items-center gap-2">
                                            <div className="p-1.5 rounded-md bg-cyan-500/10 text-cyan-600">
                                                <Monitor className="h-4 w-4" />
                                            </div>
                                            Hero Section
                                        </h3>
                                        <p className="text-sm text-muted-foreground ml-9">Configure the main landing page banner.</p>
                                    </div>
                                    <div className="p-6 space-y-6">
                                        <div className="space-y-2">
                                            <Label htmlFor="hero_title">Hero Title</Label>
                                            <Input
                                                id="hero_title"
                                                value={data.hero_title || ''}
                                                onChange={(e) => setData('hero_title', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="e.g. Smart Society Management"
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="hero_subtitle">Hero Subtitle</Label>
                                            <Textarea
                                                id="hero_subtitle"
                                                value={data.hero_subtitle || ''}
                                                onChange={(e) => setData('hero_subtitle', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="Description text..."
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="hero_badge">Badge Text</Label>
                                            <Input
                                                id="hero_badge"
                                                value={data.hero_badge || ''}
                                                onChange={(e) => setData('hero_badge', e.target.value)}
                                                className="bg-background/50"
                                                placeholder="e.g. Trusted by 500+ Societies"
                                            />
                                        </div>
                                    </div>
                                </GlassCard>

                                {/* Section Visibility */}
                                <GlassCard intensity="low" className="p-0 overflow-hidden">
                                    <div className="border-b bg-muted/40 p-4">
                                        <h3 className="font-semibold flex items-center gap-2">
                                            <div className="p-1.5 rounded-md bg-purple-500/10 text-purple-600">
                                                <MousePointerClick className="h-4 w-4" />
                                            </div>
                                            Section Visibility
                                        </h3>
                                        <p className="text-sm text-muted-foreground ml-9">Toggle sections on the homepage.</p>
                                    </div>
                                    <div className="p-6 space-y-4">
                                        {[
                                            { key: 'enable_features_section', label: 'Features Section', desc: 'Show the grid of features.' },
                                            { key: 'enable_testimonials_section', label: 'Testimonials Section', desc: 'Show client reviews.' },
                                            { key: 'enable_faq_section', label: 'FAQ Section', desc: 'Show frequently asked questions.' },
                                            { key: 'enable_cta_section', label: 'CTA Section', desc: 'Show the bottom call-to-action.' },
                                        ].map((section) => (
                                            <div key={section.key} className="flex items-center justify-between p-4 rounded-xl border border-border/50 bg-background/30 hover:bg-background/50 transition-colors">
                                                <div className="space-y-0.5">
                                                    <Label className="text-base">{section.label}</Label>
                                                    <p className="text-xs text-muted-foreground">{section.desc}</p>
                                                </div>
                                                <Switch
                                                    checked={data[section.key] === '1' || data[section.key] === true}
                                                    onCheckedChange={(checked) => setData(section.key, checked ? '1' : '0')}
                                                    className="data-[state=checked]:bg-primary"
                                                />
                                            </div>
                                        ))}
                                    </div>
                                </GlassCard>
                            </div>
                        </TabsContent>
                    </Tabs>
                </form>
            </div>
        </SuperAdminLayout>
    );
}
