import SuperAdminLayout from '@/layouts/super-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link } from '@inertiajs/react';
import { 
    Building2, 
    Users, 
    DollarSign, 
    Plus,
    ArrowUpRight,
    TrendingUp,
    Clock,
    AlertCircle,
    Sparkles,
    Zap,
    Target,
    Crown,
    Activity
} from 'lucide-react';
import { CardHeader, CardTitle, CardDescription, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Dashboard',
        href: '/super-admin/dashboard',
    },
];

interface DashboardProps {
    stats: {
        total_societies: number;
        active_societies: number;
        pending_fees: number;
        recent_societies: number;
        total_users: number;
        active_users: number;
        monthly_revenue: number;
        societies_growth: number;
        users_growth: number;
        revenue_growth: number;
        pending_fees_growth: number;
    };
    recentSocieties: Array<{
        id: number;
        name: string;
        registration_no: string;
        status: string;
        platform_fee?: { amount: number; status: string } | null;
        created_at: string;
    }>;
    recentUsers: Array<{
        id: number;
        name: string;
        email: string | null;
        phone: string;
        society: { name: string } | null;
        role: { name: string; display_name: string } | null;
        created_at: string;
    }>;
    cron_command: string;
}

export default function Dashboard({
    stats,
    recentSocieties,
    recentUsers,
    cron_command,
}: DashboardProps) {
    const getStatusColor = (status: string) => {
        switch (status.toLowerCase()) {
            case 'active':
                return 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20';
            case 'pending':
                return 'bg-amber-500/10 text-amber-600 dark:text-amber-400 border-amber-500/20';
            case 'suspended':
                return 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20';
            default:
                return 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20';
        }
    };

    const formatGrowth = (growth: number) => {
        if (growth === 0) return null;
        const sign = growth > 0 ? '+' : '';
        return `${sign}${growth.toFixed(1)}%`;
    };

    const statsCards = [
        {
            title: 'Total Societies',
            value: stats.total_societies,
            subtitle: `${stats.active_societies} active societies`,
            icon: Building2,
            change: formatGrowth(stats.societies_growth),
            positive: stats.societies_growth >= 0,
            iconClass: 'text-primary',
            iconBg: 'bg-primary/10',
            progress: stats.total_societies > 0 ? Math.min((stats.active_societies / stats.total_societies) * 100, 100) : 0,
        },
        {
            title: 'Total Users',
            value: stats.total_users,
            subtitle: `${stats.active_users} users online`,
            icon: Users,
            change: formatGrowth(stats.users_growth),
            positive: stats.users_growth >= 0,
            iconClass: 'text-blue-500',
            iconBg: 'bg-blue-500/10',
            progress: stats.total_users > 0 ? Math.min((stats.active_users / stats.total_users) * 100, 100) : 0,
        },
        {
            title: 'Monthly Revenue',
            value: `₹${stats.monthly_revenue.toLocaleString()}`,
            subtitle: 'This month earnings',
            icon: DollarSign,
            change: formatGrowth(stats.revenue_growth),
            positive: stats.revenue_growth >= 0,
            iconClass: 'text-emerald-500',
            iconBg: 'bg-emerald-500/10',
            progress: stats.monthly_revenue > 0 ? 75 : 0,
        },
        {
            title: 'Pending Fees',
            value: stats.pending_fees,
            subtitle: 'Societies with pending',
            icon: AlertCircle,
            change: formatGrowth(stats.pending_fees_growth),
            positive: stats.pending_fees_growth <= 0,
            iconClass: 'text-amber-500',
            iconBg: 'bg-amber-500/10',
            progress: stats.pending_fees > 0 ? Math.min((stats.pending_fees / stats.total_societies) * 100, 100) : 0,
        },
    ];

    return (
        <SuperAdminLayout breadcrumbs={breadcrumbs}>
            <Head title="Super Admin Dashboard" />
            
            <div className="flex h-full flex-1 flex-col gap-8 p-8 overflow-x-hidden">
                <PageHeader 
                    title="Super Admin Dashboard"
                    description="Welcome back! Here's your system overview."
                    actions={
                        <Link href="/super-admin/societies/create">
                            <Button size="sm" className="gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25">
                                <Plus className="h-4 w-4" />
                                Add New Society
                            </Button>
                        </Link>
                    }
                />

                {/* Stats Cards Grid */}
                <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
                    {statsCards.map((stat, index) => (
                        <GlassCard 
                            key={index}
                            className="group relative overflow-hidden transition-all duration-300 hover:-translate-y-1 py-6"
                            intensity="low"
                        >
                            <CardHeader className="relative pb-2">
                                <div className="flex items-start justify-between">
                                    <div className="space-y-1">
                                        <CardTitle className="text-sm font-medium text-muted-foreground">
                                            {stat.title}
                                        </CardTitle>
                                        <div className="text-3xl font-bold tracking-tight">
                                            {stat.value}
                                        </div>
                                    </div>
                                    <div className={`p-2.5 rounded-xl ${stat.iconBg} transition-colors group-hover:scale-110 duration-300`}>
                                        <stat.icon className={`h-5 w-5 ${stat.iconClass}`} />
                                    </div>
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="flex items-center justify-between mb-3">
                                    <p className="text-xs text-muted-foreground">
                                        {stat.subtitle}
                                    </p>
                                    {stat.change && (
                                        <div className={`flex items-center gap-1 text-xs font-semibold px-2 py-0.5 rounded-full ${
                                            stat.positive 
                                                ? 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400' 
                                                : 'bg-rose-500/10 text-rose-600 dark:text-rose-400'
                                        }`}>
                                            <TrendingUp className={`h-3 w-3 ${!stat.positive && 'rotate-180'}`} />
                                            {stat.change}
                                        </div>
                                    )}
                                </div>
                                <div className="h-1.5 w-full bg-slate-100 dark:bg-slate-800 rounded-full overflow-hidden">
                                    <div 
                                        className={`h-full rounded-full transition-all duration-1000 ${stat.iconClass.replace('text-', 'bg-')}`}
                                        style={{ width: `${stat.progress}%` }}
                                    />
                                </div>
                            </CardContent>
                        </GlassCard>
                    ))}
                </div>

                {/* Main Content Grid */}
                <div className="grid gap-6 lg:grid-cols-2">
                    {/* Recent Societies */}
                    <GlassCard className="flex flex-col h-full py-6">
                        <CardHeader className="pb-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <CardTitle className="text-xl flex items-center gap-2">
                                        <Sparkles className="h-5 w-5 text-primary" />
                                        Recent Societies
                                    </CardTitle>
                                    <CardDescription>
                                        Newly registered this week
                                    </CardDescription>
                                </div>
                                <Link href="/super-admin/societies">
                                    <Button variant="ghost" size="sm" className="gap-1 hover:text-primary">
                                        View all
                                        <ArrowUpRight className="h-4 w-4" />
                                    </Button>
                                </Link>
                            </div>
                        </CardHeader>
                        <CardContent className="flex-1">
                            <div className="space-y-4">
                                {recentSocieties.length > 0 ? (
                                    recentSocieties.map((society, idx) => (
                                        <div
                                            key={society.id}
                                            className="group flex items-center gap-4 p-3 rounded-xl hover:bg-slate-50 dark:hover:bg-white/5 border border-transparent hover:border-border/50 transition-all duration-200"
                                        >
                                            <div className="flex h-10 w-10 shrink-0 items-center justify-center rounded-lg bg-primary/10 text-primary">
                                                <Building2 className="h-5 w-5" />
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <p className="font-semibold text-sm truncate group-hover:text-primary transition-colors">
                                                    {society.name}
                                                </p>
                                                <p className="text-xs text-muted-foreground">
                                                    {society.registration_no}
                                                </p>
                                            </div>
                                            <div className="flex items-center gap-3">
                                                {society.platform_fee && (
                                                    <span className="text-sm font-semibold text-emerald-600 dark:text-emerald-400">
                                                        ₹{society.platform_fee.amount}
                                                    </span>
                                                )}
                                                <Badge variant="outline" className={`${getStatusColor(society.status)} border capitalize shadow-none`}>
                                                    {society.status}
                                                </Badge>
                                            </div>
                                        </div>
                                    ))
                                ) : (
                                    <div className="flex flex-col items-center justify-center py-12 text-center text-muted-foreground">
                                        <Building2 className="h-10 w-10 opacity-20 mb-2" />
                                        <p>No societies yet</p>
                                    </div>
                                )}
                            </div>
                        </CardContent>
                    </GlassCard>

                    {/* Recent Users */}
                    <GlassCard className="flex flex-col h-full py-6">
                        <CardHeader className="pb-4">
                            <div className="flex items-center justify-between">
                                <div>
                                    <CardTitle className="text-xl flex items-center gap-2">
                                        <Zap className="h-5 w-5 text-blue-500" />
                                        Recent Users
                                    </CardTitle>
                                    <CardDescription>
                                        Recently joined members
                                    </CardDescription>
                                </div>
                                <Link href="/super-admin/users">
                                    <Button variant="ghost" size="sm" className="gap-1 hover:text-blue-500">
                                        View all
                                        <ArrowUpRight className="h-4 w-4" />
                                    </Button>
                                </Link>
                            </div>
                        </CardHeader>
                        <CardContent className="flex-1">
                            <div className="space-y-4">
                                {recentUsers.length > 0 ? (
                                    recentUsers.map((user, idx) => (
                                        <div
                                            key={user.id}
                                            className="group flex items-center gap-4 p-3 rounded-xl hover:bg-slate-50 dark:hover:bg-white/5 border border-transparent hover:border-border/50 transition-all duration-200"
                                        >
                                            <div className="flex h-10 w-10 shrink-0 items-center justify-center rounded-lg bg-blue-500/10 text-blue-600">
                                                <Users className="h-5 w-5" />
                                            </div>
                                            <div className="flex-1 min-w-0">
                                                <p className="font-semibold text-sm truncate group-hover:text-blue-600 transition-colors">
                                                    {user.name}
                                                </p>
                                                <p className="text-xs text-muted-foreground">
                                                    {user.role?.display_name || user.role?.name || 'No role'}
                                                </p>
                                            </div>
                                            <div className="text-right">
                                                <p className="text-xs font-medium text-foreground/80">
                                                    {user.society?.name || 'No Society'}
                                                </p>
                                                <p className="text-[10px] text-muted-foreground flex items-center gap-1 justify-end mt-0.5">
                                                    <Clock className="h-3 w-3" />
                                                    {user.phone}
                                                </p>
                                            </div>
                                        </div>
                                    ))
                                ) : (
                                    <div className="flex flex-col items-center justify-center py-12 text-center text-muted-foreground">
                                        <Users className="h-10 w-10 opacity-20 mb-2" />
                                        <p>No users yet</p>
                                    </div>
                                )}
                            </div>
                        </CardContent>
                    </GlassCard>
                </div>

                {/* Quick Actions */}
                <GlassCard className="p-1">
                    <div className="grid md:grid-cols-4 divide-y md:divide-y-0 md:divide-x border-white/10">
                        {[
                            { 
                                label: 'Add Society', 
                                icon: Plus, 
                                href: '/super-admin/societies/create',
                                color: 'text-primary',
                                bg: 'bg-primary/10',
                                desc: 'Register new tenant'
                            },
                            { 
                                label: 'Manage Plans', 
                                icon: DollarSign, 
                                href: '/super-admin/subscription-plans',
                                color: 'text-blue-500',
                                bg: 'bg-blue-500/10',
                                desc: 'Update pricing tiers'
                            },
                            { 
                                label: 'View Societies', 
                                icon: Building2, 
                                href: '/super-admin/societies',
                                color: 'text-emerald-500',
                                bg: 'bg-emerald-500/10',
                                desc: 'Manage all tenants'
                            },
                            { 
                                label: 'Manage Users', 
                                icon: Users, 
                                href: '/super-admin/users',
                                color: 'text-amber-500',
                                bg: 'bg-amber-500/10',
                                desc: 'User administration'
                            },
                        ].map((action, idx) => (
                            <Link key={idx} href={action.href} className="group relative p-6 hover:bg-slate-50/50 dark:hover:bg-white/5 transition-colors first:rounded-l-2xl last:rounded-r-2xl">
                                <div className="flex items-center gap-4">
                                    <div className={`p-3 rounded-xl ${action.bg} ${action.color} group-hover:scale-110 transition-transform duration-300`}>
                                        <action.icon className="h-6 w-6" />
                                    </div>
                                    <div>
                                        <p className="font-semibold text-sm group-hover:text-primary transition-colors">
                                            {action.label}
                                        </p>
                                        <p className="text-xs text-muted-foreground mt-0.5">
                                            {action.desc}
                                        </p>
                                    </div>
                                </div>
                            </Link>
                        ))}
                    </div>
                </GlassCard>

                {/* Server Configuration */}
                <GlassCard className="py-6">
                    <CardHeader>
                        <CardTitle className="text-lg flex items-center gap-2">
                            <Activity className="h-5 w-5 text-emerald-500" />
                            Server Configuration
                        </CardTitle>
                        <CardDescription>
                            Configure this cron job on your server to run scheduled tasks.
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <div className="bg-slate-950 dark:bg-black p-4 rounded-xl flex items-center justify-between group relative overflow-hidden ring-1 ring-white/10">
                            <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-transparent opacity-0 group-hover:opacity-100 transition-opacity" />
                            <code className="text-emerald-400 font-mono text-sm break-all relative z-10">
                                {cron_command}
                            </code>
                            <Button 
                                variant="outline" 
                                size="sm" 
                                className="ml-4 shrink-0 border-white/20 hover:bg-emerald-500/10 hover:text-emerald-400 hover:border-emerald-500/50 relative z-10"
                                onClick={() => {
                                    navigator.clipboard.writeText(cron_command);
                                }}
                            >
                                Copy Command
                            </Button>
                        </div>
                    </CardContent>
                </GlassCard>
            </div>
        </SuperAdminLayout>
    );
}
