import SuperAdminLayout from '@/layouts/super-admin-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Plus, Edit2, Trash2, LayoutGrid } from 'lucide-react';
import * as Icons from 'lucide-react';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { Badge } from '@/components/ui/badge';

interface Feature {
    id: number;
    title: string;
    description: string;
    icon: string;
    is_active: boolean;
    sort_order: number;
}

export default function Index({ features }: { features: Feature[] }) {
    
    // Helper to render dynamic icon
    const renderIcon = (iconName: string, className?: string) => {
        // @ts-ignore
        const IconComponent = Icons[iconName] || Icons.Sparkles;
        return <IconComponent className={className} />;
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this feature?')) {
            router.delete(`/super-admin/cms/features/${id}`);
        }
    };

    return (
        <SuperAdminLayout breadcrumbs={[
            { title: 'CMS', href: '#' },
            { title: 'Features', href: '/super-admin/cms/features' },
        ]}>
            <Head title="Manage Features" />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-7xl mx-auto w-full">
                <PageHeader 
                    title="Features Overlay"
                    description="Manage the features displayed on the homepage"
                    actions={
                        <Link href="/super-admin/cms/features/create">
                            <Button size="sm" className="gap-2 bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25">
                                <Plus className="h-4 w-4" />
                                Add Feature
                            </Button>
                        </Link>
                    }
                />

                {/* Grid */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {features.length > 0 ? (
                        features.map((feature) => (
                            <GlassCard 
                                key={feature.id} 
                                className="group relative overflow-hidden flex flex-col h-full hover:border-primary/50 transition-colors"
                            >
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-transparent via-primary/50 to-transparent opacity-0 group-hover:opacity-100 transition-opacity" />
                                
                                <div className="flex justify-between items-start mb-4">
                                    <div className="p-3 rounded-xl bg-primary/10 text-primary group-hover:scale-110 transition-transform duration-300">
                                        {renderIcon(feature.icon, "h-6 w-6")}
                                    </div>
                                    <div className="flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                        <Link href={`/super-admin/cms/features/${feature.id}/edit`}>
                                            <Button size="icon" variant="ghost" className="h-8 w-8 hover:bg-primary/10 hover:text-primary">
                                                <Edit2 className="h-4 w-4" />
                                            </Button>
                                        </Link>
                                        <Button size="icon" variant="ghost" className="h-8 w-8 hover:bg-red-500/10 hover:text-red-500" onClick={() => handleDelete(feature.id)}>
                                            <Trash2 className="h-4 w-4" />
                                        </Button>
                                    </div>
                                </div>
                                
                                <h3 className="text-xl font-bold mb-2">{feature.title}</h3>
                                <p className="text-sm text-muted-foreground line-clamp-2 mb-4 flex-1">
                                    {feature.description}
                                </p>
                                
                                <div className="flex items-center justify-between text-xs text-muted-foreground mt-4 pt-4 border-t border-border/50">
                                    <Badge variant={feature.is_active ? 'default' : 'secondary'} className={feature.is_active ? 'bg-emerald-500/15 text-emerald-600 hover:bg-emerald-500/25 border-emerald-500/20' : ''}>
                                        {feature.is_active ? 'Active' : 'Inactive'}
                                    </Badge>
                                    <span className="font-mono">Order: {feature.sort_order}</span>
                                </div>
                            </GlassCard>
                        ))
                    ) : (
                        <div className="col-span-full flex flex-col items-center justify-center p-12 text-center text-muted-foreground">
                            <div className="p-4 rounded-full bg-muted/50 mb-4">
                                <LayoutGrid className="h-8 w-8 opacity-50" />
                            </div>
                            <h3 className="text-lg font-semibold text-foreground">No features found</h3>
                            <p className="mb-4">Get started by creating your first feature.</p>
                            <Link href="/super-admin/cms/features/create">
                                <Button variant="outline">Create Feature</Button>
                            </Link>
                        </div>
                    )}
                </div>
            </div>
        </SuperAdminLayout>
    );
}
