import SuperAdminLayout from '@/layouts/super-admin-layout';
import { Head, Link, useForm } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { ArrowLeft, Save, Sparkles, SortAsc, AlignLeft, Type, ToggleLeft } from 'lucide-react';
import { FormEventHandler } from 'react';
import { Switch } from '@/components/ui/switch';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import InputError from '@/components/input-error';

interface Feature {
    id: number;
    title: string;
    description: string;
    icon: string;
    sort_order: number;
    is_active: boolean | number;
}

export default function Edit({ feature }: { feature: Feature }) {
    const { data, setData, put, processing, errors } = useForm({
        title: feature.title,
        description: feature.description,
        icon: feature.icon,
        sort_order: feature.sort_order,
        is_active: Boolean(feature.is_active),
    });

    const submit: FormEventHandler = (e) => {
        e.preventDefault();
        put(`/super-admin/cms/features/${feature.id}`);
    };

    return (
        <SuperAdminLayout breadcrumbs={[
            { title: 'CMS', href: '#' },
            { title: 'Features', href: '/super-admin/cms/features' },
            { title: 'Edit', href: `/super-admin/cms/features/${feature.id}/edit` },
        ]}>
            <Head title="Edit Feature" />

            <div className="flex h-full flex-1 flex-col gap-8 p-8 max-w-4xl mx-auto w-full">
                <PageHeader 
                    title="Edit Feature"
                    description={`Update details for ${feature.title}`}
                    actions={
                        <Link href="/super-admin/cms/features">
                            <Button variant="outline" size="sm" className="gap-2">
                                <ArrowLeft className="h-4 w-4" />
                                Back
                            </Button>
                        </Link>
                    }
                />

                <form onSubmit={submit} className="space-y-6">
                    <GlassCard className="p-0 overflow-hidden">
                        <div className="border-b bg-muted/40 p-4">
                            <h3 className="font-semibold flex items-center gap-2">
                                <div className="p-1.5 rounded-md bg-primary/10 text-primary">
                                    <Sparkles className="h-4 w-4" />
                                </div>
                                Feature Details
                            </h3>
                        </div>
                        <div className="p-6 space-y-6">
                            <div className="grid gap-6 md:grid-cols-2">
                                <div className="space-y-2">
                                    <Label htmlFor="title">Title</Label>
                                    <div className="relative">
                                        <Type className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                        <Input
                                            id="title"
                                            value={data.title}
                                            onChange={(e) => setData('title', e.target.value)}
                                            placeholder="e.g. Automated Billing"
                                            className="pl-9 bg-background/50"
                                        />
                                    </div>
                                    <InputError message={errors.title} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="icon">Icon Name (Lucide)</Label>
                                    <div className="relative">
                                        <Sparkles className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                        <Input
                                            id="icon"
                                            value={data.icon}
                                            onChange={(e) => setData('icon', e.target.value)}
                                            placeholder="e.g. Shield, Users, Zap"
                                            className="pl-9 bg-background/50"
                                        />
                                    </div>
                                    <p className="text-xs text-muted-foreground">Case-sensitive Lucide icon name.</p>
                                    <InputError message={errors.icon} />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="description">Description</Label>
                                <div className="relative">
                                    <AlignLeft className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                                    <Textarea
                                        id="description"
                                        value={data.description}
                                        onChange={(e) => setData('description', e.target.value)}
                                        placeholder="Brief description of the feature..."
                                        className="pl-9 bg-background/50 min-h-[100px]"
                                    />
                                </div>
                                <InputError message={errors.description} />
                            </div>

                            <div className="grid gap-6 md:grid-cols-2">
                                <div className="space-y-2">
                                    <Label htmlFor="sort_order">Sort Order</Label>
                                    <div className="relative">
                                        <SortAsc className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                        <Input
                                            id="sort_order"
                                            type="number"
                                            value={data.sort_order}
                                            onChange={(e) => setData('sort_order', parseInt(e.target.value))}
                                            className="pl-9 bg-background/50"
                                        />
                                    </div>
                                    <InputError message={errors.sort_order} />
                                </div>

                                <div className="flex items-center justify-between p-4 rounded-xl border border-border/50 bg-background/50 self-end">
                                    <div className="space-y-0.5">
                                        <Label className="text-base flex items-center gap-2">
                                            <ToggleLeft className="h-4 w-4 text-muted-foreground" />
                                            Active Status
                                        </Label>
                                        <p className="text-xs text-muted-foreground">Show this feature on the homepage.</p>
                                    </div>
                                    <Switch
                                        checked={data.is_active}
                                        onCheckedChange={(checked) => setData('is_active', checked)}
                                    />
                                </div>
                            </div>
                        </div>
                        <div className="border-t bg-muted/40 p-4 flex justify-end">
                            <Button 
                                type="submit" 
                                disabled={processing} 
                                className="bg-gradient-to-r from-primary to-primary/80 hover:from-primary/90 hover:to-primary shadow-lg shadow-primary/25"
                            >
                                <Save className="mr-2 h-4 w-4" />
                                Update Feature
                            </Button>
                        </div>
                    </GlassCard>
                </form>
            </div>
        </SuperAdminLayout>
    );
}
