import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Spinner } from '@/components/ui/spinner';
import { useAppearance } from '@/hooks/use-appearance';
import { Head, useForm } from '@inertiajs/react';
import { Shield, Lock, Mail, Moon, Sun, Settings, Database, Users, BarChart3, Zap, Crown, Copy, Check } from 'lucide-react';
import { useState } from 'react';

interface LoginProps {
    errors?: {
        email?: string;
        password?: string;
    };
    demoMode?: boolean;
    demoCredentials?: {
        email: string;
        password: string;
    } | null;
}

export default function SuperAdminLogin({ errors, demoMode, demoCredentials }: LoginProps) {
    const { appearance, updateAppearance } = useAppearance();
    const isDark = appearance === 'dark';
    const [copied, setCopied] = useState<'email' | 'password' | 'all' | null>(null);
    const { data, setData, post, processing } = useForm({
        email: '',
        password: '',
        remember: false,
    });

    const handleCopyAndFill = () => {
        if (demoCredentials) {
            setData('email', demoCredentials.email);
            setData('password', demoCredentials.password);
            setCopied('all');
            setTimeout(() => setCopied(null), 2000);
        }
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post('/super-admin/login');
    };

    return (
        <>
            <Head title="Super Admin Login" />
            
            <div className="relative flex min-h-screen overflow-hidden">
                {/* Left Side - Super Admin Branding (60%) */}
                <div className="hidden lg:flex lg:w-[60%] relative bg-gradient-to-br from-red-50 via-orange-50 to-amber-50 dark:from-slate-900 dark:via-red-950 dark:to-orange-950">
                    {/* Soft Animated Background Gradient Orbs */}
                    <div className="absolute top-0 left-0 w-96 h-96 bg-red-200/40 dark:bg-red-500/20 rounded-full blur-3xl animate-pulse" />
                    <div className="absolute top-1/4 right-1/4 w-80 h-80 bg-orange-200/40 dark:bg-orange-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }} />
                    <div className="absolute bottom-1/4 left-1/3 w-72 h-72 bg-amber-200/40 dark:bg-amber-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                    <div className="absolute bottom-0 right-0 w-96 h-96 bg-yellow-200/40 dark:bg-yellow-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '3s' }} />
                    
                    {/* Subtle Overlay */}
                    <div className="absolute inset-0 bg-white/10 dark:bg-black/10" />
                    
                    {/* Floating Icons Animation */}
                    <div className="absolute inset-0 overflow-hidden pointer-events-none">
                        <div className="absolute top-20 left-10 animate-float" style={{ animationDelay: '0s' }}>
                            <div className="w-16 h-16 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-12 shadow-lg">
                                <Shield className="w-8 h-8 text-red-600 dark:text-red-400" />
                            </div>
                        </div>
                        <div className="absolute top-40 right-20 animate-float" style={{ animationDelay: '1s' }}>
                            <div className="w-20 h-20 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-12 shadow-lg">
                                <Settings className="w-10 h-10 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                        <div className="absolute bottom-40 left-20 animate-float" style={{ animationDelay: '2s' }}>
                            <div className="w-14 h-14 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-6 shadow-lg">
                                <Database className="w-7 h-7 text-amber-600 dark:text-amber-400" />
                            </div>
                        </div>
                        <div className="absolute bottom-20 right-32 animate-float" style={{ animationDelay: '3s' }}>
                            <div className="w-18 h-18 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-6 shadow-lg">
                                <BarChart3 className="w-9 h-9 text-yellow-600 dark:text-yellow-400" />
                            </div>
                        </div>
                        <div className="absolute top-1/2 left-1/4 animate-float" style={{ animationDelay: '1.5s' }}>
                            <div className="w-12 h-12 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center rotate-45 shadow-lg">
                                <Crown className="w-6 h-6 text-red-600 dark:text-red-400" />
                            </div>
                        </div>
                        <div className="absolute top-1/3 right-1/3 animate-float" style={{ animationDelay: '2.5s' }}>
                            <div className="w-16 h-16 rounded-2xl bg-white/60 dark:bg-white/10 backdrop-blur-sm flex items-center justify-center -rotate-12 shadow-lg">
                                <Users className="w-8 h-8 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                    </div>
                    
                    {/* Content */}
                    <div className="relative z-10 flex flex-col justify-center px-12 xl:px-16 w-full">
                        <div className="max-w-xl">
                            {/* Super Admin Badge */}
                            <div className="mb-6 inline-flex items-center gap-2 px-4 py-2 rounded-full bg-gradient-to-r from-red-500 to-orange-500 text-white shadow-lg animate-fade-in-up">
                                <Crown className="w-4 h-4" />
                                <span className="text-sm font-semibold">Super Admin Access</span>
                            </div>

                            {/* Logo with Animation */}
                            <div className="mb-10 flex items-center gap-4 animate-fade-in-up" style={{ animationDelay: '0.1s' }}>
                                <div className="flex h-16 w-16 items-center justify-center rounded-2xl bg-gradient-to-br from-red-500 to-orange-600 shadow-2xl animate-bounce-slow">
                                    <Shield className="h-9 w-9 text-white" />
                                </div>
                                <div>
                                    <span className="text-3xl font-bold text-gray-800 dark:text-white">
                                        Societify
                                    </span>
                                    <p className="text-gray-600 dark:text-gray-300 text-sm font-medium">System Administration</p>
                                </div>
                            </div>

                            <div className="animate-fade-in-up" style={{ animationDelay: '0.2s' }}>
                                <h1 className="text-5xl xl:text-6xl font-bold mb-6 leading-tight text-gray-900 dark:text-white">
                                    Master Control
                                    <br />
                                    <span className="bg-gradient-to-r from-red-600 via-orange-600 to-amber-600 bg-clip-text text-transparent">
                                        Dashboard
                                    </span>
                                </h1>
                            </div>
                            
                            <p className="text-xl text-gray-700 dark:text-gray-300 mb-12 leading-relaxed animate-fade-in-up" style={{ animationDelay: '0.4s' }}>
                                Administrator portal for complete system control. 
                                Manage societies, users, permissions, and global settings with full authority.
                            </p>

                            {/* Animated Feature Cards */}
                            <div className="grid grid-cols-2 gap-4 animate-fade-in-up" style={{ animationDelay: '0.6s' }}>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-red-500 to-red-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Shield className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Full Access</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">System-wide control</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-orange-500 to-orange-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Settings className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Configuration</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">Global settings</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-amber-500 to-amber-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <Users className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">User Management</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">All users & roles</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="group px-6 py-4 rounded-2xl bg-white/70 dark:bg-white/10 backdrop-blur-md border border-gray-200 dark:border-white/20 hover:bg-white/90 dark:hover:bg-white/20 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-yellow-500 to-yellow-600 flex items-center justify-center group-hover:rotate-12 transition-transform">
                                            <BarChart3 className="h-5 w-5 text-white" />
                                        </div>
                                        <div>
                                            <h3 className="text-gray-800 dark:text-white font-semibold text-sm">Analytics</h3>
                                            <p className="text-gray-600 dark:text-gray-400 text-xs">System insights</p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Warning Badge */}
                            <div className="mt-8 p-4 rounded-xl bg-white/50 dark:bg-white/5 backdrop-blur-sm border border-orange-200 dark:border-orange-900/30 animate-fade-in-up" style={{ animationDelay: '0.8s' }}>
                                <p className="text-sm text-gray-700 dark:text-gray-300 flex items-center gap-2">
                                    <Zap className="w-4 h-4 text-orange-600 dark:text-orange-400" />
                                    <span className="font-medium">Privileged Access:</span> This portal is for authorized system administrators only.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Right Side - Login Form (40%) */}
                <div className="flex-1 lg:w-[40%] flex items-center justify-center p-6 lg:p-12 bg-background relative">
                    {/* Theme Toggle - Pill Shape with System Option */}
                    <div className="absolute top-6 right-6 z-20">
                        <div className="inline-flex items-center rounded-full bg-muted p-1 shadow-sm border border-border/50">
                            <button
                                onClick={() => updateAppearance('light')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'light'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Sun className="h-4 w-4" />
                                <span className="hidden sm:inline">Light</span>
                            </button>
                            <button
                                onClick={() => updateAppearance('dark')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'dark'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Moon className="h-4 w-4" />
                                <span className="hidden sm:inline">Dark</span>
                            </button>
                            <button
                                onClick={() => updateAppearance('system')}
                                className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                    appearance === 'system'
                                        ? 'bg-background text-foreground shadow-sm'
                                        : 'text-muted-foreground hover:text-foreground'
                                }`}
                            >
                                <Shield className="h-4 w-4" />
                                <span className="hidden sm:inline">Auto</span>
                            </button>
                        </div>
                    </div>

                    <div className="w-full max-w-md">
                        {/* Mobile Logo */}
                        <div className="lg:hidden mb-8 flex items-center justify-center gap-3">
                            <div className="flex h-12 w-12 items-center justify-center rounded-xl bg-gradient-to-br from-red-500 to-orange-600">
                                <Shield className="h-7 w-7 text-white" />
                            </div>
                            <span className="text-2xl font-bold bg-gradient-to-r from-red-600 to-orange-600 bg-clip-text text-transparent">
                                Societify
                            </span>
                        </div>

                        {/* Login Card */}
                        <div className="rounded-2xl border border-border/50 bg-card/50 backdrop-blur-sm p-8 shadow-xl">
                            <div className="mb-8">
                                <div className="flex items-center gap-2 mb-3">
                                    <Crown className="w-6 h-6 text-red-600 dark:text-red-400" />
                                    <h2 className="text-2xl font-bold text-foreground">
                                        Super Admin
                                    </h2>
                                </div>
                                <p className="text-muted-foreground">
                                    Enter your credentials to access the admin panel
                                </p>
                            </div>

                            {/* Demo Credentials Box */}
                            {demoMode && demoCredentials && (
                                <div className="mb-6 p-4 rounded-xl bg-orange-500/10 border border-orange-500/30 space-y-3">
                                    <div className="flex items-center justify-between">
                                        <span className="text-sm font-semibold text-orange-700 dark:text-orange-400 flex items-center gap-2">
                                            <Zap className="h-4 w-4" />
                                            Demo Credentials
                                        </span>
                                        <Button
                                            type="button"
                                            variant="outline"
                                            size="sm"
                                            onClick={handleCopyAndFill}
                                            className="h-7 text-xs gap-1 border-orange-500/30 hover:bg-orange-500/10"
                                        >
                                            {copied === 'all' ? (
                                                <><Check className="h-3 w-3" /> Filled!</>
                                            ) : (
                                                <><Copy className="h-3 w-3" /> Use Demo</>  
                                            )}
                                        </Button>
                                    </div>
                                    <div className="space-y-1 text-sm">
                                        <div className="flex items-center gap-2">
                                            <span className="text-muted-foreground w-16">Email:</span>
                                            <code className="bg-background/50 px-2 py-0.5 rounded text-orange-700 dark:text-orange-300 font-mono text-xs">
                                                {demoCredentials.email}
                                            </code>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <span className="text-muted-foreground w-16">Password:</span>
                                            <code className="bg-background/50 px-2 py-0.5 rounded text-orange-700 dark:text-orange-300 font-mono text-xs">
                                                {demoCredentials.password}
                                            </code>
                                        </div>
                                    </div>
                                </div>
                            )}

                            <form onSubmit={handleSubmit} className="space-y-6">
                                {/* Email Field */}
                                <div className="space-y-2">
                                    <Label htmlFor="email" className="text-sm font-medium">
                                        Email address
                                    </Label>
                                    <div className="relative">
                                        <Mail className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                        <Input
                                            id="email"
                                            type="email"
                                            name="email"
                                            value={data.email}
                                            onChange={(e) => setData('email', e.target.value)}
                                            required
                                            autoFocus
                                            autoComplete="email"
                                            placeholder="admin@societify.com"
                                            className="pl-10 h-11"
                                        />
                                    </div>
                                    <InputError message={errors?.email} />
                                </div>

                                {/* Password Field */}
                                <div className="space-y-2">
                                    <Label htmlFor="password" className="text-sm font-medium">
                                        Password
                                    </Label>
                                    <div className="relative">
                                        <Lock className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                        <Input
                                            id="password"
                                            type="password"
                                            name="password"
                                            value={data.password}
                                            onChange={(e) => setData('password', e.target.value)}
                                            required
                                            autoComplete="current-password"
                                            placeholder="••••••••"
                                            className="pl-10 h-11"
                                        />
                                    </div>
                                    <InputError message={errors?.password} />
                                </div>

                                {/* Remember Me */}
                                <div className="flex items-center space-x-2">
                                    <Checkbox
                                        id="remember"
                                        name="remember"
                                        checked={data.remember}
                                        onCheckedChange={(checked) => setData('remember', checked as boolean)}
                                    />
                                    <Label 
                                        htmlFor="remember" 
                                        className="text-sm font-medium cursor-pointer select-none"
                                    >
                                        Remember me for 30 days
                                    </Label>
                                </div>

                                {/* Submit Button - Red/Orange for Super Admin */}
                                <Button
                                    type="submit"
                                    className="w-full h-11 text-base font-medium bg-gradient-to-r from-red-600 to-orange-600 hover:from-red-700 hover:to-orange-700 text-white shadow-lg hover:shadow-xl transition-all"
                                    disabled={processing}
                                >
                                    {processing && <Spinner />}
                                    {processing ? 'Signing in...' : 'Access Admin Panel'}
                                </Button>
                            </form>

                            {/* Footer */}
                            <div className="mt-8 pt-6 border-t border-border/50 text-center text-xs text-muted-foreground">
                                🔒 Enterprise-grade security • Administrator access only
                            </div>
                        </div>

                        {/* Support Text */}
                        <p className="mt-6 text-center text-sm text-muted-foreground">
                            Unauthorized access attempts are logged and monitored
                        </p>
                    </div>
                </div>
            </div>

            <style>{`
                @keyframes float {
                    0%, 100% {
                        transform: translateY(0px);
                    }
                    50% {
                        transform: translateY(-20px);
                    }
                }
                
                @keyframes bounce-slow {
                    0%, 100% {
                        transform: translateY(0);
                    }
                    50% {
                        transform: translateY(-10px);
                    }
                }
                
                @keyframes fade-in-up {
                    from {
                        opacity: 0;
                        transform: translateY(20px);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0);
                    }
                }
                
                .animate-float {
                    animation: float 6s ease-in-out infinite;
                }
                
                .animate-bounce-slow {
                    animation: bounce-slow 3s ease-in-out infinite;
                }
                
                .animate-fade-in-up {
                    animation: fade-in-up 0.8s ease-out forwards;
                    opacity: 0;
                }
            `}</style>
        </>
    );
}
