import { useState, useEffect } from 'react';
import { Head, router, usePage } from '@inertiajs/react';
import { CreditCard, IndianRupee, Loader2, CheckCircle2, Calendar, ArrowLeft, Clock, AlertTriangle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import axios from 'axios';

interface SubscriptionRenewProps {
    subscription: {
        plan_name: string;
        status: string;
        billing_cycle: string;
        end_date: string | null;
        monthly_price: number;
        annual_price: number;
    } | null;
    paymentRequired: {
        status: string;
        message: string;
    } | null;
}

declare global {
    interface Window {
        Razorpay: any;
    }
}

export default function SubscriptionRenew({ subscription, paymentRequired }: SubscriptionRenewProps) {
    const { auth } = usePage().props as any;
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [paymentSuccess, setPaymentSuccess] = useState(false);

    // Load Razorpay script
    useEffect(() => {
        const script = document.createElement('script');
        script.src = 'https://checkout.razorpay.com/v1/checkout.js';
        script.async = true;
        document.body.appendChild(script);
        
        return () => {
            if (document.body.contains(script)) {
                document.body.removeChild(script);
            }
        };
    }, []);

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    const getAmount = () => {
        if (!subscription) return 0;
        switch (subscription.billing_cycle) {
            case 'annual':
                return subscription.annual_price;
            case 'quarterly':
                return subscription.monthly_price * 3;
            default:
                return subscription.monthly_price;
        }
    };

    const handlePayNow = async () => {
        setIsLoading(true);
        setError(null);

        try {
            const response = await axios.post('/subscription/create-payment-order');

            if (!response.data.success) {
                setError(response.data.message || 'Failed to create payment order');
                setIsLoading(false);
                return;
            }

            const { order_id, amount, key } = response.data;

            const options = {
                key: key,
                amount: amount * 100,
                currency: 'INR',
                name: 'Societify',
                description: `${subscription?.plan_name || 'Subscription'} - ${paymentRequired?.status === 'active' ? 'Renewal' : 'Activation'}`,
                order_id: order_id,
                handler: async function (paymentResponse: any) {
                    try {
                        const verifyResponse = await axios.post('/subscription/verify-payment', {
                            razorpay_payment_id: paymentResponse.razorpay_payment_id,
                            razorpay_order_id: paymentResponse.razorpay_order_id,
                            razorpay_signature: paymentResponse.razorpay_signature,
                        });

                        if (verifyResponse.data.success) {
                            setPaymentSuccess(true);
                            setTimeout(() => {
                                router.visit('/societies/' + auth.user?.society_id + '/dashboard');
                            }, 2000);
                        } else {
                            setError(verifyResponse.data.message || 'Payment verification failed');
                        }
                    } catch (err: any) {
                        setError(err.response?.data?.message || 'Payment verification failed');
                    }
                    setIsLoading(false);
                },
                prefill: {
                    name: auth?.user?.name || '',
                    email: auth?.user?.email || '',
                    contact: auth?.user?.phone || '',
                },
                theme: {
                    color: '#8b5cf6',
                },
                modal: {
                    ondismiss: function () {
                        setIsLoading(false);
                        setError('Payment was cancelled. Please try again.');
                    },
                },
            };

            const razorpay = new window.Razorpay(options);
            razorpay.open();
        } catch (err: any) {
            console.error('Payment error:', err);
            setError(err.response?.data?.message || 'An error occurred while initiating payment');
            setIsLoading(false);
        }
    };

    if (paymentSuccess) {
        return (
            <>
                <Head title="Payment Successful" />
                <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-900 dark:to-gray-800 flex items-center justify-center p-4">
                    <div className="max-w-md w-full bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden">
                        <div className="bg-gradient-to-r from-green-500 to-emerald-500 p-6 text-white">
                            <div className="flex items-center gap-4">
                                <div className="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                                    <CheckCircle2 className="w-8 h-8" />
                                </div>
                                <div>
                                    <h1 className="text-2xl font-bold">Payment Successful!</h1>
                                    <p className="text-white/80 text-sm">Your subscription is now active</p>
                                </div>
                            </div>
                        </div>
                        <div className="p-8 text-center">
                            <Loader2 className="w-8 h-8 animate-spin mx-auto text-green-500 mb-4" />
                            <p className="text-gray-600 dark:text-gray-400">Redirecting to your dashboard...</p>
                        </div>
                    </div>
                </div>
            </>
        );
    }

    const isExpired = paymentRequired?.status === 'expired';
    const isTrial = paymentRequired?.status === 'trial';
    const isPending = paymentRequired?.status === 'pending_payment';

    return (
        <>
            <Head title="Renew Subscription" />
            <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-900 dark:to-gray-800 flex items-center justify-center p-4">
                <div className="max-w-lg w-full">
                    {/* Back Button */}
                    <button
                        onClick={() => router.back()}
                        className="mb-6 flex items-center gap-2 text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-white transition-colors"
                    >
                        <ArrowLeft className="w-4 h-4" />
                        Back to Dashboard
                    </button>

                    {/* Main Card */}
                    <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden">
                        {/* Header */}
                        <div className={`p-6 text-white ${
                            isExpired ? 'bg-gradient-to-r from-red-500 to-rose-600' :
                            isTrial || isPending ? 'bg-gradient-to-r from-amber-500 to-orange-500' :
                            'bg-gradient-to-r from-cyan-500 to-indigo-600'
                        }`}>
                            <div className="flex items-center gap-4">
                                <div className="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                                    {isExpired ? <Clock className="w-8 h-8" /> : 
                                     isTrial || isPending ? <AlertTriangle className="w-8 h-8" /> :
                                     <CreditCard className="w-8 h-8" />}
                                </div>
                                <div>
                                    <h1 className="text-2xl font-bold">
                                        {isExpired ? 'Subscription Expired' :
                                         isTrial ? 'Trial Ended' :
                                         isPending ? 'Payment Required' :
                                         'Renew Subscription'}
                                    </h1>
                                    <p className="text-white/80 text-sm">{paymentRequired?.message}</p>
                                    {(paymentRequired?.status === 'expired' || paymentRequired?.status === 'expired_grace' || paymentRequired?.status === 'trial' || paymentRequired?.status === 'pending_payment') && (
                                        <p className="text-white/60 text-xs mt-1">
                                            💡 Contact your super admin for subscription renewal if you don't have payment access.
                                        </p>
                                    )}
                                </div>
                            </div>
                        </div>

                        {/* Content */}
                        <div className="p-6 space-y-6">
                            {/* Plan Details */}
                            {subscription && (
                                <div className="bg-gray-50 dark:bg-gray-900 rounded-xl p-4 space-y-3">
                                    <h3 className="font-semibold text-gray-900 dark:text-white">Plan Details</h3>
                                    <div className="space-y-2 text-sm">
                                        <div className="flex justify-between">
                                            <span className="text-gray-500">Plan</span>
                                            <span className="font-medium text-gray-900 dark:text-white">{subscription.plan_name}</span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-gray-500">Billing Cycle</span>
                                            <span className="font-medium text-gray-900 dark:text-white capitalize">{subscription.billing_cycle}</span>
                                        </div>
                                        {subscription.end_date && (
                                            <div className="flex justify-between">
                                                <span className="text-gray-500">{isExpired || isTrial ? 'Expired On' : 'Expires On'}</span>
                                                <span className="font-medium text-gray-900 dark:text-white flex items-center gap-1">
                                                    <Calendar className="w-3 h-3" />
                                                    {new Date(subscription.end_date).toLocaleDateString('en-IN', { 
                                                        day: 'numeric', 
                                                        month: 'short', 
                                                        year: 'numeric' 
                                                    })}
                                                </span>
                                            </div>
                                        )}
                                        <div className="border-t border-gray-200 dark:border-gray-700 pt-2 mt-2">
                                            <div className="flex justify-between items-center">
                                                <span className="font-semibold text-gray-900 dark:text-white">Amount</span>
                                                <span className="text-xl font-bold text-cyan-600">{formatCurrency(getAmount())}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* Error Message */}
                            {error && (
                                <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-xl p-4">
                                    <p className="text-red-700 dark:text-red-300 text-sm">{error}</p>
                                </div>
                            )}

                            {/* Pay Now Button */}
                            <Button 
                                onClick={handlePayNow}
                                disabled={isLoading || !subscription}
                                className="w-full bg-gradient-to-r from-cyan-500 to-indigo-600 hover:from-cyan-600 hover:to-indigo-700 text-white py-6 text-lg rounded-xl shadow-lg"
                            >
                                {isLoading ? (
                                    <>
                                        <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                                        Processing...
                                    </>
                                ) : (
                                    <>
                                        <IndianRupee className="w-5 h-5 mr-2" />
                                        {isExpired || isTrial || isPending ? 'Pay Now' : 'Renew Now'} - {formatCurrency(getAmount())}
                                    </>
                                )}
                            </Button>
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
}
