import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, usePage } from '@inertiajs/react';
import {
    Users,
    Wrench,
    UserCheck,
    Building2,
    UserPlus,
    AlertCircle,
    Clock,
    Home,
} from 'lucide-react';
import { CardHeader, CardTitle, CardDescription, CardContent } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { Badge } from '@/components/ui/badge';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Dashboard',
        href: '/societies/{society}/sub-admin/dashboard',
    },
];

interface DashboardProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    kpis: {
        total_residents: number;
        open_tickets: number;
        today_visitors: number;
        assigned_units: number;
        recent_additions: number;
        pending_actions: number;
    };
    recentActivity: {
        tickets: Array<{
            id: number;
            ticket_no: string;
            subject: string;
            status: string;
            priority: string;
            user_name: string;
            vendor_name: string | null;
            created_at: string;
        }>;
        visitors: Array<{
            id: number;
            name: string;
            type: string;
            status: string;
            user_name: string;
            created_at: string;
        }>;
        residents: Array<{
            id: number;
            name: string;
            phone: string;
            unit_no: string;
            created_at: string;
        }>;
    };
}

export default function Dashboard({
    society,
    kpis,
    recentActivity,
}: DashboardProps) {
    const { auth } = usePage().props as any;
    const user = auth.user;

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'resolved':
            case 'closed':
            case 'checked_out':
                return 'bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 border-emerald-500/20';
            case 'checked_in':
            case 'approved':
                return 'bg-blue-500/10 text-blue-600 dark:text-blue-400 border-blue-500/20';
            case 'pending':
            case 'open':
            case 'assigned':
            case 'in_progress':
                return 'bg-amber-500/10 text-amber-600 dark:text-amber-400 border-amber-500/20';
            case 'rejected':
                return 'bg-rose-500/10 text-rose-600 dark:text-rose-400 border-rose-500/20';
            default:
                return 'bg-slate-500/10 text-slate-600 dark:text-slate-400 border-slate-500/20';
        }
    };

    const statsCards = [
        {
            title: 'Total Residents',
            value: kpis.total_residents,
            subtitle: 'Registered residents',
            icon: Users,
            iconClass: 'text-blue-500',
            iconBg: 'bg-blue-500/10',
            progress: 75,
        },
        {
            title: 'Open Tickets',
            value: kpis.open_tickets,
            subtitle: 'Active requests',
            icon: Wrench,
            iconClass: 'text-amber-500',
            iconBg: 'bg-amber-500/10',
            progress: kpis.open_tickets > 0 ? 50 : 0,
        },
        {
            title: "Today's Visitors",
            value: kpis.today_visitors,
            subtitle: 'Checked in today',
            icon: UserCheck,
            iconClass: 'text-emerald-500',
            iconBg: 'bg-emerald-500/10',
            progress: kpis.today_visitors > 0 ? 60 : 0,
        },
        {
            title: 'Assigned Units',
            value: kpis.assigned_units,
            subtitle: 'Of total units',
            icon: Home,
            iconClass: 'text-cyan-500',
            iconBg: 'bg-cyan-500/10',
            progress: 85,
        },
        {
            title: 'Recent Additions',
            value: kpis.recent_additions,
            subtitle: 'Last 7 days',
            icon: UserPlus,
            iconClass: 'text-indigo-500',
            iconBg: 'bg-indigo-500/10',
            progress: 40,
        },
        {
            title: 'Pending Actions',
            value: kpis.pending_actions,
            subtitle: 'Requires attention',
            icon: AlertCircle,
            iconClass: 'text-rose-500',
            iconBg: 'bg-rose-500/10',
            progress: kpis.pending_actions > 0 ? 100 : 0,
        },
    ];

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Sub-Admin Dashboard" />
            
            <div className="flex h-full flex-1 flex-col gap-8 p-8 overflow-x-hidden">
                <PageHeader 
                    title="Sub-Admin Dashboard"
                    description={`Welcome back, ${user.name}! Here's your society overview.`}
                />

                {/* KPI/Stats Grid */}
                <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
                    {statsCards.map((stat, index) => (
                        <GlassCard 
                            key={index}
                            className="group relative overflow-hidden transition-all duration-300 hover:-translate-y-1 py-6"
                            intensity="low"
                        >
                            <CardHeader className="relative pb-2">
                                <div className="flex items-start justify-between">
                                    <div className="space-y-1">
                                        <CardTitle className="text-sm font-medium text-muted-foreground">
                                            {stat.title}
                                        </CardTitle>
                                        <div className="text-3xl font-bold tracking-tight">
                                            {stat.value}
                                        </div>
                                    </div>
                                    <div className={`p-2.5 rounded-xl ${stat.iconBg} transition-colors group-hover:scale-110 duration-300`}>
                                        <stat.icon className={`h-5 w-5 ${stat.iconClass}`} />
                                    </div>
                                </div>
                            </CardHeader>
                            <CardContent>
                                <p className="text-xs text-muted-foreground mb-3">
                                    {stat.subtitle}
                                </p>
                                <div className="h-1.5 w-full bg-slate-100 dark:bg-slate-800 rounded-full overflow-hidden">
                                    <div 
                                        className={`h-full rounded-full transition-all duration-1000 ${stat.iconClass.replace('text-', 'bg-')}`}
                                        style={{ width: `${stat.progress}%` }}
                                    />
                                </div>
                            </CardContent>
                        </GlassCard>
                    ))}
                </div>

                {/* Recent Activity Section */}
                <GlassCard className="flex flex-col h-full py-6">
                    <CardHeader className="pb-4">
                        <div className="flex items-center justify-between">
                            <div>
                                <CardTitle className="text-xl flex items-center gap-2">
                                    <Clock className="h-5 w-5 text-primary" />
                                    Recent Activity
                                </CardTitle>
                                <CardDescription>
                                    Latest updates from your society
                                </CardDescription>
                            </div>
                        </div>
                    </CardHeader>
                    <CardContent>
                        <Tabs defaultValue="tickets" className="w-full">
                            <TabsList className="grid w-full grid-cols-3 lg:w-[400px] mb-6 p-1 bg-muted/50 rounded-xl">
                                <TabsTrigger value="tickets" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-amber-600 data-[state=active]:shadow-sm font-medium">Tickets</TabsTrigger>
                                <TabsTrigger value="visitors" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-emerald-600 data-[state=active]:shadow-sm font-medium">Visitors</TabsTrigger>
                                <TabsTrigger value="residents" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-blue-600 data-[state=active]:shadow-sm font-medium">Residents</TabsTrigger>
                            </TabsList>
                            
                            <TabsContent value="tickets" className="mt-0">
                                <div className="space-y-3">
                                    {recentActivity.tickets.length > 0 ? (
                                        recentActivity.tickets.map((ticket) => (
                                            <div key={ticket.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-slate-50 dark:hover:bg-white/5 border border-transparent hover:border-border/50 transition-all duration-200">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 shrink-0 rounded-lg bg-amber-500/10 flex items-center justify-center text-amber-600">
                                                        <Wrench className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-semibold text-foreground">{ticket.ticket_no}</p>
                                                        <p className="text-xs text-muted-foreground">{ticket.subject}</p>
                                                        <p className="text-[10px] text-muted-foreground mt-0.5">by {ticket.user_name}</p>
                                                    </div>
                                                </div>
                                                <div className="text-right flex flex-col items-end gap-1">
                                                    <Badge variant="outline" className={`${getStatusColor(ticket.status)} border capitalize shadow-none`}>
                                                        {ticket.status}
                                                    </Badge>
                                                    <span className="text-[10px] text-muted-foreground">{ticket.created_at}</span>
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="flex flex-col items-center justify-center py-12 text-center text-muted-foreground">
                                            <Wrench className="h-10 w-10 opacity-20 mb-2" />
                                            <p>No recent tickets</p>
                                        </div>
                                    )}
                                </div>
                            </TabsContent>

                            <TabsContent value="visitors" className="mt-0">
                                <div className="space-y-3">
                                    {recentActivity.visitors.length > 0 ? (
                                        recentActivity.visitors.map((visitor) => (
                                            <div key={visitor.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-slate-50 dark:hover:bg-white/5 border border-transparent hover:border-border/50 transition-all duration-200">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 shrink-0 rounded-lg bg-emerald-500/10 flex items-center justify-center text-emerald-600">
                                                        <UserCheck className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-semibold text-foreground">{visitor.name}</p>
                                                        <p className="text-xs text-muted-foreground capitalize">{visitor.type}</p>
                                                        <p className="text-[10px] text-muted-foreground mt-0.5">Host: {visitor.user_name}</p>
                                                    </div>
                                                </div>
                                                <div className="text-right flex flex-col items-end gap-1">
                                                    <Badge variant="outline" className={`${getStatusColor(visitor.status)} border capitalize shadow-none`}>
                                                        {visitor.status}
                                                    </Badge>
                                                    <span className="text-[10px] text-muted-foreground">{visitor.created_at}</span>
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="flex flex-col items-center justify-center py-12 text-center text-muted-foreground">
                                            <UserCheck className="h-10 w-10 opacity-20 mb-2" />
                                            <p>No recent visitors</p>
                                        </div>
                                    )}
                                </div>
                            </TabsContent>

                            <TabsContent value="residents" className="mt-0">
                                <div className="space-y-3">
                                    {recentActivity.residents.length > 0 ? (
                                        recentActivity.residents.map((resident) => (
                                            <div key={resident.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-slate-50 dark:hover:bg-white/5 border border-transparent hover:border-border/50 transition-all duration-200">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 shrink-0 rounded-lg bg-blue-500/10 flex items-center justify-center text-blue-600">
                                                        <Users className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-semibold text-foreground">{resident.name}</p>
                                                        <p className="text-xs text-muted-foreground">{resident.phone}</p>
                                                        <p className="text-[10px] text-muted-foreground mt-0.5">Unit: {resident.unit_no}</p>
                                                    </div>
                                                </div>
                                                <div className="text-right">
                                                    <span className="text-[10px] text-muted-foreground">{resident.created_at}</span>
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="flex flex-col items-center justify-center py-12 text-center text-muted-foreground">
                                            <Users className="h-10 w-10 opacity-20 mb-2" />
                                            <p>No recent residents</p>
                                        </div>
                                    )}
                                </div>
                            </TabsContent>
                        </Tabs>
                    </CardContent>
                </GlassCard>
            </div>
        </SocietyAdminLayout>
    );
}
