import { useState } from 'react';
import { Check } from 'lucide-react';
import { Switch } from '@/components/ui/switch';
import { Label } from '@/components/ui/label';

interface Plan {
    id: number;
    name: string;
    display_name: string;
    monthly_price: number;
    annual_price: number;
    features: any;
    description: string;
}

interface PlanSelectionProps {
    plans: Plan[];
    data: any;
    setData: (key: string, value: any) => void;
}

export default function PlanSelection({ plans, data, setData }: PlanSelectionProps) {
    const [billingCycle, setBillingCycle] = useState<'monthly' | 'annual'>(data.billing_cycle === 'annual' ? 'annual' : 'monthly');

    const handleBillingCycleChange = (checked: boolean) => {
        const cycle = checked ? 'annual' : 'monthly';
        setBillingCycle(cycle);
        setData('billing_cycle', cycle);
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    return (
        <div className="space-y-8">
            <div className="text-center">
                <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Choose a Plan</h2>
                <p className="text-gray-500 dark:text-gray-400 mb-6">Select the plan that best fits your society's needs.</p>
                
                <div className="flex items-center justify-center gap-4">
                    <Label htmlFor="billing-cycle" className={`text-sm font-medium ${billingCycle === 'monthly' ? 'text-gray-900 dark:text-white' : 'text-gray-500'}`}>Monthly</Label>
                    <Switch
                        id="billing-cycle"
                        checked={billingCycle === 'annual'}
                        onCheckedChange={handleBillingCycleChange}
                    />
                    <Label htmlFor="billing-cycle" className={`text-sm font-medium ${billingCycle === 'annual' ? 'text-gray-900 dark:text-white' : 'text-gray-500'}`}>
                        Annual <span className="text-green-600 text-xs ml-1">(Save ~17%)</span>
                    </Label>
                </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                {plans.map((plan) => {
                    const price = billingCycle === 'annual' ? plan.annual_price : plan.monthly_price;
                    const isSelected = data.plan_id === plan.id;

                    return (
                        <div
                            key={plan.id}
                            className={`relative rounded-2xl border-2 p-6 cursor-pointer transition-all duration-300 hover:shadow-lg ${
                                isSelected
                                    ? 'border-blue-600 bg-blue-50 dark:bg-blue-900/20 dark:border-blue-500'
                                    : 'border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800 hover:border-blue-300'
                            }`}
                            onClick={() => setData('plan_id', plan.id)}
                        >
                            {isSelected && (
                                <div className="absolute top-4 right-4 text-blue-600 dark:text-blue-400">
                                    <Check className="w-6 h-6" />
                                </div>
                            )}

                            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-2">{plan.display_name}</h3>
                            <p className="text-sm text-gray-500 dark:text-gray-400 mb-4 h-10">{plan.description}</p>

                            <div className="mb-6">
                                <span className="text-3xl font-bold text-gray-900 dark:text-white">{formatCurrency(price)}</span>
                                <span className="text-gray-500 dark:text-gray-400">/{billingCycle === 'annual' ? 'year' : 'month'}</span>
                            </div>

                            <ul className="space-y-3 mb-6">
                                {plan.features && Object.entries(plan.features).map(([key, value]) => (
                                    value && (
                                        <li key={key} className="flex items-center gap-2 text-sm text-gray-700 dark:text-gray-300">
                                            <Check className="w-4 h-4 text-green-500 flex-shrink-0" />
                                            <span>{key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}</span>
                                        </li>
                                    )
                                ))}
                            </ul>
                        </div>
                    );
                })}
            </div>
        </div>
    );
}
