import { CreditCard, ShieldCheck, CheckCircle2, AlertCircle, Loader2, IndianRupee } from 'lucide-react';
import { Button } from '@/components/ui/button';

interface Plan {
    id: number;
    name: string;
    display_name: string;
    monthly_price: number;
    quarterly_price?: number;
    annual_price: number;
}

interface PaymentStepProps {
    plan: Plan | undefined;
    billingCycle: string;
    amount: number;
    onPayNow: () => void;
    isLoading: boolean;
    error: string | null;
    paymentCompleted: boolean;
    settings: any;
}

export default function PaymentStep({ 
    plan, 
    billingCycle, 
    amount, 
    onPayNow, 
    isLoading, 
    error, 
    paymentCompleted,
    settings 
}: PaymentStepProps) {
    const formatCurrency = (value: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(value);
    };

    const getBillingLabel = () => {
        switch (billingCycle) {
            case 'annual':
                return 'Yearly';
            case 'quarterly':
                return 'Quarterly';
            default:
                return 'Monthly';
        }
    };

    // Check if Razorpay is enabled
    const razorpayEnabled = settings?.razorpay_enabled === '1';

    if (!plan) {
        return (
            <div className="text-center py-8">
                <AlertCircle className="w-12 h-12 mx-auto text-yellow-500 mb-4" />
                <h2 className="text-xl font-bold text-gray-900 dark:text-white mb-2">No Plan Selected</h2>
                <p className="text-gray-500 dark:text-gray-400">Please go back and select a subscription plan.</p>
            </div>
        );
    }

    if (paymentCompleted) {
        return (
            <div className="text-center py-8">
                <div className="w-20 h-20 mx-auto bg-green-100 dark:bg-green-900/30 rounded-full flex items-center justify-center mb-6">
                    <CheckCircle2 className="w-12 h-12 text-green-600 dark:text-green-400" />
                </div>
                <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">Payment Successful!</h2>
                <p className="text-gray-500 dark:text-gray-400 mb-6">
                    Your payment of {formatCurrency(amount)} has been received successfully.
                </p>
                <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-xl p-4 max-w-md mx-auto">
                    <p className="text-sm text-green-700 dark:text-green-400">
                        Click "Continue" to complete your society setup and start using the platform.
                    </p>
                </div>
            </div>
        );
    }

    if (!razorpayEnabled) {
        return (
            <div className="space-y-8">
                <div className="text-center">
                    <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Complete Payment</h2>
                    <p className="text-gray-500 dark:text-gray-400">Secure payment to activate your subscription</p>
                </div>

                <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-xl p-6 flex items-start gap-4">
                    <AlertCircle className="w-6 h-6 text-yellow-600 dark:text-yellow-400 flex-shrink-0 mt-1" />
                    <div>
                        <h3 className="font-semibold text-yellow-900 dark:text-yellow-300">Payment Gateway Not Configured</h3>
                        <p className="text-sm text-yellow-700 dark:text-yellow-400 mt-1">
                            The payment gateway is not enabled. Please contact the administrator to configure Razorpay in the system settings.
                        </p>
                    </div>
                </div>
            </div>
        );
    }

    return (
        <div className="space-y-8">
            <div className="text-center">
                <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Complete Payment</h2>
                <p className="text-gray-500 dark:text-gray-400">Secure payment to activate your subscription</p>
            </div>

            {/* Order Summary */}
            <div className="bg-gray-50 dark:bg-gray-900 rounded-2xl p-6 max-w-md mx-auto">
                <h3 className="font-semibold text-gray-900 dark:text-white mb-4">Order Summary</h3>
                
                <div className="space-y-3">
                    <div className="flex justify-between text-sm">
                        <span className="text-gray-500 dark:text-gray-400">Plan</span>
                        <span className="font-medium text-gray-900 dark:text-white">{plan.display_name}</span>
                    </div>
                    <div className="flex justify-between text-sm">
                        <span className="text-gray-500 dark:text-gray-400">Billing Cycle</span>
                        <span className="font-medium text-gray-900 dark:text-white">{getBillingLabel()}</span>
                    </div>
                    <div className="border-t border-gray-200 dark:border-gray-700 pt-3 mt-3">
                        <div className="flex justify-between items-center">
                            <span className="text-lg font-semibold text-gray-900 dark:text-white">Total</span>
                            <span className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                {formatCurrency(amount)}
                            </span>
                        </div>
                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1 text-right">
                            (Inclusive of all taxes)
                        </p>
                    </div>
                </div>
            </div>

            {/* Error Message */}
            {error && (
                <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-xl p-4 flex items-start gap-3 max-w-md mx-auto">
                    <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400 flex-shrink-0 mt-0.5" />
                    <p className="text-sm text-red-700 dark:text-red-400">{error}</p>
                </div>
            )}

            {/* Pay Button */}
            <div className="flex justify-center">
                <Button
                    onClick={onPayNow}
                    disabled={isLoading}
                    className="bg-blue-600 hover:bg-blue-700 text-white px-8 py-6 text-lg rounded-xl shadow-lg shadow-blue-200 dark:shadow-blue-900/20"
                >
                    {isLoading ? (
                        <>
                            <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                            Processing...
                        </>
                    ) : (
                        <>
                            <IndianRupee className="w-5 h-5 mr-2" />
                            Pay {formatCurrency(amount)}
                        </>
                    )}
                </Button>
            </div>

            {/* Security Notice */}
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-xl p-6 flex items-start gap-4 max-w-md mx-auto">
                <ShieldCheck className="w-6 h-6 text-blue-600 dark:text-blue-400 flex-shrink-0 mt-1" />
                <div>
                    <h3 className="font-semibold text-blue-900 dark:text-blue-300">Secure Transaction</h3>
                    <p className="text-sm text-blue-700 dark:text-blue-400 mt-1">
                        This is a secure 256-bit SSL encrypted payment powered by Razorpay. Your payment details are safe.
                    </p>
                </div>
            </div>
        </div>
    );
}
