import { useState, useEffect, useCallback } from 'react';
import { useForm, usePage } from '@inertiajs/react';
import { motion, AnimatePresence } from 'framer-motion';
import { Check, ChevronRight, Building2, CreditCard, ShieldCheck, Sparkles, Loader2 } from 'lucide-react';
import SocietyDetailsForm from './SocietyDetailsForm';
import PlanSelection from './PlanSelection';
import PaymentStep from './PaymentStep';
import ActivationStep from './ActivationStep';
import { Button } from '@/components/ui/button';
import { store } from '@/routes/onboarding/society';
import axios from 'axios';

interface Plan {
    id: number;
    name: string;
    display_name: string;
    monthly_price: number;
    quarterly_price?: number;
    annual_price: number;
    features: any;
    description: string;
}

interface OnboardingWizardProps {
    plans: Plan[];
    settings: any;
}

declare global {
    interface Window {
        Razorpay: any;
    }
}

export default function OnboardingWizard({ plans, settings }: OnboardingWizardProps) {
    const [step, setStep] = useState(1);
    const [paymentCompleted, setPaymentCompleted] = useState(false);
    const [paymentData, setPaymentData] = useState<{
        razorpay_payment_id: string;
        razorpay_order_id: string;
        razorpay_signature: string;
    } | null>(null);
    const [isCreatingOrder, setIsCreatingOrder] = useState(false);
    const [paymentError, setPaymentError] = useState<string | null>(null);
    
    const { data, setData, post, processing, errors } = useForm({
        name: '',
        email: '',
        phone: '',
        address: '',
        city: '',
        state: '',
        pincode: '',
        country: '',
        plan_id: '',
        billing_cycle: 'monthly',
        payment_method: 'razorpay',
        razorpay_payment_id: '',
        razorpay_order_id: '',
        razorpay_signature: '',
    });

    const steps = [
        { id: 1, name: 'Society Details', icon: Building2 },
        { id: 2, name: 'Select Plan', icon: Sparkles },
        { id: 3, name: 'Payment', icon: CreditCard },
        { id: 4, name: 'Activation', icon: ShieldCheck },
    ];

    const nextStep = () => setStep((s) => Math.min(s + 1, 4));
    const prevStep = () => setStep((s) => Math.max(s - 1, 1));

    // Load Razorpay script
    useEffect(() => {
        const script = document.createElement('script');
        script.src = 'https://checkout.razorpay.com/v1/checkout.js';
        script.async = true;
        document.body.appendChild(script);
        
        return () => {
            document.body.removeChild(script);
        };
    }, []);

    const getSelectedPlan = useCallback(() => {
        return plans.find(p => p.id === Number(data.plan_id));
    }, [plans, data.plan_id]);

    const getAmount = useCallback(() => {
        const plan = getSelectedPlan();
        if (!plan) return 0;
        
        switch (data.billing_cycle) {
            case 'annual':
                return plan.annual_price;
            case 'quarterly':
                return plan.quarterly_price ?? (plan.monthly_price * 3);
            default:
                return plan.monthly_price;
        }
    }, [data.billing_cycle, getSelectedPlan]);

    const initiatePayment = async () => {
        setIsCreatingOrder(true);
        setPaymentError(null);

        try {
            const response = await axios.post('/onboarding/payment/create-order', {
                plan_id: data.plan_id,
                billing_cycle: data.billing_cycle,
            });

            if (!response.data.success) {
                setPaymentError(response.data.message || 'Failed to create payment order');
                setIsCreatingOrder(false);
                return;
            }

            const { order_id, amount, key } = response.data;

            const options = {
                key: key,
                amount: amount * 100,
                currency: 'INR',
                name: 'Societify',
                description: `${getSelectedPlan()?.display_name} - ${data.billing_cycle} Subscription`,
                order_id: order_id,
                handler: function (response: any) {
                    setPaymentData({
                        razorpay_payment_id: response.razorpay_payment_id,
                        razorpay_order_id: response.razorpay_order_id,
                        razorpay_signature: response.razorpay_signature,
                    });
                    setData({
                        ...data,
                        razorpay_payment_id: response.razorpay_payment_id,
                        razorpay_order_id: response.razorpay_order_id,
                        razorpay_signature: response.razorpay_signature,
                    });
                    setPaymentCompleted(true);
                    setIsCreatingOrder(false);
                    nextStep(); // Move to activation step
                },
                prefill: {
                    name: data.name,
                    email: data.email,
                    contact: data.phone,
                },
                theme: {
                    color: '#2563eb',
                },
                modal: {
                    ondismiss: function () {
                        setIsCreatingOrder(false);
                        setPaymentError('Payment was cancelled. Please try again.');
                    },
                },
            };

            const razorpay = new window.Razorpay(options);
            razorpay.open();
        } catch (error: any) {
            console.error('Payment error:', error);
            setPaymentError(error.response?.data?.message || 'An error occurred while processing payment');
            setIsCreatingOrder(false);
        }
    };

    const handleSubmit = () => {
        if (!paymentCompleted || !paymentData) {
            setPaymentError('Please complete payment first');
            return;
        }

        // Update form data with payment details before submitting
        setData('razorpay_payment_id', paymentData.razorpay_payment_id);
        setData('razorpay_order_id', paymentData.razorpay_order_id);
        setData('razorpay_signature', paymentData.razorpay_signature);
        
        post(store().url);
    };

    const canProceedToPayment = () => {
        return data.plan_id && data.billing_cycle;
    };

    return (
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900 flex flex-col items-center justify-center p-4">
            <div className="w-full max-w-4xl bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden">
                {/* Header / Progress Bar */}
                <div className="bg-white dark:bg-gray-800 p-6 border-b border-gray-100 dark:border-gray-700">
                    <div className="flex items-center justify-between relative">
                        {/* Progress Line */}
                        <div className="absolute left-0 top-1/2 w-full h-1 bg-gray-100 dark:bg-gray-700 -z-10" />
                        <div 
                            className="absolute left-0 top-1/2 h-1 bg-blue-600 transition-all duration-500 -z-10"
                            style={{ width: `${((step - 1) / (steps.length - 1)) * 100}%` }}
                        />

                        {steps.map((s) => (
                            <div key={s.id} className="flex flex-col items-center gap-2 bg-white dark:bg-gray-800 px-2">
                                <div
                                    className={`w-10 h-10 rounded-full flex items-center justify-center transition-all duration-300 ${
                                        step >= s.id
                                            ? 'bg-blue-600 text-white shadow-lg shadow-blue-200 dark:shadow-blue-900/20'
                                            : 'bg-gray-100 dark:bg-gray-700 text-gray-400 dark:text-gray-500'
                                    }`}
                                >
                                    {step > s.id ? <Check className="w-6 h-6" /> : <s.icon className="w-5 h-5" />}
                                </div>
                                <span
                                    className={`text-xs font-medium transition-colors duration-300 ${
                                        step >= s.id ? 'text-blue-600 dark:text-blue-400' : 'text-gray-400 dark:text-gray-500'
                                    }`}
                                >
                                    {s.name}
                                </span>
                            </div>
                        ))}
                    </div>
                </div>

                {/* Content Area */}
                <div className="p-6 md:p-10 min-h-[400px]">
                    <AnimatePresence mode="wait">
                        <motion.div
                            key={step}
                            initial={{ opacity: 0, x: 20 }}
                            animate={{ opacity: 1, x: 0 }}
                            exit={{ opacity: 0, x: -20 }}
                            transition={{ duration: 0.3 }}
                        >
                            {step === 1 && (
                                <SocietyDetailsForm data={data} setData={setData} errors={errors} />
                            )}
                            {step === 2 && (
                                <PlanSelection plans={plans} data={data} setData={setData} />
                            )}
                            {step === 3 && (
                                <PaymentStep 
                                    plan={getSelectedPlan()} 
                                    billingCycle={data.billing_cycle}
                                    amount={getAmount()}
                                    onPayNow={initiatePayment}
                                    isLoading={isCreatingOrder}
                                    error={paymentError}
                                    paymentCompleted={paymentCompleted}
                                    settings={settings}
                                />
                            )}
                            {step === 4 && (
                                <ActivationStep 
                                    onSubmit={handleSubmit} 
                                    processing={processing}
                                    paymentCompleted={paymentCompleted}
                                />
                            )}
                        </motion.div>
                    </AnimatePresence>
                </div>

                {/* Footer / Navigation */}
                <div className="bg-gray-50 dark:bg-gray-800/50 p-6 border-t border-gray-100 dark:border-gray-700 flex justify-between items-center">
                    <Button
                        variant="ghost"
                        onClick={prevStep}
                        disabled={step === 1 || processing || isCreatingOrder}
                        className={step === 1 ? 'invisible' : ''}
                    >
                        Back
                    </Button>

                    {step < 3 && (
                        <Button 
                            onClick={nextStep} 
                            disabled={(step === 2 && !canProceedToPayment())}
                            className="bg-blue-600 hover:bg-blue-700"
                        >
                            Next Step <ChevronRight className="w-4 h-4 ml-2" />
                        </Button>
                    )}
                    
                    {step === 3 && !paymentCompleted && (
                        <Button 
                            onClick={initiatePayment}
                            disabled={isCreatingOrder || !canProceedToPayment()}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            {isCreatingOrder ? (
                                <>
                                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                                    Processing...
                                </>
                            ) : (
                                <>Pay Now - ₹{getAmount().toLocaleString('en-IN')}</>
                            )}
                        </Button>
                    )}

                    {step === 3 && paymentCompleted && (
                        <Button 
                            onClick={nextStep}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            Continue <ChevronRight className="w-4 h-4 ml-2" />
                        </Button>
                    )}

                    {step === 4 && (
                        <Button 
                            onClick={handleSubmit} 
                            disabled={processing || !paymentCompleted}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            {processing ? (
                                <>
                                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                                    Activating...
                                </>
                            ) : (
                                'Complete Setup'
                            )}
                        </Button>
                    )}
                </div>
            </div>
        </div>
    );
}
