import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { ShieldCheck, Plus, Search, Eye, LogIn, LogOut, CheckCircle2, XCircle, Clock, User, Car, QrCode } from 'lucide-react';
import { useState, useEffect } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Visitors',
        href: '/societies/{society}/visitors',
    },
];

interface Visitor {
    id: number;
    name: string;
    phone: string;
    email: string | null;
    type: string;
    purpose: string;
    no_of_visitors: number;
    vehicle_number: string | null;
    gatepass_code: string | null;
    entry_time: string | null;
    exit_time: string | null;
    status: string;
    created_at: string;
    user: {
        id: number;
        name: string;
        phone: string;
        unit_id: number | null;
        unit: {
            id: number;
            unit_no: string;
        } | null;
    } | null;
    checked_in_by: {
        id: number;
        name: string;
    } | null;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    visitors: {
        data: Visitor[];
        links: any;
        meta: any;
    };
    residents: any[];
    filters: {
        search?: string;
        status?: string;
        type?: string;
        date_from?: string;
        date_to?: string;
    };
    stats: {
        total_today: number;
        checked_in: number;
        pending: number;
    };
}

const statusConfig = {
    pending: { label: 'Pending', color: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400', icon: Clock },
    approved: { label: 'Approved', color: 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400', icon: CheckCircle2 },
    rejected: { label: 'Rejected', color: 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400', icon: XCircle },
    checked_in: { label: 'Checked In', color: 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400', icon: LogIn },
    checked_out: { label: 'Checked Out', color: 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400', icon: LogOut },
};

const typeConfig = {
    guest: 'Guest',
    delivery: 'Delivery',
    service: 'Service',
    staff: 'Staff',
};

export default function Index({ society, visitors, residents, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');
    const [type, setType] = useState(filters.type || 'all');
    const [dateFrom, setDateFrom] = useState(filters.date_from || '');
    const [dateTo, setDateTo] = useState(filters.date_to || '');

    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isViewOpen, setIsViewOpen] = useState(false);
    const [viewingVisitor, setViewingVisitor] = useState<Visitor | null>(null);

    // Create Form
    const createForm = useForm({
        name: '',
        phone: '',
        type: 'guest',
        purpose: '',
        no_of_visitors: 1,
        vehicle_number: '',
        user_id: 'none',
        entry_time: '',
    });

    // Reset form when modal closes
    useEffect(() => {
        if (!isCreateOpen) {
            createForm.reset();
        }
    }, [isCreateOpen]);

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (status && status !== 'all') filterParams.status = status;
        if (type && type !== 'all') filterParams.type = type;
        if (dateFrom) filterParams.date_from = dateFrom;
        if (dateTo) filterParams.date_to = dateTo;

        router.get(`/societies/${society.id}/visitors`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/visitors`, {
            onSuccess: () => setIsCreateOpen(false),
        });
    };

    const openViewModal = (visitor: Visitor) => {
        setViewingVisitor(visitor);
        setIsViewOpen(true);
    };

    const handleCheckIn = (visitorId: number) => {
        if (confirm('Are you sure you want to check in this visitor?')) {
            router.post(`/societies/${society.id}/visitors/${visitorId}/check-in`, {}, {
                preserveScroll: true,
            });
        }
    };

    const handleCheckOut = (visitorId: number) => {
        if (confirm('Are you sure you want to check out this visitor?')) {
            router.post(`/societies/${society.id}/visitors/${visitorId}/check-out`, {}, {
                preserveScroll: true,
            });
        }
    };

    const handleApprove = (visitorId: number) => {
        if (confirm('Are you sure you want to approve this visitor?')) {
            router.post(`/societies/${society.id}/visitors/${visitorId}/approve`, {}, {
                preserveScroll: true,
            });
        }
    };

    const handleReject = (visitorId: number) => {
        if (confirm('Are you sure you want to reject this visitor?')) {
            router.post(`/societies/${society.id}/visitors/${visitorId}/reject`, {}, {
                preserveScroll: true,
            });
        }
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Visitors" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Visitors</h1>
                        <p className="text-muted-foreground">
                            Manage all visitors in your society
                        </p>
                    </div>
                    
                    <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Register Visitor
                            </Button>
                        </DialogTrigger>
                        <DialogContent className="sm:max-w-[600px]">
                            <DialogHeader>
                                <DialogTitle>Register New Visitor</DialogTitle>
                                <DialogDescription>
                                    Enter visitor details to create a gate pass.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="space-y-4 py-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Visitor Name</Label>
                                        <Input
                                            id="name"
                                            value={createForm.data.name}
                                            onChange={(e) => createForm.setData('name', e.target.value)}
                                            required
                                        />
                                        {createForm.errors.name && <p className="text-sm text-red-500">{createForm.errors.name}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="phone">Phone Number</Label>
                                        <Input
                                            id="phone"
                                            value={createForm.data.phone}
                                            onChange={(e) => createForm.setData('phone', e.target.value)}
                                            required
                                        />
                                        {createForm.errors.phone && <p className="text-sm text-red-500">{createForm.errors.phone}</p>}
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="type">Visitor Type</Label>
                                        <Select 
                                            value={createForm.data.type} 
                                            onValueChange={(value) => createForm.setData('type', value)}
                                        >
                                            <SelectTrigger id="type">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="guest">Guest</SelectItem>
                                                <SelectItem value="delivery">Delivery</SelectItem>
                                                <SelectItem value="service">Service</SelectItem>
                                                <SelectItem value="staff">Staff</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="no_of_visitors">Number of Visitors</Label>
                                        <Input
                                            id="no_of_visitors"
                                            type="number"
                                            min="1"
                                            value={createForm.data.no_of_visitors}
                                            onChange={(e) => createForm.setData('no_of_visitors', parseInt(e.target.value))}
                                            required
                                        />
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="purpose">Purpose of Visit</Label>
                                    <Input
                                        id="purpose"
                                        value={createForm.data.purpose}
                                        onChange={(e) => createForm.setData('purpose', e.target.value)}
                                        required
                                    />
                                    {createForm.errors.purpose && <p className="text-sm text-red-500">{createForm.errors.purpose}</p>}
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="user_id">Visiting Resident (Optional)</Label>
                                        <Select 
                                            value={createForm.data.user_id} 
                                            onValueChange={(value) => createForm.setData('user_id', value)}
                                        >
                                            <SelectTrigger id="user_id">
                                                <SelectValue placeholder="Select Resident" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="none">None</SelectItem>
                                                {residents.map((resident) => (
                                                    <SelectItem key={resident.id} value={resident.id.toString()}>
                                                        {resident.name} {resident.unit ? `(${resident.unit.unit_no})` : ''}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="vehicle_number">Vehicle Number (Optional)</Label>
                                        <Input
                                            id="vehicle_number"
                                            value={createForm.data.vehicle_number}
                                            onChange={(e) => createForm.setData('vehicle_number', e.target.value)}
                                            placeholder="MH-12-AB-1234"
                                        />
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="entry_time">Expected Entry Time (Optional)</Label>
                                    <Input
                                        id="entry_time"
                                        type="datetime-local"
                                        value={createForm.data.entry_time}
                                        onChange={(e) => createForm.setData('entry_time', e.target.value)}
                                    />
                                    <p className="text-xs text-muted-foreground">Leave blank for immediate entry</p>
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={createForm.processing}>
                                        {createForm.processing ? 'Registering...' : 'Register Visitor'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>
                </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-3">
                    <Card className="border-l-4 border-l-blue-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Visitors Today</CardTitle>
                            <User className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_today}</div>
                            <p className="text-xs text-muted-foreground">
                                Registered today
                            </p>
                        </CardContent>
                    </Card>
                    <Card className="border-l-4 border-l-green-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Checked In</CardTitle>
                            <LogIn className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.checked_in}</div>
                            <p className="text-xs text-muted-foreground">
                                Currently inside
                            </p>
                        </CardContent>
                    </Card>
                    <Card className="border-l-4 border-l-yellow-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Pending Approval</CardTitle>
                            <Clock className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.pending}</div>
                            <p className="text-xs text-muted-foreground">
                                Awaiting confirmation
                            </p>
                        </CardContent>
                    </Card>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4 flex-wrap">
                            <div className="flex-1 min-w-[200px]">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by name, phone, vehicle, gatepass..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="status">Status</Label>
                                <Select value={status} onValueChange={setStatus}>
                                    <SelectTrigger id="status" className="mt-1">
                                        <SelectValue placeholder="All Status" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        {Object.entries(statusConfig).map(([s, c]) => (
                                            <SelectItem key={s} value={s}>
                                                {c.label}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="type">Type</Label>
                                <Select value={type} onValueChange={setType}>
                                    <SelectTrigger id="type" className="mt-1">
                                        <SelectValue placeholder="All Types" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Types</SelectItem>
                                        {Object.entries(typeConfig).map(([t, label]) => (
                                            <SelectItem key={t} value={t}>
                                                {label}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="date_from">Date From</Label>
                                <Input
                                    id="date_from"
                                    type="date"
                                    value={dateFrom}
                                    onChange={(e) => setDateFrom(e.target.value)}
                                    className="mt-1"
                                />
                            </div>
                            <div className="w-48">
                                <Label htmlFor="date_to">Date To</Label>
                                <Input
                                    id="date_to"
                                    type="date"
                                    value={dateTo}
                                    onChange={(e) => setDateTo(e.target.value)}
                                    className="mt-1"
                                />
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Visitors Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>All Visitors</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Visitor</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Type</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Resident</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Purpose</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Entry Time</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Exit Time</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Status</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {visitors.data.length > 0 ? (
                                        visitors.data.map((visitor) => {
                                            const StatusIcon = statusConfig[visitor.status as keyof typeof statusConfig].icon;
                                            return (
                                                <tr key={visitor.id} className="border-b hover:bg-muted/50">
                                                    <td className="px-4 py-3">
                                                        <div>
                                                            <div className="font-medium">{visitor.name}</div>
                                                            <div className="text-xs text-muted-foreground">
                                                                {visitor.phone}
                                                            </div>
                                                            {visitor.gatepass_code && (
                                                                <div className="text-xs text-muted-foreground font-mono bg-muted px-1 rounded inline-block mt-1">
                                                                    {visitor.gatepass_code}
                                                                </div>
                                                            )}
                                                        </div>
                                                    </td>
                                                    <td className="px-4 py-3 text-sm capitalize">
                                                        {typeConfig[visitor.type as keyof typeof typeConfig]}
                                                    </td>
                                                    <td className="px-4 py-3 text-sm">
                                                        {visitor.user ? (
                                                            <div>
                                                                <div className="font-medium">{visitor.user.name}</div>
                                                                {visitor.user.unit && (
                                                                    <div className="text-xs text-muted-foreground">
                                                                        {visitor.user.unit.unit_no}
                                                                    </div>
                                                                )}
                                                            </div>
                                                        ) : (
                                                            <span className="text-muted-foreground">-</span>
                                                        )}
                                                    </td>
                                                    <td className="px-4 py-3 text-sm">
                                                        {visitor.purpose}
                                                    </td>
                                                    <td className="px-4 py-3 text-sm text-muted-foreground">
                                                        {visitor.entry_time
                                                            ? new Date(visitor.entry_time).toLocaleString()
                                                            : '-'}
                                                    </td>
                                                    <td className="px-4 py-3 text-sm text-muted-foreground">
                                                        {visitor.exit_time
                                                            ? new Date(visitor.exit_time).toLocaleString()
                                                            : '-'}
                                                    </td>
                                                    <td className="px-4 py-3">
                                                        <div className="flex items-center gap-2">
                                                            <span
                                                                className={`inline-flex items-center gap-1 rounded-full px-2 py-1 text-xs font-medium ${statusConfig[visitor.status as keyof typeof statusConfig].color}`}
                                                            >
                                                                <StatusIcon className="h-3 w-3" />
                                                                {statusConfig[visitor.status as keyof typeof statusConfig].label}
                                                            </span>
                                                        </div>
                                                    </td>
                                                    <td className="px-4 py-3">
                                                        <div className="flex items-center justify-end gap-2">
                                                            <Button 
                                                                variant="ghost" 
                                                                size="icon" 
                                                                className="h-8 w-8 hover:bg-blue-50 hover:text-blue-600"
                                                                onClick={() => openViewModal(visitor)}
                                                            >
                                                                <Eye className="h-4 w-4" />
                                                            </Button>

                                                            {visitor.status === 'pending' && (
                                                                <>
                                                                    <Button
                                                                        variant="ghost"
                                                                        size="icon"
                                                                        className="h-8 w-8 hover:bg-green-50 hover:text-green-600"
                                                                        onClick={() => handleApprove(visitor.id)}
                                                                        title="Approve"
                                                                    >
                                                                        <CheckCircle2 className="h-4 w-4" />
                                                                    </Button>
                                                                    <Button
                                                                        variant="ghost"
                                                                        size="icon"
                                                                        className="h-8 w-8 hover:bg-red-50 hover:text-red-600"
                                                                        onClick={() => handleReject(visitor.id)}
                                                                        title="Reject"
                                                                    >
                                                                        <XCircle className="h-4 w-4" />
                                                                    </Button>
                                                                </>
                                                            )}

                                                            {visitor.status === 'approved' && (
                                                                <Button
                                                                    variant="ghost"
                                                                    size="icon"
                                                                    className="h-8 w-8 hover:bg-green-50 hover:text-green-600"
                                                                    onClick={() => handleCheckIn(visitor.id)}
                                                                    title="Check In"
                                                                >
                                                                    <LogIn className="h-4 w-4" />
                                                                </Button>
                                                            )}
                                                            {visitor.status === 'checked_in' && (
                                                                <Button
                                                                    variant="ghost"
                                                                    size="icon"
                                                                    className="h-8 w-8 hover:bg-amber-50 hover:text-amber-600"
                                                                    onClick={() => handleCheckOut(visitor.id)}
                                                                    title="Check Out"
                                                                >
                                                                    <LogOut className="h-4 w-4" />
                                                                </Button>
                                                            )}
                                                        </div>
                                                    </td>
                                                </tr>
                                            );
                                        })
                                    ) : (
                                        <tr>
                                            <td colSpan={8} className="px-4 py-8 text-center text-muted-foreground">
                                                No visitors found
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {visitors.links && visitors.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Showing {visitors.meta.from} to {visitors.meta.to} of {visitors.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {visitors.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>

                {/* View Dialog */}
                <Dialog open={isViewOpen} onOpenChange={setIsViewOpen}>
                    <DialogContent className="sm:max-w-[600px]">
                        <DialogHeader>
                            <DialogTitle>Visitor Details</DialogTitle>
                            <DialogDescription>
                                Detailed information about the visitor.
                            </DialogDescription>
                        </DialogHeader>
                        {viewingVisitor && (
                            <div className="space-y-6 py-4">
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center gap-4">
                                        <div className="h-16 w-16 rounded-full bg-indigo-100 flex items-center justify-center text-indigo-600 font-bold text-2xl">
                                            {viewingVisitor.name.charAt(0)}
                                        </div>
                                        <div>
                                            <h3 className="text-xl font-bold">{viewingVisitor.name}</h3>
                                            <p className="text-muted-foreground">{viewingVisitor.phone}</p>
                                        </div>
                                    </div>
                                    <div className="text-right">
                                        <div className="text-sm text-muted-foreground">Gatepass Code</div>
                                        <div className="text-2xl font-mono font-bold tracking-wider">{viewingVisitor.gatepass_code}</div>
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Type</h4>
                                        <p className="text-base capitalize">{typeConfig[viewingVisitor.type as keyof typeof typeConfig]}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Status</h4>
                                        <span className={`inline-flex items-center gap-1 rounded-full px-2.5 py-0.5 text-xs font-medium mt-1 ${statusConfig[viewingVisitor.status as keyof typeof statusConfig].color}`}>
                                            {statusConfig[viewingVisitor.status as keyof typeof statusConfig].label}
                                        </span>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Purpose</h4>
                                        <p className="text-base">{viewingVisitor.purpose}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Visitors Count</h4>
                                        <p className="text-base">{viewingVisitor.no_of_visitors}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Vehicle Number</h4>
                                        <p className="text-base">{viewingVisitor.vehicle_number || '-'}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Visiting Resident</h4>
                                        <p className="text-base">
                                            {viewingVisitor.user ? (
                                                <span>{viewingVisitor.user.name} {viewingVisitor.user.unit && `(${viewingVisitor.user.unit.unit_no})`}</span>
                                            ) : (
                                                '-'
                                            )}
                                        </p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Entry Time</h4>
                                        <p className="text-base">{viewingVisitor.entry_time ? new Date(viewingVisitor.entry_time).toLocaleString() : '-'}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Exit Time</h4>
                                        <p className="text-base">{viewingVisitor.exit_time ? new Date(viewingVisitor.exit_time).toLocaleString() : '-'}</p>
                                    </div>
                                </div>
                                
                                {viewingVisitor.checked_in_by && (
                                    <div className="pt-4 border-t">
                                        <h4 className="text-sm font-medium text-muted-foreground">Checked In By</h4>
                                        <p className="text-sm">{viewingVisitor.checked_in_by.name}</p>
                                    </div>
                                )}
                            </div>
                        )}
                        <DialogFooter>
                            <Button onClick={() => setIsViewOpen(false)}>Close</Button>
                        </DialogFooter>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
