import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link } from '@inertiajs/react';
import { ArrowLeft, Building2, Mail, Phone, UserCircle2, MapPin, FileText } from 'lucide-react';
import { Button } from '@/components/ui/button';

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Vendors', href: '/societies/{society}/vendors' },
    { title: 'Details', href: '#' },
];

interface Vendor {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    company_name: string | null;
    gst_number: string | null;
    address: string | null;
    status: string;
    created_at: string;
}

interface ShowProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    vendor: Vendor;
}

export default function Show({ society, vendor }: ShowProps) {
    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title={vendor.name} />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center gap-4">
                    <Link href={`/societies/${society.id}/vendors`}>
                        <Button variant="ghost" size="sm">
                            <ArrowLeft className="mr-2 h-4 w-4" />
                            Back
                        </Button>
                    </Link>
                    <div>
                        <h1 className="text-2xl font-bold">Vendor Details</h1>
                        <p className="text-muted-foreground">View vendor profile and contact information</p>
                    </div>
                </div>

                <Card>
                    <CardHeader>
                        <div className="flex items-center gap-3">
                            <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10">
                                <UserCircle2 className="h-7 w-7 text-primary" />
                            </div>
                            <div>
                                <CardTitle className="text-xl">{vendor.name}</CardTitle>
                                {vendor.company_name && (
                                    <p className="flex items-center gap-1 text-sm text-muted-foreground">
                                        <Building2 className="h-3 w-3" />
                                        {vendor.company_name}
                                    </p>
                                )}
                            </div>
                        </div>
                    </CardHeader>
                    <CardContent className="grid gap-6 md:grid-cols-2">
                        <div className="space-y-4">
                            <div>
                                <h3 className="mb-2 text-sm font-semibold text-muted-foreground">Contact</h3>
                                <div className="space-y-2 text-sm">
                                    {vendor.phone && (
                                        <div className="flex items-center gap-2">
                                            <Phone className="h-4 w-4 text-muted-foreground" />
                                            <a href={`tel:${vendor.phone}`} className="text-primary hover:underline">
                                                {vendor.phone}
                                            </a>
                                        </div>
                                    )}
                                    {vendor.email && (
                                        <div className="flex items-center gap-2">
                                            <Mail className="h-4 w-4 text-muted-foreground" />
                                            <a href={`mailto:${vendor.email}`} className="text-primary hover:underline">
                                                {vendor.email}
                                            </a>
                                        </div>
                                    )}
                                </div>
                            </div>

                            <div>
                                <h3 className="mb-2 text-sm font-semibold text-muted-foreground">Status</h3>
                                <span
                                    className={`inline-flex rounded-full px-2 py-1 text-xs font-medium ${
                                        vendor.status === 'active'
                                            ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200'
                                            : 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200'
                                    }`}
                                >
                                    {vendor.status}
                                </span>
                            </div>
                        </div>

                        <div className="space-y-4">
                            <div>
                                <h3 className="mb-2 text-sm font-semibold text-muted-foreground">Tax Information</h3>
                                <div className="space-y-1 text-sm">
                                    <div className="flex items-center gap-2">
                                        <FileText className="h-4 w-4 text-muted-foreground" />
                                        <span className="font-medium">GST Number:</span>
                                        <span>{vendor.gst_number || '-'}</span>
                                    </div>
                                </div>
                            </div>

                            <div>
                                <h3 className="mb-2 text-sm font-semibold text-muted-foreground">Address</h3>
                                <div className="flex items-start gap-2 text-sm">
                                    <MapPin className="mt-0.5 h-4 w-4 text-muted-foreground" />
                                    <span>{vendor.address || 'No address provided'}</span>
                                </div>
                            </div>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}


