import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { Search, UserCircle2, Phone, Mail, Building2, Plus, Edit, Trash2, Eye } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Vendors',
        href: '/societies/{society}/vendors',
    },
];

interface Vendor {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    company_name: string | null;
    gst_number: string | null;
    address: string | null;
    status: string;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    vendors: {
        data: Vendor[];
        links: any;
        meta: any;
    };
    filters: {
        search?: string;
        status?: string;
    };
}

export default function Index({ society, vendors, filters }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (status && status !== 'all') filterParams.status = status;

        router.get(`/societies/${society.id}/vendors`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this vendor?')) {
            router.delete(`/societies/${society.id}/vendors/${id}`, {
                preserveScroll: true,
            });
        }
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Vendors" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Vendors</h1>
                        <p className="text-muted-foreground">
                            Manage all service providers and vendors for your society
                        </p>
                    </div>
                    <Link href={`/societies/${society.id}/vendors/create`}>
                        <Button>
                            <Plus className="mr-2 h-4 w-4" />
                            Add Vendor
                        </Button>
                    </Link>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="grid gap-4 md:grid-cols-3">
                            <div className="md:col-span-2">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by name, phone, email, or company..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="status">Status</Label>
                                <select
                                    id="status"
                                    value={status}
                                    onChange={(e) => setStatus(e.target.value)}
                                    className="mt-1 block w-full rounded-md border border-input bg-background px-3 py-2 text-sm shadow-sm focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring"
                                >
                                    <option value="all">All Status</option>
                                    <option value="active">Active</option>
                                    <option value="inactive">Inactive</option>
                                </select>
                            </div>
                        </div>

                        <div className="mt-4 flex items-end gap-4">
                            <Button onClick={handleSearch}>Filter</Button>
                        </div>
                    </CardContent>
                </Card>

                {/* Vendors Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>All Vendors</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Vendor</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Contact</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Company</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">GST</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Status</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {vendors.data.length > 0 ? (
                                        vendors.data.map((vendor) => (
                                            <tr key={vendor.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-2">
                                                        <UserCircle2 className="h-5 w-5 text-muted-foreground" />
                                                        <span className="font-medium">{vendor.name}</span>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    <div className="space-y-1">
                                                        {vendor.phone && (
                                                            <div className="flex items-center gap-1">
                                                                <Phone className="h-3 w-3 text-muted-foreground" />
                                                                <a
                                                                    href={`tel:${vendor.phone}`}
                                                                    className="text-primary hover:underline"
                                                                >
                                                                    {vendor.phone}
                                                                </a>
                                                            </div>
                                                        )}
                                                        {vendor.email && (
                                                            <div className="flex items-center gap-1">
                                                                <Mail className="h-3 w-3 text-muted-foreground" />
                                                                <a
                                                                    href={`mailto:${vendor.email}`}
                                                                    className="text-primary hover:underline"
                                                                >
                                                                    {vendor.email}
                                                                </a>
                                                            </div>
                                                        )}
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    {vendor.company_name ? (
                                                        <div className="flex items-center gap-1">
                                                            <Building2 className="h-3 w-3 text-muted-foreground" />
                                                            <span>{vendor.company_name}</span>
                                                        </div>
                                                    ) : (
                                                        <span className="text-muted-foreground">-</span>
                                                    )}
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    {vendor.gst_number || <span className="text-muted-foreground">-</span>}
                                                </td>
                                                <td className="px-4 py-3">
                                                    <span
                                                        className={`inline-flex rounded-full px-2 py-1 text-xs font-medium ${
                                                            vendor.status === 'active'
                                                                ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200'
                                                                : 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200'
                                                        }`}
                                                    >
                                                        {vendor.status}
                                                    </span>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center justify-end gap-2">
                                                        <Link href={`/societies/${society.id}/vendors/${vendor.id}`}>
                                                            <Button variant="ghost" size="sm">
                                                                <Eye className="h-4 w-4" />
                                                            </Button>
                                                        </Link>
                                                        <Link href={`/societies/${society.id}/vendors/${vendor.id}/edit`}>
                                                            <Button variant="ghost" size="sm">
                                                                <Edit className="h-4 w-4" />
                                                            </Button>
                                                        </Link>
                                                        <Button
                                                            variant="ghost"
                                                            size="sm"
                                                            onClick={() => handleDelete(vendor.id)}
                                                        >
                                                            <Trash2 className="h-4 w-4 text-destructive" />
                                                        </Button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td
                                                colSpan={6}
                                                className="px-4 py-8 text-center text-muted-foreground"
                                            >
                                                No vendors found
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {vendors.links && vendors.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Showing {vendors.meta.from} to {vendors.meta.to} of {vendors.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {vendors.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}


