import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
    DialogTrigger,
} from "@/components/ui/dialog";
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import {
    ModernTable,
    ModernTableHeader,
    ModernTableBody,
    ModernTableHead,
    ModernTableRow,
    ModernTableCell,
} from '@/components/ui/modern-table';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Home, Plus, Search, Edit, Trash2, Eye, UserCheck } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Units',
        href: '/societies/{society}/units',
    },
];

interface Building {
    id: number;
    name: string;
}

interface MaintenanceStructure {
    id: number;
    name: string;
}

interface Resident {
    id: number;
    name: string;
    phone: string;
    unit_id: number | null;
}

interface Unit {
    id: number;
    unit_no: string;
    type: string;
    area: number | null;
    status: 'vacant' | 'occupied' | 'maintenance' | 'rented';
    building_id: number | null;
    building?: {
        id: number;
        name: string;
    };
    maintenance_structure_id: number | null;
    maintenance_structure?: {
        id: number;
        name: string;
    };
    residents: {
        id: number;
        name: string;
        phone: string;
    }[];
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    units: {
        data: Unit[];
        links: any;
        meta: any;
    };
    buildings: Building[];
    maintenanceStructures: MaintenanceStructure[];
    filters: {
        search?: string;
        building_id?: string;
        status?: string;
        type?: string;
    };
    stats: {
        total_units: number;
        occupied_units: number;
        vacant_units: number;
    };
}

export default function Index({ society, units, buildings, maintenanceStructures, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [buildingId, setBuildingId] = useState(filters.building_id || 'all');
    const [status, setStatus] = useState(filters.status || 'all');
    const [type, setType] = useState(filters.type || 'all');
    
    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isViewOpen, setIsViewOpen] = useState(false);
    const [editingUnit, setEditingUnit] = useState<Unit | null>(null);
    const [viewingUnit, setViewingUnit] = useState<Unit | null>(null);

    // Create Form
    const createForm = useForm({
        unit_no: '',
        type: 'apartment',
        area: '',
        status: 'vacant',
        building_id: 'none',
    });

    // Edit Form
    const editForm = useForm({
        unit_no: '',
        building_id: 'none',
        type: 'apartment',
        area: '',
        status: 'vacant',
        maintenance_structure_id: 'none',
    });

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (buildingId && buildingId !== 'all') filterParams.building_id = buildingId;
        if (status && status !== 'all') filterParams.status = status;
        if (type && type !== 'all') filterParams.type = type;

        router.get(`/societies/${society.id}/units`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this unit?')) {
            router.delete(`/societies/${society.id}/units/${id}`, {
                preserveScroll: true,
            });
        }
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/units`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    const handleEditClick = (unit: Unit) => {
        setEditingUnit(unit);
        
        // Auto-correct status for the form if it mismatches resident state
        let initialStatus = unit.status;
        const hasResidents = unit.residents && unit.residents.length > 0;
        
        if (!hasResidents && (unit.status === 'occupied' || unit.status === 'rented')) {
            initialStatus = 'vacant';
        }

        editForm.setData({
            unit_no: unit.unit_no,
            building_id: unit.building_id ? unit.building_id.toString() : 'none',
            type: unit.type,
            area: unit.area ? unit.area.toString() : '',
            status: initialStatus,
            maintenance_structure_id: unit.maintenance_structure_id ? unit.maintenance_structure_id.toString() : 'none',
        });
        setIsEditOpen(true);
    };

    const openViewModal = (unit: Unit) => {
        setViewingUnit(unit);
        setIsViewOpen(true);
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (!editingUnit) return;
        
        editForm.put(`/societies/${society.id}/units/${editingUnit.id}`, {
            onSuccess: () => {
                setIsEditOpen(false);
                setEditingUnit(null);
                editForm.reset();
            },
        });
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Units" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                <PageHeader 
                    title="Units"
                    description="Manage all units in your society"
                    breadcrumbs={breadcrumbs}
                    actions={
                        <Button onClick={() => setIsCreateOpen(true)} className="bg-primary hover:bg-primary/90 shadow-lg shadow-primary/20">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Unit
                        </Button>
                    }
                />

                    {/* Edit Dialog */}
                    <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                        <DialogContent className="sm:max-w-[500px]">
                            <DialogHeader>
                                <DialogTitle>Edit Unit</DialogTitle>
                                <DialogDescription>
                                    Update unit details. Click save when you're done.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleEditSubmit} className="space-y-4 py-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="edit_unit_no">Flat Number</Label>
                                        <Input
                                            id="edit_unit_no"
                                            value={editForm.data.unit_no}
                                            onChange={(e) => editForm.setData('unit_no', e.target.value)}
                                            placeholder="e.g. A-101"
                                            required
                                        />
                                        {editForm.errors.unit_no && <p className="text-sm text-red-500">{editForm.errors.unit_no}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="edit_type">Type</Label>
                                        <Select value={editForm.data.type} onValueChange={(val) => editForm.setData('type', val)}>
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select type" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="apartment">Apartment</SelectItem>
                                                <SelectItem value="flat">Flat</SelectItem>
                                                <SelectItem value="bungalow">Bungalow</SelectItem>
                                                <SelectItem value="shop">Shop</SelectItem>
                                                <SelectItem value="office">Office</SelectItem>
                                                <SelectItem value="parking">Parking</SelectItem>
                                            </SelectContent>
                                        </Select>
                                        {editForm.errors.type && <p className="text-sm text-red-500">{editForm.errors.type}</p>}
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="edit_area">Area (sqft)</Label>
                                        <Input
                                            id="edit_area"
                                            type="number"
                                            value={editForm.data.area}
                                            onChange={(e) => editForm.setData('area', e.target.value)}
                                            placeholder="e.g. 1200"
                                        />
                                        {editForm.errors.area && <p className="text-sm text-red-500">{editForm.errors.area}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="edit_status">Status</Label>
                                        <Select value={editForm.data.status} onValueChange={(val) => editForm.setData('status', val)}>
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select status" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="vacant" disabled={!!(editingUnit?.residents && editingUnit.residents.length > 0)}>Vacant</SelectItem>
                                                <SelectItem value="maintenance">Maintenance</SelectItem>
                                                <SelectItem value="occupied" disabled={!(editingUnit?.residents && editingUnit.residents.length > 0)}>Occupied</SelectItem>
                                                <SelectItem value="rented" disabled={!(editingUnit?.residents && editingUnit.residents.length > 0)}>Rented</SelectItem>
                                            </SelectContent>
                                        </Select>
                                        {editingUnit?.residents && editingUnit.residents.length > 0 ? (
                                            <p className="text-[10px] text-muted-foreground">To mark as vacant, please remove the resident assignment.</p>
                                        ) : (
                                            <p className="text-[10px] text-muted-foreground">To mark as occupied, please assign a resident.</p>
                                        )}
                                        {editForm.errors.status && <p className="text-sm text-red-500">{editForm.errors.status}</p>}
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="edit_building_id">Building</Label>
                                    <Select value={editForm.data.building_id} onValueChange={(val) => editForm.setData('building_id', val)}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select Building" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="none">None</SelectItem>
                                            {buildings.map((building) => (
                                                <SelectItem key={building.id} value={building.id.toString()}>
                                                    {building.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                    {editForm.errors.building_id && <p className="text-sm text-red-500">{editForm.errors.building_id}</p>}
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={editForm.processing}>
                                        {editForm.processing ? 'Saving...' : 'Save Changes'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>

                    {/* View Dialog */}
                    <Dialog open={isViewOpen} onOpenChange={setIsViewOpen}>
                        <DialogContent className="sm:max-w-[600px]">
                            <DialogHeader>
                                <DialogTitle>Unit Details</DialogTitle>
                                <DialogDescription>
                                    View detailed information about this unit.
                                </DialogDescription>
                            </DialogHeader>
                            {viewingUnit && (
                                <div className="space-y-6 py-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Flat Number</h4>
                                            <p className="text-lg font-semibold">{viewingUnit.unit_no}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Type</h4>
                                            <p className="text-lg font-semibold capitalize">{viewingUnit.type}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Building</h4>
                                            <p className="text-base">{viewingUnit.building?.name || '-'}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Area</h4>
                                            <p className="text-base">{viewingUnit.area ? `${viewingUnit.area} sqft` : '-'}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Status</h4>
                                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium mt-1 ${
                                                viewingUnit.status === 'occupied' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                viewingUnit.status === 'vacant' ? 'bg-amber-100 text-amber-800 dark:bg-amber-900/30 dark:text-amber-400' :
                                                viewingUnit.status === 'rented' ? 'bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400' :
                                                'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400'
                                            }`}>
                                                {viewingUnit.status.charAt(0).toUpperCase() + viewingUnit.status.slice(1)}
                                            </span>
                                        </div>
                                    </div>

                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-2">Residents</h4>
                                        {viewingUnit.residents && viewingUnit.residents.length > 0 ? (
                                            <div className="border rounded-md divide-y">
                                                {viewingUnit.residents.map((resident) => (
                                                    <div key={resident.id} className="p-3 flex justify-between items-center">
                                                        <div>
                                                            <p className="font-medium">{resident.name}</p>
                                                            <p className="text-sm text-muted-foreground">{resident.phone}</p>
                                                        </div>
                                                        <Link href={`/societies/${society.id}/residents/${resident.id}`}>
                                                            <Button variant="ghost" size="sm" className="h-8 w-8">
                                                                <UserCheck className="h-4 w-4" />
                                                            </Button>
                                                        </Link>
                                                    </div>
                                                ))}
                                            </div>
                                        ) : (
                                            <p className="text-sm text-muted-foreground italic">No residents assigned.</p>
                                        )}
                                    </div>
                                </div>
                            )}
                            <DialogFooter>
                                <Button onClick={() => setIsViewOpen(false)}>Close</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>

                    {/* Create Dialog */}
                    <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                        <DialogContent className="sm:max-w-[500px]">
                            <DialogHeader>
                                <DialogTitle>Add New Unit</DialogTitle>
                                <DialogDescription>
                                    Create a new unit in your society. Click save when you're done.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="space-y-4 py-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="unit_no">Flat Number</Label>
                                        <Input
                                            id="unit_no"
                                            value={createForm.data.unit_no}
                                            onChange={(e) => createForm.setData('unit_no', e.target.value)}
                                            placeholder="e.g. A-101"
                                            required
                                        />
                                        {createForm.errors.unit_no && <p className="text-sm text-red-500">{createForm.errors.unit_no}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="type">Type</Label>
                                        <Select value={createForm.data.type} onValueChange={(val) => createForm.setData('type', val)}>
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select type" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="apartment">Apartment</SelectItem>
                                                <SelectItem value="flat">Flat</SelectItem>
                                                <SelectItem value="bungalow">Bungalow</SelectItem>
                                                <SelectItem value="shop">Shop</SelectItem>
                                                <SelectItem value="office">Office</SelectItem>
                                                <SelectItem value="parking">Parking</SelectItem>
                                            </SelectContent>
                                        </Select>
                                        {createForm.errors.type && <p className="text-sm text-red-500">{createForm.errors.type}</p>}
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="area">Area (sqft)</Label>
                                        <Input
                                            id="area"
                                            type="number"
                                            value={createForm.data.area}
                                            onChange={(e) => createForm.setData('area', e.target.value)}
                                            placeholder="e.g. 1200"
                                        />
                                        {createForm.errors.area && <p className="text-sm text-red-500">{createForm.errors.area}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="status">Status</Label>
                                        <Select value={createForm.data.status} onValueChange={(val) => createForm.setData('status', val)}>
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select status" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="vacant">Vacant</SelectItem>
                                                <SelectItem value="maintenance">Maintenance</SelectItem>
                                            </SelectContent>
                                        </Select>
                                        <p className="text-[10px] text-muted-foreground">Status will be set to "Occupied" when a resident is assigned.</p>
                                        {createForm.errors.status && <p className="text-sm text-red-500">{createForm.errors.status}</p>}
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="building_id">Building</Label>
                                    <Select value={createForm.data.building_id} onValueChange={(val) => createForm.setData('building_id', val)}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select Building" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="none">None</SelectItem>
                                            {buildings.map((building) => (
                                                <SelectItem key={building.id} value={building.id.toString()}>
                                                    {building.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                    {createForm.errors.building_id && <p className="text-sm text-red-500">{createForm.errors.building_id}</p>}
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={createForm.processing}>
                                        {createForm.processing ? 'Creating...' : 'Create Unit'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-3">
                    <GlassCard variant="hover" className="border-t-4 border-t-blue-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Units</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-blue-400 to-indigo-600 flex items-center justify-center text-white shadow-md shadow-blue-500/20">
                                <Home className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent className="p-5">
                            <div className="text-2xl font-bold text-foreground">{stats.total_units}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                Total registered units
                            </p>
                        </CardContent>
                    </GlassCard>

                    <GlassCard variant="hover" className="border-t-4 border-t-emerald-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Occupied</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-emerald-400 to-green-600 flex items-center justify-center text-white shadow-md shadow-emerald-500/20">
                                <UserCheck className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent className="p-5">
                            <div className="text-2xl font-bold text-foreground">{stats.occupied_units}</div>
                            <p className="text-xs text-emerald-600 font-medium mt-1">
                                {stats.total_units > 0 ? Math.round((stats.occupied_units / stats.total_units) * 100) : 0}% Occupancy rate
                            </p>
                        </CardContent>
                    </GlassCard>

                    <GlassCard variant="hover" className="border-t-4 border-t-orange-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Vacant</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-orange-400 to-amber-600 flex items-center justify-center text-white shadow-md shadow-orange-500/20">
                                <Home className="h-5 w-5 opacity-50" />
                            </div>
                        </CardHeader>
                        <CardContent className="p-5">
                            <div className="text-2xl font-bold text-foreground">{stats.vacant_units}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                Available for occupancy
                            </p>
                        </CardContent>
                    </GlassCard>
                </div>

                <Tabs defaultValue="units" className="w-full">
                    <TabsList className="grid w-full grid-cols-2 lg:w-[400px] mb-6 p-1 bg-muted/50 rounded-xl">
                        <TabsTrigger value="units" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-primary data-[state=active]:shadow-sm font-medium">Units</TabsTrigger>
                        <Link href={`/societies/${society.id}/buildings`} className="flex-1">
                            <TabsTrigger value="buildings" className="w-full rounded-lg bg-transparent">Buildings</TabsTrigger>
                        </Link>
                    </TabsList>
                </Tabs>

                {/* Filters */}
                <GlassCard>
                    <CardContent className="p-6">
                        <div className="flex flex-col md:flex-row gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by flat number..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="w-full md:w-48">
                                <Label htmlFor="building_id">Building</Label>
                                <Select value={buildingId} onValueChange={setBuildingId}>
                                    <SelectTrigger id="building_id" className="mt-1">
                                        <SelectValue placeholder="All Buildings" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Buildings</SelectItem>
                                        {buildings.map((building) => (
                                            <SelectItem key={building.id} value={building.id.toString()}>
                                                {building.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-full md:w-48">
                                <Label htmlFor="status">Status</Label>
                                <Select value={status} onValueChange={setStatus}>
                                    <SelectTrigger id="status" className="mt-1">
                                        <SelectValue placeholder="All Status" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        <SelectItem value="vacant">Vacant</SelectItem>
                                        <SelectItem value="occupied">Occupied</SelectItem>
                                        <SelectItem value="rented">Rented</SelectItem>
                                        <SelectItem value="maintenance">Maintenance</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-full md:w-48">
                                <Label htmlFor="type">Type</Label>
                                <Select value={type} onValueChange={setType}>
                                    <SelectTrigger id="type" className="mt-1">
                                        <SelectValue placeholder="All Types" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Types</SelectItem>
                                        <SelectItem value="apartment">Apartment</SelectItem>
                                        <SelectItem value="flat">Flat</SelectItem>
                                        <SelectItem value="bungalow">Bungalow</SelectItem>
                                        <SelectItem value="shop">Shop</SelectItem>
                                        <SelectItem value="office">Office</SelectItem>
                                        <SelectItem value="parking">Parking</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch} className="w-full md:w-auto">Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </GlassCard>

                {/* Units Table */}
                <div className="flex flex-col gap-4">
                    <div className="rounded-xl border border-border/50 overflow-hidden">
                        <ModernTable>
                            <ModernTableHeader>
                                <ModernTableRow className="bg-muted/50 hover:bg-muted/50">
                                    <ModernTableHead>Flat Number</ModernTableHead>
                                    <ModernTableHead>Building</ModernTableHead>
                                    <ModernTableHead>Type</ModernTableHead>
                                    <ModernTableHead>Area</ModernTableHead>
                                    <ModernTableHead>Status</ModernTableHead>
                                    <ModernTableHead>Resident</ModernTableHead>
                                    <ModernTableHead className="text-right">Actions</ModernTableHead>
                                </ModernTableRow>
                            </ModernTableHeader>
                            <ModernTableBody>
                                {units.data.length > 0 ? (
                                    units.data.map((unit) => (
                                        <ModernTableRow key={unit.id}>
                                            <ModernTableCell className="font-medium">
                                                <div className="flex items-center gap-3">
                                                    <div className="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                                                        <Home className="h-4 w-4" />
                                                    </div>
                                                    <span>{unit.unit_no}</span>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell className="text-muted-foreground">{unit.building?.name || '-'}</ModernTableCell>
                                            <ModernTableCell>
                                                <span className="capitalize px-2 py-1 rounded-md bg-muted text-muted-foreground text-xs font-medium">
                                                    {unit.type}
                                                </span>
                                            </ModernTableCell>
                                            <ModernTableCell className="font-medium">{unit.area ? `${unit.area} sqft` : '-'}</ModernTableCell>
                                            <ModernTableCell>
                                                <span
                                                    className={`inline-flex items-center gap-1.5 rounded-full px-2.5 py-0.5 text-xs font-bold capitalize ${
                                                        unit.status === 'occupied'
                                                            ? 'bg-emerald-100 text-emerald-700 dark:bg-emerald-900/30 dark:text-emerald-400'
                                                            : unit.status === 'rented'
                                                              ? 'bg-purple-100 text-purple-700 dark:bg-purple-900/30 dark:text-purple-400'
                                                              : unit.status === 'maintenance'
                                                                ? 'bg-amber-100 text-amber-700 dark:bg-amber-900/30 dark:text-amber-400'
                                                                : 'bg-slate-100 text-slate-700 dark:bg-slate-800 dark:text-slate-400'
                                                    }`}
                                                >
                                                    <span className={`h-1.5 w-1.5 rounded-full ${
                                                        unit.status === 'occupied' ? 'bg-emerald-500' :
                                                        unit.status === 'rented' ? 'bg-purple-500' :
                                                        unit.status === 'maintenance' ? 'bg-amber-500' :
                                                        'bg-slate-500'
                                                    }`} />
                                                    {unit.status}
                                                </span>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                {unit.residents && unit.residents.length > 0 ? (
                                                    <div className="flex flex-col gap-1">
                                                        {unit.residents.slice(0, 2).map((resident) => (
                                                            <div key={resident.id} className="text-xs">
                                                                <span className="font-medium block">{resident.name}</span>
                                                                <span className="text-muted-foreground">{resident.phone}</span>
                                                            </div>
                                                        ))}
                                                        {unit.residents.length > 2 && (
                                                            <span className="text-xs text-muted-foreground italic">
                                                                +{unit.residents.length - 2} more
                                                            </span>
                                                        )}
                                                    </div>
                                                ) : (
                                                    <span className="text-sm text-muted-foreground">-</span>
                                                )}
                                            </ModernTableCell>
                                            <ModernTableCell className="text-right">
                                                <div className="flex items-center justify-end gap-2">
                                                    <Button 
                                                        variant="ghost" 
                                                        size="icon" 
                                                        className="h-8 w-8 hover:bg-blue-50 hover:text-blue-600"
                                                        onClick={() => openViewModal(unit)}
                                                    >
                                                        <Eye className="h-4 w-4" />
                                                    </Button>
                                                    <Button 
                                                        variant="ghost" 
                                                        size="icon" 
                                                        className="h-8 w-8 hover:bg-amber-50 hover:text-amber-600"
                                                        onClick={() => handleEditClick(unit)}
                                                    >
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        className="h-8 w-8 hover:bg-red-50 hover:text-red-600"
                                                        onClick={() => handleDelete(unit.id)}
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </Button>
                                                </div>
                                            </ModernTableCell>
                                        </ModernTableRow>
                                    ))
                                ) : (
                                    <ModernTableRow>
                                        <ModernTableCell colSpan={7} className="h-24 text-center">
                                            <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                                <Home className="h-10 w-10 text-muted-foreground/30" />
                                                <p>No units found matching your criteria</p>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                )}
                            </ModernTableBody>
                        </ModernTable>
                    </div>

                    {/* Pagination */}
                    {units.links && units.links.length > 3 && (
                        <div className="flex items-center justify-between px-2">
                            <div className="text-sm text-muted-foreground">
                                Showing <span className="font-medium text-foreground">{units.meta.from}</span> to <span className="font-medium text-foreground">{units.meta.to}</span> of <span className="font-medium text-foreground">{units.meta.total}</span> results
                            </div>
                            <div className="flex gap-1">
                                {units.links.map((link: any, index: number) => (
                                    <Link
                                        key={index}
                                        href={link.url || '#'}
                                        className={`px-3 py-1 rounded-md text-sm transition-colors ${
                                            link.active
                                                ? 'bg-primary text-primary-foreground font-medium shadow-sm'
                                                : 'bg-background hover:bg-muted text-muted-foreground border border-border/50'
                                        } ${!link.url && 'opacity-50 cursor-not-allowed'}`}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </SocietyAdminLayout>
    );
}

